import { Stage } from "aws-cdk-lib";
import { type StageDeployment } from "aws-cdk-lib/pipelines";
import { Construct } from "constructs";
import type { IAwsCredentialsProvider } from "./aws-credentials";
import type { DockerCredentials } from "./docker-credentials";
import type { Synth } from "./steps";
import { GitHubWave, type StageOptions, type WaveOptions } from "./wave";
import { type PipelinePhases } from "./workflow";
/**
 * Properties for configuring a GitHub Actions-based deployment pipeline.
 *
 * @remarks
 * `GitHubActionsPipelineProps` enables configuration of the GitHub Actions workflow
 * for a CDK pipeline, including defining the workflow environment, AWS credentials,
 * Docker registry credentials, job phases, and version overrides for specific actions.
 * It also provides options for setting workflow file paths, naming conventions, and
 * a synthesizer for the CDK application.
 */
export interface GitHubActionsPipelineProps extends PipelinePhases {
    /**
     * Optional name for the GitHub Actions workflow.
     *
     * @default "Deploy"
     */
    readonly workflowName?: string;
    /**
     * Directory path where workflow YAML files will be generated.
     *
     * @default ".github/workflows"
     */
    readonly workflowOutdir?: string;
    /**
     * Name of the generated workflow file (without extension).
     *
     * @default "deploy"
     */
    readonly workflowFilename?: string;
    /**
     * Enables a single publishing job per asset type within the workflow.
     *
     * @remarks
     * When set to `true`, this option consolidates publishing jobs by asset type
     * (e.g., Docker images, file assets), which can reduce redundant jobs and streamline
     * the workflow, especially in pipelines with multiple assets of the same type.
     *
     * @default false
     */
    readonly singlePublisherPerAssetType?: boolean;
    /**
     * Environment variables to be included in the workflow.
     *
     * @remarks
     * This allows setting custom environment variables for jobs within the workflow,
     * which may be useful for configuration or runtime settings that the jobs rely on.
     */
    readonly workflowEnv?: Record<string, string>;
    /**
     * AWS credentials provider for actions requiring AWS authentication.
     *
     * @remarks
     * This provider supplies AWS credentials (e.g., access keys) for actions that
     * interact with AWS services. The provider should implement `IAwsCredentialsProvider`.
     */
    readonly awsCredentials: IAwsCredentialsProvider;
    /**
     * Docker credentials required for registry authentication within the workflow.
     *
     * @remarks
     * Specify one or more `DockerCredentials` instances for authenticating against Docker
     * registries (such as DockerHub, ECR, GHCR, or custom registries) used in the pipeline.
     */
    readonly dockerCredentials?: DockerCredentials[];
    /**
     * Version overrides for specific GitHub Actions used in the workflow.
     *
     * @remarks
     * Use this to specify particular versions of actions within the workflow (e.g.,
     * actions/checkout@v2). This is useful for managing dependencies and ensuring compatibility.
     */
    readonly versionOverrides?: Record<string, string>;
    /**
     * Synthesizer for the CDK application.
     *
     * @remarks
     * The synthesizer generates CloudFormation templates and other assets required
     * for deployment. This is a critical part of the CDK application lifecycle.
     */
    readonly synth: Synth;
}
/**
 * Constructs a GitHub Actions pipeline for deploying AWS resources.
 *
 * @remarks
 * The `GitHubActionsPipeline` provides methods to define and manage deployment stages and job waves in
 * a GitHub Actions pipeline, utilizing AWS credentials and CDK output for cloud infrastructure automation.
 */
export declare class GitHubActionsPipeline extends Construct {
    private readonly innerPipeline;
    /**
     * Constructs a new instance of `GitHubActionsPipeline`.
     *
     * @param scope - The parent construct scope.
     * @param id - Unique identifier for this pipeline construct.
     * @param props - Configuration properties for the pipeline.
     */
    constructor(scope: Construct, id: string, props: GitHubActionsPipelineProps);
    /**
     * Returns the name of the workflow.
     */
    get workflowName(): string;
    /**
     * Returns the output directory path for the workflow files.
     */
    get workflowOutdir(): string;
    /**
     * Returns the filename for the workflow file.
     */
    get workflowFilename(): string;
    /**
     * Adds a stage to the pipeline with GitHub-specific configuration options.
     *
     * @param stage - The CDK Stage to add to the pipeline.
     * @param options - Optional configuration for the stage.
     * @returns Deployment details for the added stage.
     */
    addStage(stage: Stage, options?: StageOptions): StageDeployment;
    /**
     * Adds a wave of jobs to the pipeline.
     *
     * @param id - Unique identifier for the wave.
     * @param options - Options for configuring the wave.
     * @returns The created GitHub wave instance.
     */
    addWave(id: string, options?: WaveOptions): GitHubWave;
}
