"use strict";
var __createBinding = (this && this.__createBinding) || (Object.create ? (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    var desc = Object.getOwnPropertyDescriptor(m, k);
    if (!desc || ("get" in desc ? !m.__esModule : desc.writable || desc.configurable)) {
      desc = { enumerable: true, get: function() { return m[k]; } };
    }
    Object.defineProperty(o, k2, desc);
}) : (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    o[k2] = m[k];
}));
var __setModuleDefault = (this && this.__setModuleDefault) || (Object.create ? (function(o, v) {
    Object.defineProperty(o, "default", { enumerable: true, value: v });
}) : function(o, v) {
    o["default"] = v;
});
var __importStar = (this && this.__importStar) || function (mod) {
    if (mod && mod.__esModule) return mod;
    var result = {};
    if (mod != null) for (var k in mod) if (k !== "default" && Object.prototype.hasOwnProperty.call(mod, k)) __createBinding(result, mod, k);
    __setModuleDefault(result, mod);
    return result;
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.StagePipelineJob = exports.DeployPipelineJob = exports.PublishPipelineJob = exports.SynthPipelineJob = exports.PipelineJob = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const path = __importStar(require("node:path"));
const github_actions_cdk_1 = require("github-actions-cdk");
const assets_1 = require("./private/assets");
const utils_1 = require("./private/utils");
const CDKOUT_ARTIFACT = "cdk.out";
/**
 * Represents a job within the pipeline that requires AWS credentials and CDK output.
 *
 * @remarks
 * The `PipelineJob` class extends the `Job` class and includes specific properties and methods for managing
 * AWS authentication, CDK output references, and version control for GitHub Actions used in the pipeline.
 */
class PipelineJob extends github_actions_cdk_1.Job {
    static [JSII_RTTI_SYMBOL_1] = { fqn: "@github-actions-cdk/aws-cdk.PipelineJob", version: "0.0.22" };
    /** AWS credentials provider associated with this job. */
    awsCredentials;
    /**
     * Docker credentials required for registry authentication within the workflow.
     *
     * @remarks
     * Specify one or more `DockerCredentials` instances for authenticating against Docker
     * registries (such as DockerHub, ECR, GHCR, or custom registries) used in the pipeline.
     */
    dockerCredentials;
    /** Specific version overrides for GitHub Actions, if any are provided. */
    versionOverrides;
    /** Directory containing the CDK output files for this job. */
    cdkoutDir;
    /**
     * Constructs a new instance of `PipelineJob`.
     *
     * @param scope - The scope in which to define this job construct.
     * @param id - Unique identifier for this job within the workflow.
     * @param props - Properties for configuring the pipeline job.
     */
    constructor(scope, id, props) {
        super(scope, id, props);
        this.awsCredentials = props.awsCredentials;
        this.dockerCredentials = props.dockerCredentials;
        this.versionOverrides = props.versionOverrides ?? {};
        this.cdkoutDir = props.cdkoutDir;
    }
    /**
     * Looks up the version override for a given action identifier, if available.
     *
     * @param actionIdentifier - The identifier of the GitHub Action to retrieve the version for.
     * @returns The overridden version (or SHA) for the action, if specified; otherwise, `undefined`.
     */
    lookupVersion(actionIdentifier) {
        return this.versionOverrides[actionIdentifier] ?? undefined;
    }
}
exports.PipelineJob = PipelineJob;
/**
 * A job that synthesizes the CloudFormation template using CDK.
 */
class SynthPipelineJob extends PipelineJob {
    static [JSII_RTTI_SYMBOL_1] = { fqn: "@github-actions-cdk/aws-cdk.SynthPipelineJob", version: "0.0.22" };
    constructor(scope, id, props) {
        super(scope, id, props);
        new github_actions_cdk_1.actions.CheckoutV4(this, "checkout", {
            name: "Checkout",
            version: this.lookupVersion(github_actions_cdk_1.actions.CheckoutV4.IDENTIFIER),
        });
        if (props.installCommands && props.installCommands.length > 0) {
            new github_actions_cdk_1.RunStep(this, "install", {
                name: "Install",
                run: props.installCommands,
            });
        }
        // Docker credentials configuration
        if (this.dockerCredentials) {
            for (const creds of this.dockerCredentials) {
                creds.credentialSteps(this);
            }
        }
        if (props.preBuild)
            props.preBuild.steps(this);
        new github_actions_cdk_1.RunStep(this, "build", {
            name: "Build",
            run: props.commands,
        });
        if (props.postBuild)
            props.postBuild.steps(this);
        new github_actions_cdk_1.actions.UploadArtifactV4(this, "upload", {
            name: `Upload ${CDKOUT_ARTIFACT}`,
            artifactName: CDKOUT_ARTIFACT,
            path: props.cdkoutDir,
            includeHiddenFiles: true,
            version: this.lookupVersion(github_actions_cdk_1.actions.UploadArtifactV4.IDENTIFIER),
        });
    }
}
exports.SynthPipelineJob = SynthPipelineJob;
/**
 * A job that publishes stack assets to AWS.
 *
 * @remarks
 * The `PublishPipelineJob` class handles the process of publishing assets to AWS.
 * It defines the steps required to download artifacts, install necessary dependencies,
 * and execute the publish command for each asset. The job integrates with AWS
 * credentials for secure authentication and provides hooks for outputting asset hashes.
 */
class PublishPipelineJob extends PipelineJob {
    static [JSII_RTTI_SYMBOL_1] = { fqn: "@github-actions-cdk/aws-cdk.PublishPipelineJob", version: "0.0.22" };
    /**
     * Constructs a new instance of `PublishPipelineJob`.
     *
     * @param scope - The parent construct scope.
     * @param id - Unique identifier for this publish job.
     * @param props - Configuration properties for the publish job.
     *
     * @remarks
     * The constructor initializes the publish job by setting up the necessary steps
     * to download artifacts, install dependencies, and publish assets. It iterates
     * through each asset and creates the appropriate publish steps.
     */
    constructor(scope, id, props) {
        super(scope, id, props);
        if (props.assets.length === 0) {
            throw new Error("Asset Publish step must have at least 1 asset");
        }
        // Download artifact step
        new github_actions_cdk_1.actions.DownloadArtifactV4(this, "DownloadArtifact", {
            name: `Download ${CDKOUT_ARTIFACT}`,
            artifactName: CDKOUT_ARTIFACT,
            path: props.cdkoutDir,
            version: this.lookupVersion(github_actions_cdk_1.actions.DownloadArtifactV4.IDENTIFIER),
        });
        // Install CDK assets
        const installSuffix = props.cdkCliVersion ? `@${props.cdkCliVersion}` : "";
        new github_actions_cdk_1.RunStep(this, "install", {
            name: "Install",
            run: `npm install --no-save cdk-assets${installSuffix}`,
        });
        // AWS credentials configuration
        props.awsCredentials.credentialSteps(this, "us-east-1");
        // Docker credentials configuration
        if (this.dockerCredentials) {
            for (const creds of this.dockerCredentials) {
                creds.credentialSteps(this);
            }
        }
        if (props.prePublish)
            props.prePublish.steps(this);
        const scriptGen = new assets_1.PublishAssetScriptGenerator(props.cdkoutDir, props.assets);
        // Write script to cdk.out directory
        const scriptFilename = path.join(props.cdkoutDir, `publish-assets-${id}.sh`);
        scriptGen.writePublishScript(scriptFilename);
        const publishStep = new github_actions_cdk_1.RunStep(this, "publish", {
            name: `Publish ${id}`,
            run: `/bin/bash ${props.cdkoutDir}/${(0, utils_1.posixPath)(path.relative(props.cdkoutDir, scriptFilename))}`,
        });
        scriptGen.assetIdMap.forEach((outputName, hash) => {
            props.assetHashMap[hash] = `\${{ needs.${this.id}.outputs.${outputName} }}`;
            this.addOutput(outputName, publishStep.outputExpression(outputName));
        });
        if (props.postPublish)
            props.postPublish.steps(this);
    }
    containsDockerAsset() {
        return this.id.includes("DockerAsset");
    }
}
exports.PublishPipelineJob = PublishPipelineJob;
/**
 * A job that deploys a CloudFormation stack.
 *
 * @remarks
 * The `DeployPipelineJob` class is responsible for executing the deployment of a
 * specified CloudFormation stack. It integrates with AWS credentials for authentication
 * and ensures that the stack is deployed with the correct template and asset replacements.
 * The job will throw errors if required properties are not provided, ensuring
 * robustness in the deployment process.
 */
class DeployPipelineJob extends PipelineJob {
    static [JSII_RTTI_SYMBOL_1] = { fqn: "@github-actions-cdk/aws-cdk.DeployPipelineJob", version: "0.0.22" };
    /**
     * Constructs a new instance of `DeployPipelineJob`.
     *
     * @param scope - The parent construct scope.
     * @param id - Unique identifier for this deployment job.
     * @param props - Configuration properties for the deployment job.
     *
     * @remarks
     * The constructor validates required properties for the stack and sets up the
     * necessary steps to deploy the CloudFormation stack using the provided asset hash
     * mappings and options. It initializes the deployment action with AWS CloudFormation.
     */
    constructor(scope, id, props) {
        super(scope, id, props);
        // Validate required properties
        if (!props.stack.region) {
            throw new Error('"region" is required');
        }
        if (!props.stack.templateUrl) {
            throw new Error(`unable to determine template URL for stack ${props.stack.stackArtifactId}`);
        }
        // Configure AWS credentials for deployment
        props.awsCredentials.credentialSteps(this, props.stack.region, props.stack.assumeRoleArn);
        // Function to replace asset hash in the template
        const replaceAssetHash = (template) => {
            const hash = path.parse(template.split("/").pop() ?? "").name;
            if (props.assetHashMap[hash] === undefined) {
                throw new Error(`Template asset hash ${hash} not found.`);
            }
            return template.replace(hash, props.assetHashMap[hash]);
        };
        // Create the CloudFormation deployment action
        new github_actions_cdk_1.actions.AwsCloudFormationGitHubDeployV1(this, "deploy", {
            stackName: props.stack.stackName,
            template: replaceAssetHash(props.stack.templateUrl),
            noFailOnEmptyChangeset: "1",
            roleArn: props.stack.executionRoleArn,
            capabilities: props.stackOptions?.capabilities?.join(","),
            version: this.lookupVersion(github_actions_cdk_1.actions.AwsCloudFormationGitHubDeployV1.IDENTIFIER),
        });
    }
}
exports.DeployPipelineJob = DeployPipelineJob;
/**
 * A job that executes a specific phase of steps in the pipeline.
 */
class StagePipelineJob extends PipelineJob {
    static [JSII_RTTI_SYMBOL_1] = { fqn: "@github-actions-cdk/aws-cdk.StagePipelineJob", version: "0.0.22" };
    constructor(scope, id, props) {
        super(scope, id, props);
        props.phase.steps(this);
    }
}
exports.StagePipelineJob = StagePipelineJob;
//# sourceMappingURL=data:application/json;base64,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