"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.DockerCredentials = exports.DockerCredentialsUsage = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const github_actions_cdk_1 = require("github-actions-cdk");
const jobs_1 = require("./jobs");
/**
 * Enumeration of Docker credential usage scenarios within a pipeline.
 *
 * @remarks
 * `DockerCredentialsUsage` defines where Docker credentials are required,
 * including the `SYNTH` phase for build/synth steps and `ASSET_PUBLISHING` for publishing assets.
 */
var DockerCredentialsUsage;
(function (DockerCredentialsUsage) {
    /** Credentials used for synthesis and build steps. */
    DockerCredentialsUsage["SYNTH"] = "SYNTH";
    /** Credentials used for publishing assets (e.g., Docker images). */
    DockerCredentialsUsage["ASSET_PUBLISHING"] = "ASSET_PUBLISHING";
})(DockerCredentialsUsage || (exports.DockerCredentialsUsage = DockerCredentialsUsage = {}));
/**
 * Base class for providing Docker credentials to various registry types.
 *
 * @remarks
 * The `DockerCredentials` class provides a unified interface for handling Docker login credentials
 * across different registry providers (e.g., DockerHub, ECR, GHCR). Concrete implementations of
 * this class define specific registry login steps.
 */
class DockerCredentials {
    usages;
    static [JSII_RTTI_SYMBOL_1] = { fqn: "@github-actions-cdk/aws-cdk.DockerCredentials", version: "0.0.22" };
    /**
     * Factory method for DockerHub credentials.
     */
    static dockerHub(props) {
        return new DockerHubCredentials(props);
    }
    /**
     * Factory method for AWS ECR credentials.
     */
    static ecr(props) {
        return new EcrCredentials(props);
    }
    /**
     * Factory method for GitHub Container Registry (GHCR) credentials.
     */
    static ghcr(props = {}) {
        return new GhcrCredentials(props);
    }
    /**
     * Factory method for custom registry credentials.
     */
    static customRegistry(props) {
        return new CustomProvider(props);
    }
    /**
     * Returns Docker login steps required for the specified job.
     *
     * @param job - The GitHub Actions job for which Docker login steps are generated.
     * @returns An array of `RegularStep` instances for Docker authentication.
     */
    credentialSteps(job) {
        if (this.usages) {
            if (job instanceof jobs_1.SynthPipelineJob && !this.usages.includes(DockerCredentialsUsage.SYNTH)) {
                return [];
            }
            if (job instanceof jobs_1.PublishPipelineJob &&
                !this.usages.includes(DockerCredentialsUsage.ASSET_PUBLISHING)) {
                return [];
            }
        }
        return this._credentialSteps(job);
    }
    /**
     * Initializes Docker credentials with specified usage scopes.
     *
     * @param usages - The usage scopes for the credentials (optional).
     */
    constructor(usages) {
        this.usages = usages;
    }
}
exports.DockerCredentials = DockerCredentials;
/**
 * DockerHub credentials class for authentication against DockerHub.
 */
class DockerHubCredentials extends DockerCredentials {
    usernameKey;
    personalAccessTokenKey;
    constructor(props) {
        super(props.usages);
        this.usernameKey = props.usernameKey ?? "DOCKERHUB_USERNAME";
        this.personalAccessTokenKey = props.personalAccessTokenKey ?? "DOCKERHUB_TOKEN";
    }
    /** @internal */
    _credentialSteps(job) {
        return [
            new github_actions_cdk_1.actions.DockerLoginV3(job, "docker-hub-login", {
                username: github_actions_cdk_1.Expression.fromSecrets(this.usernameKey),
                password: github_actions_cdk_1.Expression.fromSecrets(this.personalAccessTokenKey),
            }),
        ];
    }
}
/**
 * AWS ECR credentials class for authentication against AWS Elastic Container Registry.
 */
class EcrCredentials extends DockerCredentials {
    registry;
    usernameKey;
    personalAccessTokenKey;
    constructor(props) {
        super(props.usages);
        this.registry = props.registry;
    }
    /** @internal */
    _credentialSteps(job) {
        return [
            new github_actions_cdk_1.actions.DockerLoginV3(job, "ecr-login", {
                registry: this.registry,
                ecr: "true",
            }),
        ];
    }
}
/**
 * GHCR credentials class for authentication against GitHub Container Registry.
 */
class GhcrCredentials extends DockerCredentials {
    constructor(props = {}) {
        super(props.usages);
    }
    /** @internal */
    _credentialSteps(job) {
        return [
            new github_actions_cdk_1.actions.DockerLoginV3(job, "ghcr-login", {
                registry: "ghcr.io",
                username: github_actions_cdk_1.Expression.fromGitHub("actor"),
                password: github_actions_cdk_1.Expression.fromSecrets("GITHUB_TOKEN"),
            }),
        ];
    }
}
/**
 * Custom provider credentials class for authentication against a custom Docker registry.
 */
class CustomProvider extends DockerCredentials {
    registry;
    usernameKey;
    passwordKey;
    constructor(props) {
        super(props.usages);
        this.registry = props.registry;
        this.usernameKey = props.usernameKey;
        this.passwordKey = props.passwordKey;
    }
    /** @internal */
    _credentialSteps(job) {
        return [
            new github_actions_cdk_1.actions.DockerLoginV3(job, "custom-docker-login", {
                registry: this.registry,
                username: github_actions_cdk_1.Expression.fromSecrets(this.usernameKey),
                password: github_actions_cdk_1.Expression.fromSecrets(this.passwordKey),
            }),
        ];
    }
}
//# sourceMappingURL=data:application/json;base64,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