"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.AwsCredentials = exports.GitHubActionsRoleArn = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const github_actions_cdk_1 = require("github-actions-cdk");
/**
 * AWS credentials provider that retrieves credentials from GitHub Secrets.
 */
class GitHubSecretsProvider {
    accessKeyId;
    secretAccessKey;
    sessionToken;
    /**
     * Constructs a new instance of `GitHubSecretsProvider`.
     *
     * @param props - Optional properties for configuring the GitHub Secrets provider.
     */
    constructor(props) {
        this.accessKeyId = props?.accessKeyId ?? "AWS_ACCESS_KEY_ID";
        this.secretAccessKey = props?.secretAccessKey ?? "AWS_SECRET_ACCESS_KEY";
        this.sessionToken = props?.sessionToken;
    }
    /**
     * Returns the permission level required by this provider.
     * In this case, no special permissions are required.
     */
    permissionLevel() {
        return github_actions_cdk_1.PermissionLevel.NONE;
    }
    /**
     * Configures AWS credentials for a GitHub Actions job using GitHub Secrets.
     *
     * @param job - The job in which to configure the credentials.
     * @param region - The AWS region where the credentials are applicable.
     * @param assumeRoleArn - An optional ARN for a role to assume.
     * @returns An array of steps to authenticate with AWS using the provided secrets.
     */
    credentialSteps(job, region, assumeRoleArn) {
        return [
            new github_actions_cdk_1.actions.ConfigureAwsCredentialsV4(job, "id", {
                name: "Authenticate Via GitHub Secrets",
                awsRegion: region,
                awsAccessKeyId: github_actions_cdk_1.Expression.fromSecrets(this.accessKeyId),
                awsSecretAccessKey: github_actions_cdk_1.Expression.fromSecrets(this.secretAccessKey),
                ...(this.sessionToken
                    ? {
                        sessionToken: github_actions_cdk_1.Expression.fromSecrets(this.sessionToken),
                    }
                    : undefined),
                ...(assumeRoleArn
                    ? {
                        roleToAssume: assumeRoleArn,
                        roleExternalId: "Pipeline",
                    }
                    : undefined),
            }),
        ];
    }
}
/**
 * AWS credentials provider that uses OpenID Connect for authentication.
 */
class OpenIdConnectProvider {
    gitHubActionsRoleArn;
    roleSessionName;
    /**
     * Constructs a new instance of `OpenIdConnectProvider`.
     *
     * @param props - Properties for configuring the OpenID Connect provider.
     * @throws Error if `gitHubActionsRoleArn` is unresolved.
     */
    constructor(props) {
        if (aws_cdk_lib_1.Token.isUnresolved(props.gitHubActionsRoleArn)) {
            throw new Error(`The provided gitHubActionsRoleArn (${props.gitHubActionsRoleArn}) is unresolved. Please provide a concrete value.`);
        }
        this.gitHubActionsRoleArn = props.gitHubActionsRoleArn;
        this.roleSessionName = props.roleSessionName;
    }
    /**
     * Returns the permission level required by this provider.
     * This provider requires write permissions.
     */
    permissionLevel() {
        return github_actions_cdk_1.PermissionLevel.WRITE;
    }
    /**
     * Configures AWS credentials for a GitHub Actions job using OpenID Connect.
     *
     * @param job - The job in which to configure the credentials.
     * @param region - The AWS region where the credentials are applicable.
     * @param assumeRoleArn - An optional ARN for a role to assume with elevated permissions.
     * @returns An array of steps to authenticate with AWS using OpenID Connect.
     */
    credentialSteps(job, region, assumeRoleArn) {
        const steps = [];
        steps.push(new github_actions_cdk_1.actions.ConfigureAwsCredentialsV4(job, "authenticate", {
            name: "Authenticate Via OIDC",
            awsRegion: region,
            roleToAssume: this.gitHubActionsRoleArn,
            roleSessionName: this.roleSessionName,
        }));
        if (assumeRoleArn) {
            const getDeployRole = (arn) => arn.replace("cfn-exec", "deploy");
            steps.push(new github_actions_cdk_1.actions.ConfigureAwsCredentialsV4(job, "assume-role", {
                name: "Assume CDK Deploy Role",
                awsRegion: region,
                awsAccessKeyId: github_actions_cdk_1.Expression.fromEnv("AWS_ACCESS_KEY_ID"),
                awsSecretAccessKey: github_actions_cdk_1.Expression.fromEnv("AWS_SECRET_ACCESS_KEY"),
                awsSessionToken: github_actions_cdk_1.Expression.fromEnv("AWS_SESSION_TOKEN"),
                roleToAssume: getDeployRole(assumeRoleArn),
                roleExternalId: "Pipeline",
            }));
        }
        return steps;
    }
}
/**
 * Helper class for generating ARNs for GitHub Actions roles.
 */
class GitHubActionsRoleArn {
    static [JSII_RTTI_SYMBOL_1] = { fqn: "@github-actions-cdk/aws-cdk.GitHubActionsRoleArn", version: "0.0.22" };
    /**
     * Creates an ARN for a GitHub Actions role based on the account ID.
     *
     * @param accountId - The AWS account ID.
     * @param roleName - The name of the IAM role (defaults to "GitHubActionsRole").
     * @returns The full ARN of the specified role.
     */
    static fromAccount(accountId, roleName = "GitHubActionsRole") {
        return `arn:aws:iam::${accountId}:role/${roleName}`;
    }
}
exports.GitHubActionsRoleArn = GitHubActionsRoleArn;
/**
 * Factory class for creating instances of AWS credentials providers.
 */
class AwsCredentials {
    static [JSII_RTTI_SYMBOL_1] = { fqn: "@github-actions-cdk/aws-cdk.AwsCredentials", version: "0.0.22" };
    /**
     * Creates an AWS credentials provider that uses GitHub secrets.
     *
     * @param props - Optional properties for configuring the GitHub Secrets provider.
     * @returns An instance of `GitHubSecretsProvider`.
     */
    static fromGitHubSecrets(props) {
        return new GitHubSecretsProvider(props);
    }
    /**
     * Creates an AWS credentials provider that uses OpenID Connect.
     *
     * @param props - Properties for configuring the OpenID Connect provider.
     * @returns An instance of `OpenIdConnectProvider`.
     */
    static fromOpenIdConnect(props) {
        return new OpenIdConnectProvider(props);
    }
}
exports.AwsCredentials = AwsCredentials;
//# sourceMappingURL=data:application/json;base64,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