import { type Job, PermissionLevel, type RegularStep } from "github-actions-cdk";
/**
 * Interface representing a provider for AWS credentials.
 *
 * Implementations of this interface are responsible for defining how
 * AWS credentials are obtained and how they are configured within
 * GitHub Actions jobs.
 */
export interface IAwsCredentialsProvider {
    /**
     * Retrieves the permission level required by this credentials provider.
     */
    permissionLevel(): PermissionLevel;
    /**
     * Generates a series of steps to configure AWS credentials for a GitHub Actions job.
     *
     * @param job - The GitHub Actions job in which to configure the credentials.
     * @param region - The AWS region in which the credentials will be used.
     * @param assumeRoleArn - An optional ARN for a role to assume with elevated permissions.
     * @returns An array of `RegularStep` instances to be executed in the job.
     */
    credentialSteps(job: Job, region: string, assumeRoleArn?: string): RegularStep[];
}
/**
 * Properties for configuring the GitHub Secrets provider.
 */
export interface GitHubSecretsProviderProps {
    /**
     * The name of the GitHub secret that holds the AWS access key ID.
     * @default "AWS_ACCESS_KEY_ID"
     */
    readonly accessKeyId: string;
    /**
     * The name of the GitHub secret that holds the AWS secret access key.
     * @default "AWS_SECRET_ACCESS_KEY"
     */
    readonly secretAccessKey: string;
    /**
     * The name of the GitHub secret that holds the AWS session token.
     * @default - no session token is used
     */
    readonly sessionToken?: string;
}
/**
 * Properties for configuring the OpenID Connect provider.
 */
export interface OpenIdConnectProviderProps {
    /**
     * The ARN of the role that GitHub Actions will assume via OpenID Connect.
     */
    readonly gitHubActionsRoleArn: string;
    /**
     * Optional role session name to use when assuming the role.
     * @default - no role session name
     */
    readonly roleSessionName?: string;
}
/**
 * Helper class for generating ARNs for GitHub Actions roles.
 */
export declare class GitHubActionsRoleArn {
    /**
     * Creates an ARN for a GitHub Actions role based on the account ID.
     *
     * @param accountId - The AWS account ID.
     * @param roleName - The name of the IAM role (defaults to "GitHubActionsRole").
     * @returns The full ARN of the specified role.
     */
    static fromAccount(accountId: string, roleName?: string): string;
}
/**
 * Factory class for creating instances of AWS credentials providers.
 */
export declare class AwsCredentials {
    /**
     * Creates an AWS credentials provider that uses GitHub secrets.
     *
     * @param props - Optional properties for configuring the GitHub Secrets provider.
     * @returns An instance of `GitHubSecretsProvider`.
     */
    static fromGitHubSecrets(props?: GitHubSecretsProviderProps): IAwsCredentialsProvider;
    /**
     * Creates an AWS credentials provider that uses OpenID Connect.
     *
     * @param props - Properties for configuring the OpenID Connect provider.
     * @returns An instance of `OpenIdConnectProvider`.
     */
    static fromOpenIdConnect(props: OpenIdConnectProviderProps): IAwsCredentialsProvider;
}
