import type { PipelineBase, StackAsset, StackDeployment } from "aws-cdk-lib/pipelines";
import type { Construct } from "constructs";
import type { WorkflowProps } from "github-actions-cdk";
import { Workflow } from "github-actions-cdk";
import type { IAwsCredentialsProvider } from "./aws-credentials";
import type { DockerCredentials } from "./docker-credentials";
import type { IJobPhase, StackOptions } from "./jobs";
import { StageJob, Synth } from "./steps";
/**
 * Interface representing optional job phases for a build and publish jobs ot the pipeline.
 */
export interface PipelinePhases {
    /**
     * Optional phase for the synth job to execute before the main build steps.
     */
    readonly preBuild?: IJobPhase;
    /**
     * Optional phase for the synth job to execute after the main build steps.
     */
    readonly postBuild?: IJobPhase;
    /**
     * Optional phase for publish jobs to execute before the main publish steps.
     */
    readonly prePublish?: IJobPhase;
    /**
     * Optional phase for publish jobs to execute after the main publish steps.
     */
    readonly postPublish?: IJobPhase;
}
/**
 * Properties for defining a Pipeline Workflow.
 *
 * @remarks
 * This interface extends WorkflowProps and adds properties specific to AWS CDK Pipelines and job execution.
 */
export interface PipelineWorkflowProps extends WorkflowProps {
    /**
     * The CDK pipeline, including its stages and job configuration.
     * Defines the sequence and structure of actions for synthesizing, publishing, and deploying.
     */
    readonly pipeline: PipelineBase;
    /**
     * Whether to use a single publisher job for each type of asset.
     *
     * @remarks
     * If `true`, each asset type (e.g., file assets, Docker images) will be published by a single job in the workflow,
     * consolidating multiple asset publication steps into one job. This can reduce the total number of jobs needed,
     * making the workflow more efficient when dealing with large numbers of assets.
     *
     * Defaults to `false`, meaning each asset is published in its own job.
     */
    readonly singlePublisherPerAssetType?: boolean;
    /** Configuration options for individual stacks in the pipeline. */
    readonly stackOptions: Record<string, StackOptions>;
    /**
     * Optional phases to execute before or after main build and publish steps.
     *
     * @remarks
     * Defines custom phases (e.g., pre- and post-build/publish) that run at specific points in the pipeline workflow,
     * allowing for additional setup, cleanup, or validation steps.
     */
    readonly phases?: PipelinePhases;
    /** Provider for AWS credentials required to interact with AWS services. */
    readonly awsCredentials: IAwsCredentialsProvider;
    /**
     * Docker credentials required for registry authentication within the workflow.
     *
     * @remarks
     * Specify one or more `DockerCredentials` instances for authenticating against Docker
     * registries (such as DockerHub, ECR, GHCR, or custom registries) used in the pipeline.
     */
    readonly dockerCredentials?: DockerCredentials[];
    /** Overrides for specific action versions in GitHub Actions. */
    readonly versionOverrides?: Record<string, string>;
    /** Directory where CDK generates CloudFormation templates. */
    readonly cdkoutDir: string;
}
/**
 * Represents a GitHub Actions workflow to manage CDK pipeline jobs for synthesizing, publishing, and deploying AWS resources.
 *
 * @remarks
 * Extends `Workflow` from `github-actions-cdk`, and provides structured job orchestration based on the AWS CDK pipeline graph.
 */
export declare class PipelineWorkflow extends Workflow {
    private readonly pipelineJobProps;
    private readonly stackOptions;
    private readonly assetHashMap;
    /**
     * Initializes a new `PipelineWorkflow`.
     *
     * @param scope - The scope within which this construct is created.
     * @param id - The unique identifier for this workflow.
     * @param props - Configuration properties for the pipeline workflow.
     */
    constructor(scope: Construct, id: string, props: PipelineWorkflowProps);
    /**
     * Creates a job for synthesizing the CDK application.
     *
     * @param id - Unique identifier for the synth job.
     * @param needs - List of dependencies for this job.
     * @param synth - Synth step configuration.
     * @param preBuild - Optional jobs to run before the synth job.
     * @param postBuild - Optional jobs to run after the synth job.
     */
    protected createSynthJob(id: string, needs: string[], synth: Synth, preBuild?: IJobPhase, postBuild?: IJobPhase): void;
    /**
     * Creates a job for publishing stack assets.
     *
     * @param id - Unique identifier for the publish job.
     * @param needs - List of dependencies for this job.
     * @param assets - List of assets to publish.
     */
    protected createPublishJob(id: string, needs: string[], assets: StackAsset[], prePublish?: IJobPhase, postPublish?: IJobPhase): void;
    /**
     * Creates a job for deploying a stack to AWS.
     *
     * @param id - Unique identifier for the deploy job.
     * @param needs - List of dependencies for this job.
     * @param stack - Stack deployment information.
     */
    protected createDeployJob(id: string, needs: string[], stack: StackDeployment): void;
    /**
     * Creates a job for running a stage job in the pipeline.
     *
     * @param id - Unique identifier for the stage job.
     * @param needs - List of dependencies for this job.
     * @param job - Configuration of the stage job.
     */
    protected createStageJob(id: string, needs: string[], job: StageJob): void;
    /**
     * Retrieves a list of dependencies for a given graph node.
     *
     * @param node - The graph node to analyze for dependencies.
     * @returns An array of unique IDs representing dependencies of the node.
     */
    private getDependencies;
}
