import type { Job, RegularStep } from "github-actions-cdk";
/**
 * Enumeration of Docker credential usage scenarios within a pipeline.
 *
 * @remarks
 * `DockerCredentialsUsage` defines where Docker credentials are required,
 * including the `SYNTH` phase for build/synth steps and `ASSET_PUBLISHING` for publishing assets.
 */
export declare enum DockerCredentialsUsage {
    /** Credentials used for synthesis and build steps. */
    SYNTH = "SYNTH",
    /** Credentials used for publishing assets (e.g., Docker images). */
    ASSET_PUBLISHING = "ASSET_PUBLISHING"
}
/**
 * Base class for providing Docker credentials to various registry types.
 *
 * @remarks
 * The `DockerCredentials` class provides a unified interface for handling Docker login credentials
 * across different registry providers (e.g., DockerHub, ECR, GHCR). Concrete implementations of
 * this class define specific registry login steps.
 */
export declare abstract class DockerCredentials {
    protected readonly usages?: DockerCredentialsUsage[] | undefined;
    /**
     * Factory method for DockerHub credentials.
     */
    static dockerHub(props: DockerHubCredentialsProps): DockerCredentials;
    /**
     * Factory method for AWS ECR credentials.
     */
    static ecr(props: EcrCredentialsProps): DockerCredentials;
    /**
     * Factory method for GitHub Container Registry (GHCR) credentials.
     */
    static ghcr(props?: GhcrCredentialsProps): DockerCredentials;
    /**
     * Factory method for custom registry credentials.
     */
    static customRegistry(props: CustomProviderProps): DockerCredentials;
    /**
     * Returns Docker login steps required for the specified job.
     *
     * @param job - The GitHub Actions job for which Docker login steps are generated.
     * @returns An array of `RegularStep` instances for Docker authentication.
     */
    credentialSteps(job: Job): RegularStep[];
    /**
     * @internal
     * Abstract method for registry-specific login steps.
     *
     * @remarks
     * Concrete implementations should define this method to return
     * the specific steps required for logging into the registry.
     */
    protected abstract _credentialSteps(job: Job): RegularStep[];
    /**
     * Initializes Docker credentials with specified usage scopes.
     *
     * @param usages - The usage scopes for the credentials (optional).
     */
    constructor(usages?: DockerCredentialsUsage[] | undefined);
}
/**
 * Properties required for DockerHub credentials.
 */
export interface DockerHubCredentialsProps {
    /** Environment variable key for the DockerHub username. Defaults to "DOCKERHUB_USERNAME". */
    readonly usernameKey?: string;
    /** Environment variable key for the DockerHub access token. Defaults to "DOCKERHUB_TOKEN". */
    readonly personalAccessTokenKey?: string;
    /** Usage scopes for the credentials. */
    readonly usages?: DockerCredentialsUsage[];
}
/**
 * Properties required for AWS ECR credentials.
 */
export interface EcrCredentialsProps {
    /** The ECR registry URL. */
    readonly registry: string;
    /** Usage scopes for the credentials. */
    readonly usages?: DockerCredentialsUsage[];
}
/**
 * Properties required for GitHub Container Registry (GHCR) credentials.
 */
export interface GhcrCredentialsProps {
    /** Usage scopes for the credentials. */
    readonly usages?: DockerCredentialsUsage[];
}
/**
 * Properties required for a custom Docker registry.
 */
export interface CustomProviderProps {
    /** The registry URL. */
    readonly registry: string;
    /** Environment variable key for the registry username. */
    readonly usernameKey: string;
    /** Environment variable key for the registry password. */
    readonly passwordKey: string;
    /** Usage scopes for the credentials. */
    readonly usages?: DockerCredentialsUsage[];
}
