"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.AzureDiskPersistentVolumeCachingMode = exports.AzureDiskPersistentVolumeKind = exports.EmptyDirMedium = exports.Volume = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const constructs_1 = require("constructs");
const k8s = require("./imports/k8s");
/**
 * Volume represents a named volume in a pod that may be accessed by any
 * container in the pod.
 *
 * Docker also has a concept of volumes, though it is somewhat looser and less
 * managed. In Docker, a volume is simply a directory on disk or in another
 * Container. Lifetimes are not managed and until very recently there were only
 * local-disk-backed volumes. Docker now provides volume drivers, but the
 * functionality is very limited for now (e.g. as of Docker 1.7 only one volume
 * driver is allowed per Container and there is no way to pass parameters to
 * volumes).
 *
 * A Kubernetes volume, on the other hand, has an explicit lifetime - the same
 * as the Pod that encloses it. Consequently, a volume outlives any Containers
 * that run within the Pod, and data is preserved across Container restarts. Of
 * course, when a Pod ceases to exist, the volume will cease to exist, too.
 * Perhaps more importantly than this, Kubernetes supports many types of
 * volumes, and a Pod can use any number of them simultaneously.
 *
 * At its core, a volume is just a directory, possibly with some data in it,
 * which is accessible to the Containers in a Pod. How that directory comes to
 * be, the medium that backs it, and the contents of it are determined by the
 * particular volume type used.
 *
 * To use a volume, a Pod specifies what volumes to provide for the Pod (the
 * .spec.volumes field) and where to mount those into Containers (the
 * .spec.containers[*].volumeMounts field).
 *
 * A process in a container sees a filesystem view composed from their Docker
 * image and volumes. The Docker image is at the root of the filesystem
 * hierarchy, and any volumes are mounted at the specified paths within the
 * image. Volumes can not mount onto other volumes
 */
class Volume extends constructs_1.Construct {
    constructor(scope, id, name, config) {
        super(scope, id);
        this.name = name;
        this.config = config;
    }
    /**
     * Mounts an Amazon Web Services (AWS) EBS volume into your pod.
     * Unlike emptyDir, which is erased when a pod is removed, the contents of an EBS volume are
     * persisted and the volume is unmounted. This means that an EBS volume can be pre-populated with data,
     * and that data can be shared between pods.
     *
     * There are some restrictions when using an awsElasticBlockStore volume:
     *
     * - the nodes on which pods are running must be AWS EC2 instances.
     * - those instances need to be in the same region and availability zone as the EBS volume.
     * - EBS only supports a single EC2 instance mounting a volume.
     */
    static fromAwsElasticBlockStore(scope, id, volumeId, options = {}) {
        var _b, _c, _d;
        return new Volume(scope, id, (_b = options.name) !== null && _b !== void 0 ? _b : `ebs-${volumeId}`, {
            awsElasticBlockStore: {
                volumeId,
                fsType: (_c = options.fsType) !== null && _c !== void 0 ? _c : 'ext4',
                partition: options.partition,
                readOnly: (_d = options.readOnly) !== null && _d !== void 0 ? _d : false,
            },
        });
    }
    /**
     * Mounts a Microsoft Azure Data Disk into a pod.
     */
    static fromAzureDisk(scope, id, diskName, diskUri, options = {}) {
        var _b, _c, _d, _e, _f;
        return new Volume(scope, id, (_b = options.name) !== null && _b !== void 0 ? _b : `azuredisk-${diskName}`, {
            azureDisk: {
                diskName,
                diskUri,
                cachingMode: (_c = options.cachingMode) !== null && _c !== void 0 ? _c : AzureDiskPersistentVolumeCachingMode.NONE,
                fsType: (_d = options.fsType) !== null && _d !== void 0 ? _d : 'ext4',
                kind: (_e = options.kind) !== null && _e !== void 0 ? _e : AzureDiskPersistentVolumeKind.SHARED,
                readOnly: (_f = options.readOnly) !== null && _f !== void 0 ? _f : false,
            },
        });
    }
    /**
     * Mounts a Google Compute Engine (GCE) persistent disk (PD) into your Pod.
     * Unlike emptyDir, which is erased when a pod is removed, the contents of a PD are
     * preserved and the volume is merely unmounted. This means that a PD can be pre-populated
     * with data, and that data can be shared between pods.
     *
     * There are some restrictions when using a gcePersistentDisk:
     *
     * - the nodes on which Pods are running must be GCE VMs
     * - those VMs need to be in the same GCE project and zone as the persistent disk
     */
    static fromGcePersistentDisk(scope, id, pdName, options = {}) {
        var _b, _c, _d;
        return new Volume(scope, id, (_b = options.name) !== null && _b !== void 0 ? _b : `gcedisk-${pdName}`, {
            gcePersistentDisk: {
                pdName,
                fsType: (_c = options.fsType) !== null && _c !== void 0 ? _c : 'ext4',
                partition: options.partition,
                readOnly: (_d = options.readOnly) !== null && _d !== void 0 ? _d : false,
            },
        });
    }
    /**
     * Populate the volume from a ConfigMap.
     *
     * The configMap resource provides a way to inject configuration data into
     * Pods. The data stored in a ConfigMap object can be referenced in a volume
     * of type configMap and then consumed by containerized applications running
     * in a Pod.
     *
     * When referencing a configMap object, you can simply provide its name in the
     * volume to reference it. You can also customize the path to use for a
     * specific entry in the ConfigMap.
     *
     * @param configMap The config map to use to populate the volume.
     * @param options Options
     */
    static fromConfigMap(scope, id, configMap, options = {}) {
        var _b;
        return new Volume(scope, id, (_b = options.name) !== null && _b !== void 0 ? _b : `configmap-${configMap.name}`, {
            configMap: {
                name: configMap.name,
                defaultMode: options.defaultMode,
                optional: options.optional,
                items: Volume.renderItems(options.items),
            },
        });
    }
    /**
     * An emptyDir volume is first created when a Pod is assigned to a Node, and
     * exists as long as that Pod is running on that node. As the name says, it is
     * initially empty. Containers in the Pod can all read and write the same
     * files in the emptyDir volume, though that volume can be mounted at the same
     * or different paths in each Container. When a Pod is removed from a node for
     * any reason, the data in the emptyDir is deleted forever.
     *
     * @see http://kubernetes.io/docs/user-guide/volumes#emptydir
     *
     * @param options - Additional options.
     */
    static fromEmptyDir(scope, id, name, options = {}) {
        return new Volume(scope, id, name, {
            emptyDir: {
                medium: options.medium,
                sizeLimit: options.sizeLimit
                    ? k8s.Quantity.fromString(`${options.sizeLimit.toMebibytes()}Mi`)
                    : undefined,
            },
        });
    }
    /**
     * Populate the volume from a Secret.
     *
     * A secret volume is used to pass sensitive information, such as passwords, to Pods.
     * You can store secrets in the Kubernetes API and mount them as files for use by pods
     * without coupling to Kubernetes directly.
     *
     * secret volumes are backed by tmpfs (a RAM-backed filesystem)
     * so they are never written to non-volatile storage.
     *
     * @see https://kubernetes.io/docs/concepts/storage/volumes/#secret
     *
     * @param secr The secret to use to populate the volume.
     * @param options Options
     */
    static fromSecret(scope, id, secr, options = {}) {
        var _b;
        return new Volume(scope, id, (_b = options.name) !== null && _b !== void 0 ? _b : `secret-${secr.name}`, {
            secret: {
                secretName: secr.name,
                defaultMode: options.defaultMode,
                optional: options.optional,
                items: Volume.renderItems(options.items),
            },
        });
    }
    /**
     * Used to mount a PersistentVolume into a Pod.
     * PersistentVolumeClaims are a way for users to "claim" durable storage (such as a GCE PersistentDisk or an iSCSI volume)
     * without knowing the details of the particular cloud environment.
     *
     * @see https://kubernetes.io/docs/concepts/storage/persistent-volumes/
     */
    static fromPersistentVolumeClaim(scope, id, claim, options = {}) {
        var _b, _c;
        return new Volume(scope, id, (_b = options.name) !== null && _b !== void 0 ? _b : `pvc-${claim.name}`, {
            persistentVolumeClaim: {
                claimName: claim.name,
                readOnly: (_c = options.readOnly) !== null && _c !== void 0 ? _c : false,
            },
        });
    }
    asVolume() {
        return this;
    }
    /**
     * @internal
     */
    _toKube() {
        return {
            name: this.name,
            ...this.config,
        };
    }
}
exports.Volume = Volume;
_a = JSII_RTTI_SYMBOL_1;
Volume[_a] = { fqn: "cdk8s-plus-22.Volume", version: "2.0.0-rc.8" };
/**
  * @internal
 */
Volume.renderItems = (items) => {
    if (!items) {
        return undefined;
    }
    const result = new Array();
    for (const key of Object.keys(items).sort()) {
        result.push({
            key,
            path: items[key].path,
            mode: items[key].mode,
        });
    }
    return result;
};
/**
 * The medium on which to store the volume.
 */
var EmptyDirMedium;
(function (EmptyDirMedium) {
    /**
     * The default volume of the backing node.
     */
    EmptyDirMedium["DEFAULT"] = "";
    /**
     * Mount a tmpfs (RAM-backed filesystem) for you instead. While tmpfs is very
     * fast, be aware that unlike disks, tmpfs is cleared on node reboot and any
     * files you write will count against your Container's memory limit.
     */
    EmptyDirMedium["MEMORY"] = "Memory";
})(EmptyDirMedium = exports.EmptyDirMedium || (exports.EmptyDirMedium = {}));
/**
 * Azure Disk kinds.
 */
var AzureDiskPersistentVolumeKind;
(function (AzureDiskPersistentVolumeKind) {
    /**
     * Multiple blob disks per storage account.
     */
    AzureDiskPersistentVolumeKind["SHARED"] = "Shared";
    /**
     * Single blob disk per storage account.
     */
    AzureDiskPersistentVolumeKind["DEDICATED"] = "Dedicated";
    /**
     * Azure managed data disk.
     */
    AzureDiskPersistentVolumeKind["MANAGED"] = "Managed";
})(AzureDiskPersistentVolumeKind = exports.AzureDiskPersistentVolumeKind || (exports.AzureDiskPersistentVolumeKind = {}));
/**
 * Azure disk caching modes.
 */
var AzureDiskPersistentVolumeCachingMode;
(function (AzureDiskPersistentVolumeCachingMode) {
    /**
     * None.
     */
    AzureDiskPersistentVolumeCachingMode["NONE"] = "None";
    /**
     * ReadOnly.
     */
    AzureDiskPersistentVolumeCachingMode["READ_ONLY"] = "ReadOnly";
    /**
     * ReadWrite.
     */
    AzureDiskPersistentVolumeCachingMode["READ_WRITE"] = "ReadWrite";
})(AzureDiskPersistentVolumeCachingMode = exports.AzureDiskPersistentVolumeCachingMode || (exports.AzureDiskPersistentVolumeCachingMode = {}));
//# sourceMappingURL=data:application/json;base64,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