"""HTTP exceptions and status codes for Vega Web Framework"""

from typing import Any, Dict, Optional


class HTTPException(Exception):
    """
    HTTP exception that can be raised to return an HTTP error response.

    Compatible with FastAPI's HTTPException API for easy migration.

    Args:
        status_code: HTTP status code
        detail: Error message or detail object
        headers: Optional HTTP headers to include in the response

    Example:
        raise HTTPException(status_code=404, detail="User not found")
        raise HTTPException(
            status_code=401,
            detail="Not authenticated",
            headers={"WWW-Authenticate": "Bearer"}
        )
    """

    def __init__(
        self,
        status_code: int,
        detail: Any = None,
        headers: Optional[Dict[str, str]] = None
    ) -> None:
        self.status_code = status_code
        self.detail = detail
        self.headers = headers
        super().__init__(detail)

    def __repr__(self) -> str:
        return f"{self.__class__.__name__}(status_code={self.status_code}, detail={self.detail})"


class ValidationError(HTTPException):
    """Raised when request validation fails (422)"""

    def __init__(self, detail: Any = "Validation Error", headers: Optional[Dict[str, str]] = None):
        super().__init__(status_code=422, detail=detail, headers=headers)


class NotFoundError(HTTPException):
    """Raised when resource is not found (404)"""

    def __init__(self, detail: Any = "Not Found", headers: Optional[Dict[str, str]] = None):
        super().__init__(status_code=404, detail=detail, headers=headers)


class UnauthorizedError(HTTPException):
    """Raised when authentication is required (401)"""

    def __init__(self, detail: Any = "Unauthorized", headers: Optional[Dict[str, str]] = None):
        super().__init__(status_code=401, detail=detail, headers=headers)


class ForbiddenError(HTTPException):
    """Raised when access is forbidden (403)"""

    def __init__(self, detail: Any = "Forbidden", headers: Optional[Dict[str, str]] = None):
        super().__init__(status_code=403, detail=detail, headers=headers)


class BadRequestError(HTTPException):
    """Raised for bad requests (400)"""

    def __init__(self, detail: Any = "Bad Request", headers: Optional[Dict[str, str]] = None):
        super().__init__(status_code=400, detail=detail, headers=headers)


# HTTP Status codes - compatible with FastAPI's status module
class status:
    """HTTP status codes (compatible with fastapi.status)"""

    # 1xx Informational
    HTTP_100_CONTINUE = 100
    HTTP_101_SWITCHING_PROTOCOLS = 101
    HTTP_102_PROCESSING = 102
    HTTP_103_EARLY_HINTS = 103

    # 2xx Success
    HTTP_200_OK = 200
    HTTP_201_CREATED = 201
    HTTP_202_ACCEPTED = 202
    HTTP_203_NON_AUTHORITATIVE_INFORMATION = 203
    HTTP_204_NO_CONTENT = 204
    HTTP_205_RESET_CONTENT = 205
    HTTP_206_PARTIAL_CONTENT = 206
    HTTP_207_MULTI_STATUS = 207
    HTTP_208_ALREADY_REPORTED = 208
    HTTP_226_IM_USED = 226

    # 3xx Redirection
    HTTP_300_MULTIPLE_CHOICES = 300
    HTTP_301_MOVED_PERMANENTLY = 301
    HTTP_302_FOUND = 302
    HTTP_303_SEE_OTHER = 303
    HTTP_304_NOT_MODIFIED = 304
    HTTP_305_USE_PROXY = 305
    HTTP_306_RESERVED = 306
    HTTP_307_TEMPORARY_REDIRECT = 307
    HTTP_308_PERMANENT_REDIRECT = 308

    # 4xx Client Error
    HTTP_400_BAD_REQUEST = 400
    HTTP_401_UNAUTHORIZED = 401
    HTTP_402_PAYMENT_REQUIRED = 402
    HTTP_403_FORBIDDEN = 403
    HTTP_404_NOT_FOUND = 404
    HTTP_405_METHOD_NOT_ALLOWED = 405
    HTTP_406_NOT_ACCEPTABLE = 406
    HTTP_407_PROXY_AUTHENTICATION_REQUIRED = 407
    HTTP_408_REQUEST_TIMEOUT = 408
    HTTP_409_CONFLICT = 409
    HTTP_410_GONE = 410
    HTTP_411_LENGTH_REQUIRED = 411
    HTTP_412_PRECONDITION_FAILED = 412
    HTTP_413_REQUEST_ENTITY_TOO_LARGE = 413
    HTTP_414_REQUEST_URI_TOO_LONG = 414
    HTTP_415_UNSUPPORTED_MEDIA_TYPE = 415
    HTTP_416_REQUESTED_RANGE_NOT_SATISFIABLE = 416
    HTTP_417_EXPECTATION_FAILED = 417
    HTTP_418_IM_A_TEAPOT = 418
    HTTP_421_MISDIRECTED_REQUEST = 421
    HTTP_422_UNPROCESSABLE_ENTITY = 422
    HTTP_423_LOCKED = 423
    HTTP_424_FAILED_DEPENDENCY = 424
    HTTP_425_TOO_EARLY = 425
    HTTP_426_UPGRADE_REQUIRED = 426
    HTTP_428_PRECONDITION_REQUIRED = 428
    HTTP_429_TOO_MANY_REQUESTS = 429
    HTTP_431_REQUEST_HEADER_FIELDS_TOO_LARGE = 431
    HTTP_451_UNAVAILABLE_FOR_LEGAL_REASONS = 451

    # 5xx Server Error
    HTTP_500_INTERNAL_SERVER_ERROR = 500
    HTTP_501_NOT_IMPLEMENTED = 501
    HTTP_502_BAD_GATEWAY = 502
    HTTP_503_SERVICE_UNAVAILABLE = 503
    HTTP_504_GATEWAY_TIMEOUT = 504
    HTTP_505_HTTP_VERSION_NOT_SUPPORTED = 505
    HTTP_506_VARIANT_ALSO_NEGOTIATES = 506
    HTTP_507_INSUFFICIENT_STORAGE = 507
    HTTP_508_LOOP_DETECTED = 508
    HTTP_510_NOT_EXTENDED = 510
    HTTP_511_NETWORK_AUTHENTICATION_REQUIRED = 511
