from __future__ import unicode_literals

import datetime
import json
import subprocess
import sys
import threading
import uuid
from logging import getLogger
from typing import Any, Dict, List, Tuple

import nbformat
from flask import Blueprint, abort, jsonify, render_template, request, url_for

from notebooker import execute_notebook
from notebooker.constants import JobStatus
from notebooker.serialization.serialization import get_fresh_serializer
from notebooker.utils.conversion import generate_ipynb_from_py
from notebooker.utils.filesystem import get_output_dir, get_template_dir
from notebooker.utils.templates import _get_parameters_cell_idx, _get_preview, get_all_possible_templates
from notebooker.utils.web import (
    convert_report_name_url_to_path,
    json_to_python,
    validate_generate_pdf_output,
    validate_mailto,
    validate_title,
)
from notebooker.web.handle_overrides import handle_overrides
from notebooker.web.utils import get_serializer

try:
    FileNotFoundError
except NameError:
    FileNotFoundError = IOError

run_report_bp = Blueprint("run_report_bp", __name__)
logger = getLogger(__name__)


@run_report_bp.route("/run_report/get_preview/<path:report_name>", methods=["GET"])
def run_report_get_preview(report_name):
    """
    Get a preview of the Notebook Template which is about to be executed.

    :param report_name: The parameter here should be a "/"-delimited string which mirrors the directory structure of \
        the notebook templates.

    :returns: An HTML rendering of a notebook template which has been converted from .py -> .ipynb -> .html
    """
    report_name = convert_report_name_url_to_path(report_name)
    # Handle the case where a rendered ipynb asks for "custom.css"
    if ".css" in report_name:
        return ""
    return _get_preview(report_name)


@run_report_bp.route("/run_report/<path:report_name>", methods=["GET"])
def run_report_http(report_name):
    """
    The "Run Report" interface is generated by this method.

    :param report_name: The parameter here should be a "/"-delimited string which mirrors the directory structure of \
        the notebook templates.

    :returns: An HTML template which is the Run Report interface.
    """
    report_name = convert_report_name_url_to_path(report_name)
    json_params = request.args.get("json_params")
    initial_python_parameters = json_to_python(json_params) or ""
    try:
        path = generate_ipynb_from_py(get_template_dir(), report_name)
    except FileNotFoundError as e:
        logger.exception(e)
        return "", 404
    nb = nbformat.read(path, as_version=nbformat.v4.nbformat)
    metadata_idx = _get_parameters_cell_idx(nb)
    parameters_as_html = ""
    has_prefix = has_suffix = False
    if metadata_idx is not None:
        metadata = nb["cells"][metadata_idx]
        parameters_as_html = metadata["source"].strip()
        has_prefix, has_suffix = (bool(nb["cells"][:metadata_idx]), bool(nb["cells"][metadata_idx + 1 :]))
    logger.info("initial_python_parameters = {}".format(initial_python_parameters))
    return render_template(
        "run_report.html",
        parameters_as_html=parameters_as_html,
        has_prefix=has_prefix,
        has_suffix=has_suffix,
        report_name=report_name,
        all_reports=get_all_possible_templates(),
        initialPythonParameters=initial_python_parameters,
    )


def _monitor_stderr(process, job_id):
    stderr = []
    # Unsure whether flask app contexts are thread-safe; just reinitialise the serializer here.
    result_serializer = get_fresh_serializer()
    while True:
        line = process.stderr.readline().decode("utf-8")
        if line == "" and process.poll() is not None:
            break
        stderr.append(line)
        logger.info(line)  # So that we have it in the log, not just in memory.
        result_serializer.update_stdout(job_id, new_lines=[line])
    return "".join(stderr)


def run_report(report_name, report_title, mailto, overrides, generate_pdf_output=False, prepare_only=False):
    """
    Actually run the report in earnest.
    Uses a subprocess to execute the report asynchronously, which is identical to the non-webapp entrypoint.
    :param report_name: `str` The report which we are executing
    :param report_title: `str` The user-specified title of the report
    :param mailto: `Optional[str]` Who the results will be emailed to
    :param overrides: `Optional[Dict[str, Any]]` The parameters to be passed into the report
    :param generate_pdf_output: `bool` Whether we're generating a PDF. Defaults to False.
    :param prepare_only: `bool` Whether to do everything except execute the notebook. Useful for testing.
    :return: The unique job_id.
    """
    job_id = str(uuid.uuid4())
    job_start_time = datetime.datetime.now()
    result_serializer = get_serializer()
    result_serializer.save_check_stub(
        job_id,
        report_name,
        report_title=report_title,
        job_start_time=job_start_time,
        status=JobStatus.SUBMITTED,
        overrides=overrides,
        mailto=mailto,
        generate_pdf_output=generate_pdf_output,
    )
    p = subprocess.Popen(
        [
            sys.executable,
            "-m",
            execute_notebook.__name__,
            "--job-id",
            job_id,
            "--output-base-dir",
            get_output_dir(),
            "--template-base-dir",
            get_template_dir(),
            "--report-name",
            report_name,
            "--report-title",
            report_title,
            "--mailto",
            mailto,
            "--overrides-as-json",
            json.dumps(overrides),
            "--mongo-db-name",
            result_serializer.database_name,
            "--mongo-host",
            result_serializer.mongo_host,

            *(
                ("--mongo-user", result_serializer.user)
                if result_serializer.user is not None else
                ()
            ),
            *(
                ("--mongo-password", result_serializer.password)
                if result_serializer.password is not None else
                ()
            ),
            "--result-collection-name",
            result_serializer.result_collection_name,
            "--pdf-output" if generate_pdf_output else "--no-pdf-output",
            "--serializer-cls",
            result_serializer.__class__.__name__,
        ]
        + (["--prepare-notebook-only"] if prepare_only else []),
        stderr=subprocess.PIPE,
    )
    stderr_thread = threading.Thread(target=_monitor_stderr, args=(p, job_id))
    stderr_thread.daemon = True
    stderr_thread.start()
    return job_id


def _handle_run_report(
    report_name: str, overrides_dict: Dict[str, Any], issues: List[str]
) -> Tuple[str, int, Dict[str, str]]:
    # Find and cleanse the title of the report
    report_title = validate_title(request.values.get("report_title"), issues)
    # Get mailto email address
    mailto = validate_mailto(request.values.get("mailto"), issues)
    # Find whether to generate PDF output
    generate_pdf_output = validate_generate_pdf_output(request.values.get("generatepdf"), issues)
    if issues:
        return jsonify({"status": "Failed", "content": ("\n".join(issues))})
    report_name = convert_report_name_url_to_path(report_name)
    job_id = run_report(report_name, report_title, mailto, overrides_dict, generate_pdf_output=generate_pdf_output)
    return (
        jsonify({"id": job_id}),
        202,  # HTTP Accepted code
        {"Location": url_for("pending_results_bp.task_status", report_name=report_name, job_id=job_id)},
    )


@run_report_bp.route("/run_report_json/<path:report_name>", methods=["POST"])
def run_report_json(report_name):
    """
    Execute a notebook from a JSON request.

    :param report_name: The parameter here should be a "/"-delimited string which mirrors the directory structure of \
        the notebook templates.

    :returns: 202-redirects to the "task_status" interface.
    """
    issues = []
    # Get JSON overrides
    overrides_dict = json.loads(request.values.get("overrides"))
    return _handle_run_report(report_name, overrides_dict, issues)


@run_report_bp.route("/run_report/<path:report_name>", methods=["POST"])
def run_checks_http(report_name):
    """
    Execute a notebook from an HTTP request.

    :param report_name: The parameter here should be a "/"-delimited string which mirrors the directory structure of \
        the notebook templates.

    :returns: 202-redirects to the "task_status" interface.
    """
    issues = []
    # Get and process raw python overrides
    overrides_dict = handle_overrides(request.values.get("overrides"), issues)
    return _handle_run_report(report_name, overrides_dict, issues)


def _rerun_report(job_id, prepare_only=False):
    result = get_serializer().get_check_result(job_id)
    if not result:
        abort(404)
    prefix = "Rerun of "
    title = result.report_title if result.report_title.startswith(prefix) else (prefix + result.report_title)
    new_job_id = run_report(
        result.report_name,
        title,
        result.mailto,
        result.overrides,
        generate_pdf_output=result.generate_pdf_output,
        prepare_only=prepare_only,
    )
    return new_job_id


@run_report_bp.route("/rerun_report/<job_id>/<path:report_name>", methods=["POST"])
def rerun_report(job_id, report_name):
    """
    Rerun a notebook using its already-existing parameters.

    :param job_id: The Job ID of the report which we are rerunning.
    :param report_name: The parameter here should be a "/"-delimited string which mirrors the directory structure of \
        the notebook templates.

    :returns: 202-redirects to the "task_status" interface.
    """
    new_job_id = _rerun_report(job_id)
    return jsonify(
        {"results_url": url_for("serve_results_bp.task_results", report_name=report_name, job_id=new_job_id)}
    )
