# Rules for Creating New Type Definitions

When creating new type definitions in `arize_toolkit/types.py`, follow these established patterns to ensure consistency, validation, and ease of use.

## IMPORTANT: Research Required Fields First

**Never create enum fields without verifying the actual valid values.** Always research and request the proper field definitions before implementing.

### 1. GraphQL Schema Investigation

Before creating any new enum type, you must:

**Request GraphQL Schema Documentation:**

- Ask for the specific GraphQL enum type definition
- Request the complete list of valid values from the schema
- Get the exact spelling and casing used by the API

**Example Request:**

```
"Can you provide the GraphQL schema definition for [EnumTypeName]? 
I need the complete list of valid enum values and their exact spelling 
to create the Python type definition."
```

**Use Schema Introspection:**

- If you have access to the GraphQL endpoint, use introspection queries
- Look for existing enum definitions in the schema
- Check field types that reference the enum you're trying to create

### 2. API Documentation Review

**Check Existing Documentation:**

- Review API documentation for the endpoint/field
- Look for parameter validation rules
- Check example requests/responses for valid values

**Search Existing Codebase:**

- Look for similar enum usage in existing queries
- Check GraphQL query definitions for field usage
- Review existing model field definitions that might use the enum

**Example Search Commands:**

```bash
# Search for existing enum usage
grep -r "enum_field_name" arize_toolkit/
grep -r "EnumTypeName" arize_toolkit/

# Look for GraphQL field definitions
grep -r "fieldName:" arize_toolkit/queries/
```

### 3. Validation Against Real Usage

**Check Test Files:**

- Review existing tests for valid parameter values
- Look at test mock data for realistic enum values
- Check integration tests for API usage patterns

**Review Client Method Usage:**

- Look at client methods that would use the enum
- Check parameter validation in existing functions
- Review docstring examples for valid values

### 4. Request Verification

When unsure about enum values, always request verification:

**For GraphQL Enums:**

```
"I need to create a Python enum for the GraphQL type [TypeName]. 
Can you provide:
1. The complete GraphQL enum definition
2. All valid enum values with exact casing
3. Any aliases or deprecated values that should be supported
4. The primary/canonical value for each option"
```

**For API Parameters:**

```
"I'm creating an enum for the [parameter_name] field. 
What are all the valid values this parameter accepts?
Please include:
- Exact spelling and casing
- Any legacy/deprecated values still supported
- The preferred/canonical form for each value"
```

## Type Structure

### 1. Base Class Usage

All custom types must inherit from `InputValidationEnum`:

```python
from enum import Enum

class InputValidationEnum(Enum):
    @classmethod
    def _missing_(cls, value):
        # Search through all enum members and their aliases
        for member in cls:
            # Check if the value matches any of the values in the member's value tuple
            if isinstance(member.value, tuple) and value in member.value:
                return member
        return None

    @classmethod
    def from_input(cls, user_input):
        for operator in cls:
            if user_input in operator.value:
                return operator.name
        raise ValueError(f"{user_input} is not a valid {cls.__name__}")

class YourNewType(InputValidationEnum):
    # Your enum values here - ONLY after verifying with schema/API
```

### 2. Value Definition Patterns

Use tuples to define multiple aliases for the same logical value:

```python
class ExampleType(InputValidationEnum):
    # PRIMARY VALUE MUST MATCH API/SCHEMA EXACTLY
    primary_value = "primary_value", "Primary Value", "primaryValue"

    # Include common variations users might input
    secondary_value = (
        "secondary_value",  # ← This MUST be the exact API value
        "secondaryValue",
        "Secondary Value",
        "SECONDARY_VALUE",
        "secondary",
    )
```

**Critical Rules:**

- **First value in tuple MUST be the exact API/GraphQL value**
- Additional values are user-convenience aliases
- Never guess API values - always verify first

### 3. Verification Workflow

Before implementing any enum:

1. **Identify the Source:**

   ```python
   # Document where the enum values come from
   class NewEnumType(InputValidationEnum):
       """Enum for [purpose].

       Values verified from:
       - GraphQL schema: [schema_location]
       - API documentation: [doc_link]
       - Date verified: [date]
       """
   ```

1. **Request Schema Definition:**

   - Always ask for the complete GraphQL enum definition
   - Get examples of usage in actual queries
   - Verify casing and spelling

1. **Cross-Reference Usage:**

   - Check how the enum is used in existing GraphQL queries
   - Look for patterns in test data
   - Verify against client method expectations

1. **Test with Real Data:**

   - Use actual API responses to validate enum values
   - Test with the GraphQL endpoint if available
   - Verify error messages for invalid values

## Example: Proper Research Process

**Step 1: Identify Need**

```python
# In a model, you see:
some_field: Optional[str] = Field(description="The widget type")

# But you know it should be an enum. DON'T GUESS the values!
```

**Step 2: Research Request**

```
"I see a field 'some_field' that appears to be an enumerated type 
but is currently defined as a string. Can you provide:

1. The GraphQL schema definition for this field
2. All valid enum values with exact spelling
3. Any context about what these values represent
4. Examples of usage in existing queries"
```

**Step 3: Implementation After Verification**

```python
# Only after receiving the actual schema definition:
class WidgetType(InputValidationEnum):
    """Widget types supported by the dashboard API.

    Values verified from GraphQL schema on 2024-01-01.
    Schema location: Dashboard.widgets.type
    """

    # These values come from the actual GraphQL schema:
    statistic = "statistic", "Statistic", "stat", "statistical"
    line_chart = "lineChart", "Line Chart", "line", "chart"
    bar_chart = "barChart", "Bar Chart", "bar", "histogram"
```

### 4. Naming Conventions

#### Enum Class Names

- Use PascalCase: `PerformanceMetric`, `DriftMetric`, `ComparisonOperator`
- Be descriptive and domain-specific
- Avoid generic names like `Type` or `Enum`

#### Enum Value Names

- Use snake_case for the primary identifier: `f_1`, `euclidean_distance`, `greater_than`
- **Primary identifier MUST match the API exactly**
- Keep them concise but clear

#### Alias Patterns

After verifying the canonical API value, include these variations:

1. **API/Schema value**: `performanceMetric` (REQUIRED - must be first)
1. **snake_case**: `performance_metric`
1. **PascalCase**: `PerformanceMetric`
1. **UPPER_CASE**: `PERFORMANCE_METRIC`
1. **Human-readable**: `"Performance Metric"`
1. **Common abbreviations**: `"PM"`, `"perf_metric"`
1. **Alternative spellings**: `"grey"`, `"gray"`

## Domain-Specific Patterns

### 1. Metrics and Measurements

```python
class NewMetricType(InputValidationEnum):
    # Include mathematical notation and common names
    mean_squared_error = "mse", "Mean Squared Error", "MSE", "mean_squared_error"
    root_mean_squared_error = (
        "rmse",
        "Root Mean Squared Error",
        "RMSE",
        "root_mean_squared_error",
    )

    # Include domain-specific terminology
    area_under_curve = "auc", "Area Under Curve", "AUC", "area_under_curve", "roc_auc"
```

### 2. Operators and Comparisons

```python
class NewOperatorType(InputValidationEnum):
    # Include symbolic and word representations
    greater_than = "greaterThan", "Greater Than", ">", "gt"
    less_than = "lessThan", "Less Than", "<", "lt"
    equals = "equals", "Equals", "=", "==", "eq"
    not_equals = "notEquals", "Not Equal", "!=", "ne", "not_equal"
```

### 3. Technical Configurations

```python
class NewConfigType(InputValidationEnum):
    # Include both technical and user-friendly names
    json_format = "json", "JSON", "json_format", "application/json"
    xml_format = "xml", "XML", "xml_format", "application/xml", "text/xml"
```

### 4. Model and Data Types

```python
class NewModelType(InputValidationEnum):
    # Include both internal names and user-facing descriptions
    classification = (
        "classification",
        "Classification",
        "score_categorical",  # Internal system name
        "categorical",
        "clf",
    )
    regression = (
        "regression",
        "Regression",
        "numeric",  # Internal system name
        "continuous",
        "reg",
    )
```

## Best Practices

### 1. Research Existing Usage

Before creating a new type:

- Check if similar functionality exists in other enum types
- Review GraphQL schema documentation for expected values
- Check API documentation for valid parameter values
- Look at existing test cases for usage patterns

### 2. Include Comprehensive Aliases

Always consider:

- **API compatibility**: Include values expected by the GraphQL API
- **User convenience**: Include values users would naturally type
- **Legacy support**: Include deprecated but still-used values
- **Cross-platform**: Include variations from different systems

### 3. Validation and Error Handling

The base `InputValidationEnum` provides:

- Automatic alias resolution via `_missing_`
- Input validation via `from_input`
- Clear error messages for invalid inputs

### 4. Documentation

Add clear docstrings for complex types:

```python
class ComplexType(InputValidationEnum):
    """Enumeration for complex configuration options.

    This enum supports multiple input formats for user convenience
    while maintaining consistency with the underlying API.
    """

    option_one = "option_one", "Option One", "opt1"
    option_two = "option_two", "Option Two", "opt2"
```

## Testing Requirements

When adding new type definitions:

1. **Test all aliases** in the corresponding test file:

```python
def test_new_type_aliases():
    """Test that all aliases resolve correctly"""
    assert NewType.from_input("primary_value") == "primary_value"
    assert NewType.from_input("Primary Value") == "primary_value"
    assert NewType.from_input("primaryValue") == "primary_value"
```

2. **Test validation errors**:

```python
def test_new_type_invalid_input():
    """Test that invalid inputs raise appropriate errors"""
    with pytest.raises(ValueError, match="not a valid NewType"):
        NewType.from_input("invalid_value")
```

3. **Test in real usage contexts** where the type is used in models or client methods.

## Integration with Models

When using new types in model definitions:

```python
from arize_toolkit.types import YourNewType


class YourModel(GraphQLModel):
    some_field: Optional[YourNewType] = Field(
        default=None, description="Description of the field using YourNewType"
    )
```

## Validation Checklist

Before submitting a new enum type:

- [ ] Verified exact API/GraphQL values with schema documentation
- [ ] Confirmed spelling and casing of all canonical values
- [ ] Checked existing codebase for similar enum usage
- [ ] Added appropriate user-friendly aliases
- [ ] Included docstring with verification source and date
- [ ] Tested with actual API data if possible
- [ ] Added comprehensive test coverage for all aliases

## Error Prevention

**Common Mistakes to Avoid:**

❌ **Don't guess enum values:**

```python
# WRONG - guessing what the values might be
class BadEnum(InputValidationEnum):
    probably_this = "probably_this", "Probably This"
    maybe_that = "maybe_that", "Maybe That"
```

✅ **Do verify with schema first:**

```python
# RIGHT - verified with actual GraphQL schema
class GoodEnum(InputValidationEnum):
    """Verified from GraphQL schema 2024-01-01"""

    exact_api_value = "exactApiValue", "Exact API Value", "exact"
```

❌ **Don't assume casing:**

```python
# WRONG - assuming camelCase
performance_metric = "performanceMetric"  # Might be "performance_metric" in API
```

✅ **Do verify exact casing:**

```python
# RIGHT - verified exact API spelling
performance_metric = "performance_metric"  # Confirmed from schema
```

## Documentation Requirements

Always document the source of enum values:

```python
class NewEnum(InputValidationEnum):
    """Enumeration for [purpose].

    Values verified from:
    - GraphQL schema: [schema_path or introspection query]
    - API endpoint: [endpoint_name]
    - Documentation: [doc_link]
    - Verification date: [YYYY-MM-DD]

    Last updated: [date] by [person]
    """

    verified_value = "apiValue", "User Friendly Name", "alias"
```

## Common Patterns by Domain

### Performance Metrics

- Include mathematical symbols and full names
- Support both abbreviated and full forms
- Consider domain-specific terminology (ML, statistics, etc.)

### Data Types

- Include both technical and user-friendly names
- Support common variations and synonyms
- Consider database and API naming conventions

### Operators

- Include symbolic representations
- Support both programming and natural language forms
- Include common abbreviations

### Configuration Options

- Include both snake_case and camelCase
- Support both technical and descriptive names
- Consider backwards compatibility

## Example: Complete Type Definition

```python
class ExampleMetricType(InputValidationEnum):
    """Metrics for evaluating model performance.

    Supports multiple input formats including mathematical notation,
    full names, and common abbreviations.
    """

    accuracy = "accuracy", "Accuracy", "acc", "correct_rate"
    precision = "precision", "Precision", "prec", "positive_predictive_value", "ppv"
    recall = "recall", "Recall", "sensitivity", "true_positive_rate", "tpr"
    f_1 = "f_1", "F1 Score", "f1", "F1", "f_score", "f_measure"

    # More complex example with many aliases
    area_under_curve = (
        "auc",
        "Area Under Curve",
        "AUC",
        "area_under_curve",
        "roc_auc",
        "ROC AUC",
        "auroc",
        "AUROC",
    )
```

Following these patterns ensures that new type definitions are accurate, consistent, and maintainable while preventing errors caused by incorrect enum values.
