# GraphQL Query Test Rules

When creating tests for GraphQL queries and mutations in the Arize Toolkit, follow these guidelines to ensure consistent, maintainable, and comprehensive test coverage.

## Test Class Structure

1. **Class Naming**

   - Name test classes after the query/mutation they test: `Test{QueryName}Query` or `Test{MutationName}Mutation`
   - Example: `TestGetModelQuery`, `TestCreateFileImportJobMutation`

1. **Test Method Organization**

   - Group related tests within the class
   - Use descriptive method names that indicate the scenario being tested
   - Follow the pattern: `test_{query_name}_{scenario}`
   - Example: `test_get_model_query_success`, `test_create_file_import_job_failure`

## Test Method Structure

1. **Method Documentation**

   ```python
   def test_query_name_scenario(self, gql_client):
       """Test [scenario description]."""
       # Test implementation
   ```

1. **Test Components**

   - Mock Response Setup
   - Query/Mutation Execution
   - Assertions
   - Client Verification

1. **Mock Response Pattern**
   Always use the `gql_client` fixture defined in conftest.py to mock requests to the graphql client.

   ```python
   mock_response = {
       "queryName": {  # or "mutationName" for mutations
           "resultField": {"field1": "value1", "field2": "value2"}
       }
   }
   gql_client.execute.return_value = mock_response
   ```

## Test Categories

1. **Query Structure Tests**

   ```python
   def test_query_structure(self):
       """Test that the query structure is correct and includes all necessary fields."""
       query = QueryClass.graphql_query
       assert "query QueryName" in query  # or "mutation MutationName"
       assert "expectedField" in query
   ```

1. **Success Case Tests**

   ```python
   def test_query_name_success(self, gql_client):
       """Test successful query execution."""
       mock_response = {...}
       gql_client.execute.return_value = mock_response

       result = QueryClass.run_graphql_query(gql_client, **input_params)

       assert result.field1 == expected_value1
       assert result.field2 == expected_value2
       gql_client.execute.assert_called_once()
   ```

1. **Error Case Tests**

   ```python
   def test_query_name_error(self, gql_client):
       """Test error handling."""
       mock_response = {"errors": [{"message": "Error message"}]}
       gql_client.execute.return_value = mock_response

       with pytest.raises(QueryClass.QueryException, match="Error message"):
           QueryClass.run_graphql_query(gql_client, **input_params)
   ```

1. **Input Validation Tests**

   ```python
   @pytest.mark.parametrize(
       "input_data,expected_error",
       [
           ({"invalid_field": "value"}, "field_name"),
           ({"missing_required": "value"}, "required_field"),
       ],
   )
   def test_query_name_input_validation(self, input_data, expected_error):
       """Test input validation."""
       with pytest.raises(Exception) as e:
           InputModel.model_validate(input_data)
       assert expected_error in str(e)
   ```

## Fixtures

1. **Common Fixtures**

   ```python
   @pytest.fixture
   def mock_response_data():
       return {"field1": "value1", "field2": "value2"}


   @pytest.fixture
   def mock_input_data():
       return {"required_field": "value", "optional_field": "value"}
   ```

1. **Model Fixtures**

   ```python
   @pytest.fixture
   def mock_model_data():
       return ModelClass(field1="value1", field2="value2")
   ```

## Best Practices

1. **Test Coverage**

   - Test query structure
   - Test successful execution
   - Test error handling
   - Test input validation
   - Test edge cases (empty results, pagination, etc.)

1. **Mocking**

   - Use `gql_client` fixture for GraphQL client
   - Mock responses should match API response structure
   - Verify client calls with `assert_called_once()`

1. **Assertions**

   - Assert all relevant fields in the response
   - Use specific assertions for each field
   - Include type checking where appropriate
   - Verify error messages match expected patterns

1. **Input Validation**

   - Test required fields
   - Test field type validation
   - Test enum value validation
   - Test custom validation rules

1. **Error Handling**

   - Test API errors
   - Test validation errors
   - Test missing data scenarios
   - Test invalid input scenarios

## Example Test Class

```python
class TestGetModelQuery:
    def test_query_structure(self):
        """Test that the query structure is correct."""
        query = GetModelQuery.graphql_query
        assert "query GetModel" in query
        assert "model" in query
        assert "id" in query
        assert "name" in query

    def test_get_model_query_success(self, gql_client):
        """Test successful model retrieval."""
        mock_response = {
            "node": {
                "models": {
                    "edges": [
                        {
                            "node": {
                                "id": "1",
                                "name": "TestModel",
                                "modelType": "numeric",
                            }
                        }
                    ]
                }
            }
        }
        gql_client.execute.return_value = mock_response

        result = GetModelQuery.run_graphql_query(
            gql_client, model_name="TestModel", space_id="123"
        )

        assert result.id == "1"
        assert result.name == "TestModel"
        assert result.modelType.name == "numeric"
        gql_client.execute.assert_called_once()

    def test_get_model_query_no_model(self, gql_client):
        """Test handling of non-existent model."""
        mock_response = {"node": {"models": {"edges": []}}}
        gql_client.execute.return_value = mock_response

        with pytest.raises(
            GetModelQuery.QueryException, match="No model found with the given name"
        ):
            GetModelQuery.run_graphql_query(
                gql_client, model_name="NonExistentModel", space_id="123"
            )
```

## Common Patterns

1. **Pagination Tests**

   ```python
   def test_query_pagination(self, gql_client):
       """Test pagination handling."""
       mock_response = {
           "node": {
               "items": {
                   "pageInfo": {"hasNextPage": True, "endCursor": "cursor123"},
                   "edges": [...],
               }
           }
       }
       gql_client.execute.return_value = mock_response

       result = QueryClass.iterate_over_pages(gql_client, **params)
       assert len(result) == expected_count
   ```

1. **Model Validation Tests**

   ```python
   def test_model_validation(self):
       """Test model validation rules."""
       with pytest.raises(ValidationError) as e:
           ModelClass(required_field=None, invalid_enum="invalid_value")
       assert "required_field" in str(e)
       assert "invalid_enum" in str(e)
   ```

1. **Complex Input Tests**

   ```python
   def test_complex_input_validation(self):
       """Test validation of complex nested inputs."""
       input_data = {
           "field1": "value1",
           "nested": {"field2": "value2", "array": [{"field3": "value3"}]},
       }
       result = ComplexInputModel.model_validate(input_data)
       assert result.field1 == "value1"
       assert result.nested.field2 == "value2"
   ```

   References
