# Arize Toolkit Development Guidelines

## Your Role

You are a collaborative developer on the Arize Toolkit project, working alongside the team to maintain and enhance this GraphQL-based Python SDK. Your contributions should seamlessly integrate with the existing codebase while respecting established patterns and conventions.

## Core Principles

### 1. Repository Structure Integrity

- **Respect the established architecture**: The repository follows a clear separation of concerns:
  - `arize_toolkit/models/` - Pydantic models for GraphQL types
  - `arize_toolkit/queries/` - GraphQL query and mutation definitions
  - `arize_toolkit/client.py` - High-level client methods
  - `tests/` - Comprehensive test coverage mirroring the source structure
- **Maintain consistency**: New features should follow the same organizational patterns as existing code

### 2. Development Patterns

#### GraphQL Queries/Mutations

- Every new query/mutation must:
  - Inherit from `BaseQuery`
  - Define nested classes: `Variables`, `QueryException`, and `QueryResponse`
  - Implement `_parse_graphql_result()` for response parsing
  - Use the model's `to_graphql_fields()` method for field selection

#### Models

- Use Pydantic `BaseModel` with appropriate field validators
- Implement `to_graphql_fields()` class method for GraphQL field generation
- Follow the established naming conventions (camelCase for GraphQL, snake_case for Python)

#### Client Methods

- Provide high-level, user-friendly interfaces in `client.py`
- Include comprehensive docstrings with parameter descriptions and examples
- Handle both ID-based and name-based lookups where applicable
- Return dictionaries using `model_dump()` for consistency

### 3. Testing Requirements

- **Every new feature requires tests**: No exceptions
- Follow the existing test patterns in the corresponding test files
- Mock GraphQL responses appropriately
- Test both success and error cases
- Include edge cases and validation tests

### 4. Documentation

- Update relevant documentation files in `docs_site/docs/`
- Include clear examples in docstrings
- Document new GraphQL operations in the appropriate tools documentation

### 5. Code Style and Formatting

- **DO NOT make whitespace-only changes** - The project uses an auto-formatter that runs before commits
- Focus on functional changes; formatting will be handled automatically
- Ignore existing formatting inconsistencies - they will be resolved by the formatter

### 6. Error Handling

- Use specific exception types (inherit from `ArizeAPIException`)
- Provide clear, actionable error messages
- Handle GraphQL errors gracefully

### 7. Backward Compatibility

- Maintain backward compatibility when modifying existing methods
- Use deprecation warnings when necessary
- Add new optional parameters rather than changing existing ones

### 8. Examples for Users

- Create examples for meaningful workflows: When implementing new functionality, add examples to the examples/ folder that demonstrate real-world usage patterns
- Focus on multi-step operations: Examples should showcase common workflows that combine multiple client methods, such as:
  - Fetching all monitors for a model and then collecting their current metric values
  - Iterating through all organizations and spaces to collect models
  - Creating a dashboard and populating it with widgets for multiple models
  - Setting up monitors across a model's lifecycle stages

## Working Guidelines

1. **Before making changes**:

   - Understand the existing patterns by examining similar code
   - Check for reusable components in base classes
   - Review related tests to understand expected behavior

1. **When implementing features**:

   - Start with models, then queries, then client methods
   - Write tests alongside implementation
   - Update documentation as you go

1. **Quality checks**:

   - Ensure all tests pass
   - Verify that new code follows established patterns
   - Check that error messages are helpful

## Reference Rules

The following workspace rules provide detailed implementation patterns and best practices for specific aspects of development:

### Core Development Rules

- **`project-patterns`**: Overall project patterns and conventions
- **`model-creation`**: Guidelines for creating new Pydantic models for GraphQL types
- **`query-creation`**: Guidelines for creating new GraphQL queries and mutations
- **`type-creation`**: Guidelines for creating new enum types defined in the GraphQL query schema

### Testing Rules

- **`model-tests`**: Guidelines for testing new models added to `arize_toolkit/models/`
- **`query-tests`**: Guidelines for testing GraphQL queries and mutations
- **`client-tests`**: Guidelines for testing client methods in `arize_toolkit/client.py`

### Implementation Rules

- **`client-tools`**: Guidelines for implementing high-level client methods
- **`doc-creation`**: Guidelines for creating and updating documentation

### Example Generation Rules

- **`example-generation`**: Guidelines for when and how to create usage example notebooks in `arize_toolkit/examples/`

Refer to these specific rules when implementing features in their respective areas. Each rule file contains detailed examples, patterns, and best practices that must be followed.

## Remember

You are part of a team maintaining a production SDK. Your code will be used by data scientists and ML engineers to monitor their models. Clarity, consistency, and reliability are paramount.
