# GraphQL Model Test Rules

When creating tests for new models added to `arize_toolkit/models/*.py`, follow these guidelines to ensure consistent and comprehensive test coverage.

## Test Structure

### 1. Test Class Organization

Create a test class for each model or group of related models:

```python
class Test{ModelName}:
    """Test suite for {ModelName} model."""
    
    def test_init(self):
        """Test that {ModelName} can be initialized with valid parameters."""
        # Test implementation
    
    def test_validation(self):
        """Test model validation rules."""
        # Test implementation
    
    def test_methods(self):
        """Test any custom methods on the model."""
        # Test implementation
```

### 2. Test Method Naming

- Use descriptive method names: `test_{model_name}_{scenario}`
- Include docstrings explaining what the test covers
- Examples:
  - `test_init` - Basic initialization
  - `test_validation_required_fields` - Required field validation
  - `test_format_method` - Custom method behavior
  - `test_inheritance` - Inheritance behavior

## Core Test Categories

### 1. Initialization Tests

Test that models can be created with all valid field combinations:

```python
def test_init(self):
    """Test that ModelName can be initialized with valid parameters."""
    model = ModelName(
        required_field="value",
        optional_field="optional_value",
        enum_field=EnumType.VALUE,
        nested_model=NestedModel(field="value"),
        list_field=["item1", "item2"],
    )

    # Assert all fields are set correctly
    assert model.required_field == "value"
    assert model.optional_field == "optional_value"
    assert model.enum_field == EnumType.VALUE
    assert model.nested_model.field == "value"
    assert len(model.list_field) == 2
    assert model.list_field[0] == "item1"
```

### 2. Field Validation Tests

Test Pydantic validation rules:

```python
def test_validation_required_fields(self):
    """Test that required fields must be provided."""
    with pytest.raises(ValidationError) as exc_info:
        ModelName()  # Missing required fields

    assert "required_field" in str(exc_info.value)


def test_validation_enum_fields(self):
    """Test that enum fields only accept valid values."""
    with pytest.raises(ValidationError) as exc_info:
        ModelName(required_field="value", enum_field="invalid_enum_value")

    assert "enum_field" in str(exc_info.value)
```

### 3. Model Validator Tests

For models with `@model_validator` decorators:

```python
def test_model_validator_logic(self):
    """Test custom model validation logic."""
    # Test valid case
    valid_model = ModelWithValidator(
        field1="value1", field2="value2"  # Passes validation
    )
    assert valid_model.field1 == "value1"

    # Test invalid case
    with pytest.raises(ValueError, match="Expected error message"):
        ModelWithValidator(field1="value1", field2="invalid_value")  # Fails validation
```

### 4. Default Value Tests

Test that optional fields have correct defaults:

```python
def test_default_values(self):
    """Test that optional fields have correct default values."""
    model = ModelName(required_field="value")

    assert model.optional_str is None
    assert model.optional_list == []
    assert model.bool_with_default is False
    assert model.enum_with_default == EnumType.DEFAULT_VALUE
```

### 5. Method Tests

Test any custom methods on the model:

```python
def test_custom_method(self):
    """Test custom method behavior."""
    model = ModelWithMethod(field1="value1", field2="value2")

    result = model.custom_method(param="test")

    assert result.expected_field == "expected_value"
    assert isinstance(result, ExpectedType)
```

### 6. Serialization Tests

Test model serialization for GraphQL:

```python
def test_serialization(self):
    """Test model serialization for GraphQL."""
    model = GraphQLModel(
        field_with_alias="value", nested_field=NestedModel(data="test")
    )

    # Test to_dict/model_dump
    data = model.model_dump(by_alias=True)
    assert data["fieldWithAlias"] == "value"  # Check alias is used
    assert "field_with_alias" not in data

    # Test to_graphql_fields if applicable
    graphql_fields = model.to_graphql_fields()
    assert "fieldWithAlias" in graphql_fields
```

### 7. Complex Type Tests

For models with Union types, Lists, or complex nested structures:

```python
def test_union_field(self):
    """Test Union type field handling."""
    # Test with first type
    model1 = ModelWithUnion(union_field=TypeA(field="value"))
    assert isinstance(model1.union_field, TypeA)

    # Test with second type
    model2 = ModelWithUnion(union_field=TypeB(field="value"))
    assert isinstance(model2.union_field, TypeB)


def test_list_field_validation(self):
    """Test list field validation."""
    # Test with valid list
    model = ModelWithList(items=[Item(name="item1"), Item(name="item2")])
    assert len(model.items) == 2

    # Test with invalid item type
    with pytest.raises(ValidationError):
        ModelWithList(items=["not_an_item"])
```

## Best Practices

### 1. Test Coverage

Ensure tests cover:

- All field types (required, optional, with defaults)
- All validation rules
- All custom methods
- Edge cases (empty lists, None values, etc.)
- Error scenarios

### 2. Test Data

Use realistic test data:

```python
# Good
model = FileImportJob(
    id="job123",
    jobStatus="active",
    modelName="customer-churn-model",
    modelType=ModelType.score_categorical,
)

# Avoid
model = FileImportJob(
    id="test", jobStatus="test", modelName="test", modelType=ModelType.score_categorical
)
```

### 3. Assertions

Be specific with assertions:

```python
# Good - specific assertions
assert model.id == "job123"
assert model.jobStatus == "active"
assert isinstance(model.createdAt, datetime)

# Avoid - too general
assert model is not None
assert model.id
```

### 4. Error Testing

Always test error messages:

```python
with pytest.raises(ValueError) as exc_info:
    InvalidModel(field="bad_value")

# Check the specific error message
assert "field must be" in str(exc_info.value)
```

### 5. Fixtures

Create fixtures for complex test data:

```python
@pytest.fixture
def sample_schema():
    return ClassificationSchemaInput(
        predictionLabel="prediction",
        actualLabel="actual",
        featuresList=["feature1", "feature2"],
    )


def test_with_schema(self, sample_schema):
    model = FileImportJob(
        modelSchema=sample_schema,
        # ... other fields
    )
    assert model.modelSchema.predictionLabel == "prediction"
```

## Example Test Class

Here's a complete example for a new model:

```python
class TestTableImportJobInput:
    """Test suite for TableImportJobInput model."""

    @pytest.fixture
    def valid_bigquery_config(self):
        return BigQueryTableConfig(
            projectId="my-project", dataset="my-dataset", tableName="my-table"
        )

    @pytest.fixture
    def valid_schema(self):
        return ClassificationSchemaInput(
            predictionLabel="prediction", actualLabel="actual"
        )

    def test_init_with_bigquery(self, valid_bigquery_config, valid_schema):
        """Test initialization with BigQuery configuration."""
        job = TableImportJobInput(
            tableStore=TableStore.BigQuery,
            bigQueryTableConfig=valid_bigquery_config,
            spaceId="space123",
            modelName="my-model",
            modelType=ModelType.score_categorical,
            modelEnvironmentName=ModelEnvironment.production,
            modelSchema=valid_schema,
        )

        assert job.tableStore == TableStore.BigQuery
        assert job.bigQueryTableConfig.projectId == "my-project"
        assert job.spaceId == "space123"
        assert job.modelName == "my-model"

    def test_validation_matching_store_config(self):
        """Test that table config must match tableStore type."""
        with pytest.raises(ValueError, match="bigQueryTableConfig is required"):
            TableImportJobInput(
                tableStore=TableStore.BigQuery,
                # Missing bigQueryTableConfig
                spaceId="space123",
                modelName="my-model",
                modelType=ModelType.score_categorical,
                modelEnvironmentName=ModelEnvironment.production,
                modelSchema=ClassificationSchemaInput(
                    predictionLabel="pred", actualLabel="actual"
                ),
            )

    def test_optional_fields_defaults(self, valid_bigquery_config, valid_schema):
        """Test optional fields have correct defaults."""
        job = TableImportJobInput(
            tableStore=TableStore.BigQuery,
            bigQueryTableConfig=valid_bigquery_config,
            spaceId="space123",
            modelName="my-model",
            modelType=ModelType.score_categorical,
            modelEnvironmentName=ModelEnvironment.production,
            modelSchema=valid_schema,
        )

        assert job.modelVersion is None
        assert job.batchId is None
        assert job.dryRun is False

    def test_serialization_with_alias(self, valid_bigquery_config, valid_schema):
        """Test that model serializes with field aliases."""
        job = TableImportJobInput(
            tableStore=TableStore.BigQuery,
            bigQueryTableConfig=valid_bigquery_config,
            spaceId="space123",
            modelName="my-model",
            modelType=ModelType.score_categorical,
            modelEnvironmentName=ModelEnvironment.production,
            modelSchema=valid_schema,
        )

        data = job.model_dump(by_alias=True)
        assert "schema" in data  # Check alias is used
        assert "modelSchema" not in data
```

## Checklist for New Model Tests

When adding tests for a new model, ensure you have:

- [ ] Test class named `Test{ModelName}`
- [ ] `test_init` method for basic initialization
- [ ] Tests for all required fields
- [ ] Tests for optional fields and defaults
- [ ] Tests for field validation rules
- [ ] Tests for model validators (if any)
- [ ] Tests for custom methods (if any)
- [ ] Tests for serialization/aliases (if applicable)
- [ ] Tests for error cases
- [ ] Fixtures for complex test data
- [ ] Descriptive docstrings for all tests
- [ ] Realistic test data values
