"use strict";
const assert_1 = require("@aws-cdk/assert");
const cdk = require("@aws-cdk/core");
const codepipeline = require("../lib");
module.exports = {
    'Pipeline Stages': {
        'can be inserted before another Stage'(test) {
            const stack = new cdk.Stack();
            const pipeline = new codepipeline.Pipeline(stack, 'Pipeline');
            const secondStage = pipeline.addStage({ stageName: 'SecondStage' });
            pipeline.addStage({
                stageName: 'FirstStage',
                placement: {
                    rightBefore: secondStage,
                },
            });
            assert_1.expect(stack, true).to(assert_1.haveResourceLike('AWS::CodePipeline::Pipeline', {
                "Stages": [
                    { "Name": "FirstStage" },
                    { "Name": "SecondStage" },
                ],
            }));
            test.done();
        },
        'can be inserted after another Stage'(test) {
            const stack = new cdk.Stack();
            const pipeline = new codepipeline.Pipeline(stack, 'Pipeline');
            const firstStage = pipeline.addStage({ stageName: 'FirstStage' });
            pipeline.addStage({ stageName: 'ThirdStage' });
            pipeline.addStage({
                stageName: 'SecondStage',
                placement: {
                    justAfter: firstStage,
                },
            });
            assert_1.expect(stack, true).to(assert_1.haveResourceLike('AWS::CodePipeline::Pipeline', {
                "Stages": [
                    { "Name": "FirstStage" },
                    { "Name": "SecondStage" },
                    { "Name": "ThirdStage" },
                ],
            }));
            test.done();
        },
        "attempting to insert a Stage before a Stage that doesn't exist results in an error"(test) {
            const stack = new cdk.Stack();
            const pipeline = new codepipeline.Pipeline(stack, 'Pipeline');
            const stage = pipeline.addStage({ stageName: 'Stage' });
            const anotherPipeline = new codepipeline.Pipeline(stack, 'AnotherPipeline');
            test.throws(() => {
                anotherPipeline.addStage({
                    stageName: 'AnotherStage',
                    placement: {
                        rightBefore: stage,
                    },
                });
            }, /before/i);
            test.done();
        },
        "attempting to insert a Stage after a Stage that doesn't exist results in an error"(test) {
            const stack = new cdk.Stack();
            const pipeline = new codepipeline.Pipeline(stack, 'Pipeline');
            const stage = pipeline.addStage({ stageName: 'Stage' });
            const anotherPipeline = new codepipeline.Pipeline(stack, 'AnotherPipeline');
            test.throws(() => {
                anotherPipeline.addStage({
                    stageName: 'AnotherStage',
                    placement: {
                        justAfter: stage,
                    },
                });
            }, /after/i);
            test.done();
        },
        "providing more than one placement value results in an error"(test) {
            const stack = new cdk.Stack();
            const pipeline = new codepipeline.Pipeline(stack, 'Pipeline');
            const stage = pipeline.addStage({ stageName: 'Stage' });
            test.throws(() => {
                pipeline.addStage({
                    stageName: 'SecondStage',
                    placement: {
                        rightBefore: stage,
                        justAfter: stage,
                    },
                });
                // incredibly, an arrow function below causes nodeunit to crap out with:
                // "TypeError: Function has non-object prototype 'undefined' in instanceof check"
                // tslint:disable-next-line:only-arrow-functions
            }, function (e) {
                return /rightBefore/.test(e) && /justAfter/.test(e);
            });
            test.done();
        },
    },
};
//# sourceMappingURL=data:application/json;base64,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