"use strict";
const assert_1 = require("@aws-cdk/assert");
const iam = require("@aws-cdk/aws-iam");
const cdk = require("@aws-cdk/core");
const codepipeline = require("../lib");
const validations = require("../lib/validation");
const fake_build_action_1 = require("./fake-build-action");
const fake_source_action_1 = require("./fake-source-action");
function boundsValidationResult(numberOfArtifacts, min, max) {
    const artifacts = [];
    for (let i = 0; i < numberOfArtifacts; i++) {
        artifacts.push(new codepipeline.Artifact(`TestArtifact${i}`));
    }
    return validations.validateArtifactBounds('output', artifacts, min, max, 'testCategory', 'testProvider');
}
module.exports = {
    'artifact bounds validation': {
        'artifacts count exceed maximum'(test) {
            const result = boundsValidationResult(1, 0, 0);
            test.deepEqual(result.length, 1);
            test.ok(result[0].match(/cannot have more than 0/), 'the validation should have failed');
            test.done();
        },
        'artifacts count below minimum'(test) {
            const result = boundsValidationResult(1, 2, 2);
            test.deepEqual(result.length, 1);
            test.ok(result[0].match(/must have at least 2/), 'the validation should have failed');
            test.done();
        },
        'artifacts count within bounds'(test) {
            const result = boundsValidationResult(1, 0, 2);
            test.deepEqual(result.length, 0);
            test.done();
        },
    },
    'action type validation': {
        'must be source and is source'(test) {
            const result = validations.validateSourceAction(true, codepipeline.ActionCategory.SOURCE, 'test action', 'test stage');
            test.deepEqual(result.length, 0);
            test.done();
        },
        'must be source and is not source'(test) {
            const result = validations.validateSourceAction(true, codepipeline.ActionCategory.DEPLOY, 'test action', 'test stage');
            test.deepEqual(result.length, 1);
            test.ok(result[0].match(/may only contain Source actions/), 'the validation should have failed');
            test.done();
        },
        'cannot be source and is source'(test) {
            const result = validations.validateSourceAction(false, codepipeline.ActionCategory.SOURCE, 'test action', 'test stage');
            test.deepEqual(result.length, 1);
            test.ok(result[0].match(/may only occur in first stage/), 'the validation should have failed');
            test.done();
        },
        'cannot be source and is not source'(test) {
            const result = validations.validateSourceAction(false, codepipeline.ActionCategory.DEPLOY, 'test action', 'test stage');
            test.deepEqual(result.length, 0);
            test.done();
        },
    },
    'action name validation': {
        'throws an exception when adding an Action with an empty name to the Pipeline'(test) {
            const stack = new cdk.Stack();
            const action = new fake_source_action_1.FakeSourceAction({
                actionName: '',
                output: new codepipeline.Artifact(),
            });
            const pipeline = new codepipeline.Pipeline(stack, 'Pipeline');
            const stage = pipeline.addStage({ stageName: 'Source' });
            test.throws(() => {
                stage.addAction(action);
            }, /Action name must match regular expression:/);
            test.done();
        },
    },
    'action Artifacts validation': {
        'validates that input Artifacts are within bounds'(test) {
            const stack = new cdk.Stack();
            const sourceOutput = new codepipeline.Artifact();
            const extraOutput1 = new codepipeline.Artifact('A1');
            const extraOutput2 = new codepipeline.Artifact('A2');
            const extraOutput3 = new codepipeline.Artifact('A3');
            new codepipeline.Pipeline(stack, 'Pipeline', {
                stages: [
                    {
                        stageName: 'Source',
                        actions: [
                            new fake_source_action_1.FakeSourceAction({
                                actionName: 'Source',
                                output: sourceOutput,
                                extraOutputs: [
                                    extraOutput1,
                                    extraOutput2,
                                    extraOutput3,
                                ],
                            }),
                        ],
                    },
                    {
                        stageName: 'Build',
                        actions: [
                            new fake_build_action_1.FakeBuildAction({
                                actionName: 'Build',
                                input: sourceOutput,
                                extraInputs: [
                                    extraOutput1,
                                    extraOutput2,
                                    extraOutput3,
                                ],
                            }),
                        ],
                    },
                ],
            });
            test.throws(() => {
                assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CodePipeline::Pipeline', {}));
            }, /Build\/Fake cannot have more than 3 input artifacts/);
            test.done();
        },
        'validates that output Artifacts are within bounds'(test) {
            const stack = new cdk.Stack();
            const sourceOutput = new codepipeline.Artifact();
            const extraOutput1 = new codepipeline.Artifact('A1');
            const extraOutput2 = new codepipeline.Artifact('A2');
            const extraOutput3 = new codepipeline.Artifact('A3');
            const extraOutput4 = new codepipeline.Artifact('A4');
            new codepipeline.Pipeline(stack, 'Pipeline', {
                stages: [
                    {
                        stageName: 'Source',
                        actions: [
                            new fake_source_action_1.FakeSourceAction({
                                actionName: 'Source',
                                output: sourceOutput,
                                extraOutputs: [
                                    extraOutput1,
                                    extraOutput2,
                                    extraOutput3,
                                    extraOutput4,
                                ],
                            }),
                        ],
                    },
                    {
                        stageName: 'Build',
                        actions: [
                            new fake_build_action_1.FakeBuildAction({
                                actionName: 'Build',
                                input: sourceOutput,
                            }),
                        ],
                    },
                ],
            });
            test.throws(() => {
                assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CodePipeline::Pipeline', {}));
            }, /Source\/Fake cannot have more than 4 output artifacts/);
            test.done();
        },
    },
    'automatically assigns artifact names to the Actions'(test) {
        const stack = new cdk.Stack();
        const pipeline = new codepipeline.Pipeline(stack, 'pipeline');
        const sourceOutput = new codepipeline.Artifact();
        const sourceAction = new fake_source_action_1.FakeSourceAction({
            actionName: 'CodeCommit',
            output: sourceOutput,
        });
        pipeline.addStage({
            stageName: 'Source',
            actions: [sourceAction],
        });
        pipeline.addStage({
            stageName: 'Build',
            actions: [
                new fake_build_action_1.FakeBuildAction({
                    actionName: 'CodeBuild',
                    input: sourceOutput,
                    output: new codepipeline.Artifact(),
                }),
            ],
        });
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CodePipeline::Pipeline', {
            "Stages": [
                {
                    "Name": "Source",
                    "Actions": [
                        {
                            "Name": "CodeCommit",
                            "InputArtifacts": [],
                            "OutputArtifacts": [
                                {
                                    "Name": "Artifact_Source_CodeCommit",
                                },
                            ],
                        }
                    ],
                },
                {
                    "Name": "Build",
                    "Actions": [
                        {
                            "Name": "CodeBuild",
                            "InputArtifacts": [
                                {
                                    "Name": "Artifact_Source_CodeCommit",
                                }
                            ],
                            "OutputArtifacts": [
                                {
                                    "Name": "Artifact_Build_CodeBuild",
                                },
                            ],
                        }
                    ],
                },
            ],
        }));
        test.done();
    },
    'the same Action can be safely added to 2 different Stages'(test) {
        const stack = new cdk.Stack();
        const sourceOutput = new codepipeline.Artifact();
        const pipeline = new codepipeline.Pipeline(stack, 'Pipeline', {
            stages: [
                {
                    stageName: 'Source',
                    actions: [
                        new fake_source_action_1.FakeSourceAction({
                            actionName: 'Source',
                            output: sourceOutput,
                        }),
                    ],
                },
            ],
        });
        const action = new fake_build_action_1.FakeBuildAction({ actionName: 'FakeAction', input: sourceOutput });
        const stage2 = {
            stageName: 'Stage2',
            actions: [action],
        };
        const stage3 = {
            stageName: 'Stage3',
            actions: [action],
        };
        pipeline.addStage(stage2);
        test.doesNotThrow(() => {
            pipeline.addStage(stage3);
        }, /FakeAction/);
        test.done();
    },
    'input Artifacts': {
        'can be added multiple times to an Action safely'(test) {
            const artifact = new codepipeline.Artifact('SomeArtifact');
            test.doesNotThrow(() => {
                new fake_build_action_1.FakeBuildAction({
                    actionName: 'CodeBuild',
                    input: artifact,
                    extraInputs: [artifact],
                });
            });
            test.done();
        },
        'can have duplicate names'(test) {
            const artifact1 = new codepipeline.Artifact('SomeArtifact');
            const artifact2 = new codepipeline.Artifact('SomeArtifact');
            test.doesNotThrow(() => {
                new fake_build_action_1.FakeBuildAction({
                    actionName: 'CodeBuild',
                    input: artifact1,
                    extraInputs: [artifact2],
                });
            });
            test.done();
        },
    },
    'output Artifacts': {
        'accept multiple Artifacts with the same name safely'(test) {
            test.doesNotThrow(() => {
                new fake_source_action_1.FakeSourceAction({
                    actionName: 'CodeBuild',
                    output: new codepipeline.Artifact('Artifact1'),
                    extraOutputs: [
                        new codepipeline.Artifact('Artifact1'),
                        new codepipeline.Artifact('Artifact1'),
                    ],
                });
            });
            test.done();
        },
    },
    'an Action with a non-AWS owner cannot have a Role passed for it'(test) {
        const stack = new cdk.Stack();
        const sourceOutput = new codepipeline.Artifact();
        const pipeline = new codepipeline.Pipeline(stack, 'Pipeline', {
            stages: [
                {
                    stageName: 'Source',
                    actions: [
                        new fake_source_action_1.FakeSourceAction({
                            actionName: 'source',
                            output: sourceOutput,
                        }),
                    ],
                },
            ],
        });
        const buildStage = pipeline.addStage({ stageName: 'Build' });
        // constructing it is fine
        const buildAction = new fake_build_action_1.FakeBuildAction({
            actionName: 'build',
            input: sourceOutput,
            owner: 'ThirdParty',
            role: new iam.Role(stack, 'Role', {
                assumedBy: new iam.AnyPrincipal(),
            }),
        });
        // an attempt to add it to the Pipeline is where things blow up
        test.throws(() => {
            buildStage.addAction(buildAction);
        }, /Role is not supported for actions with an owner different than 'AWS' - got 'ThirdParty' \(Action: 'build' in Stage: 'Build'\)/);
        test.done();
    },
};
//# sourceMappingURL=data:application/json;base64,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