"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const events = require("@aws-cdk/aws-events");
const cdk = require("@aws-cdk/core");
const validation = require("./validation");
/**
 * A Stage in a Pipeline.
 *
 * Stages are added to a Pipeline by calling {@link Pipeline#addStage},
 * which returns an instance of {@link codepipeline.IStage}.
 *
 * This class is private to the CodePipeline module.
 */
class Stage {
    /**
     * Create a new Stage.
     */
    constructor(props, pipeline) {
        this._actions = new Array();
        validation.validateName('Stage', props.stageName);
        this.stageName = props.stageName;
        this._pipeline = pipeline;
        this.scope = new cdk.Construct(pipeline, this.stageName);
        for (const action of props.actions || []) {
            this.addAction(action);
        }
    }
    /**
     * Get a duplicate of this stage's list of actions.
     */
    get actions() {
        return this._actions.slice();
    }
    get pipeline() {
        return this._pipeline;
    }
    render() {
        // first, assign names to output Artifacts who don't have one
        for (const action of this.actions) {
            const outputArtifacts = action.outputs;
            const unnamedOutputs = outputArtifacts.filter(o => !o.artifactName);
            for (const outputArtifact of outputArtifacts) {
                if (!outputArtifact.artifactName) {
                    const artifactName = `Artifact_${this.stageName}_${action.actionName}` + (unnamedOutputs.length === 1
                        ? ''
                        : '_' + (unnamedOutputs.indexOf(outputArtifact) + 1));
                    outputArtifact._setName(artifactName);
                }
            }
        }
        return {
            name: this.stageName,
            actions: this._actions.map(action => this.renderAction(action)),
        };
    }
    addAction(action) {
        const actionName = action.actionProperties.actionName;
        // validate the name
        validation.validateName('Action', actionName);
        // check for duplicate Actions and names
        if (this._actions.find(a => a.actionName === actionName)) {
            throw new Error(`Stage ${this.stageName} already contains an action with name '${actionName}'`);
        }
        this._actions.push(this.attachActionToPipeline(action));
    }
    onStateChange(name, target, options) {
        const rule = new events.Rule(this.scope, name, options);
        rule.addTarget(target);
        rule.addEventPattern({
            detailType: ['CodePipeline Stage Execution State Change'],
            source: ['aws.codepipeline'],
            resources: [this.pipeline.pipelineArn],
            detail: {
                stage: [this.stageName],
            },
        });
        return rule;
    }
    validate() {
        return [
            ...this.validateHasActions(),
            ...this.validateActions(),
        ];
    }
    validateHasActions() {
        if (this._actions.length === 0) {
            return [`Stage '${this.stageName}' must have at least one action`];
        }
        return [];
    }
    validateActions() {
        const ret = new Array();
        for (const action of this.actions) {
            ret.push(...this.validateAction(action));
        }
        return ret;
    }
    validateAction(action) {
        return validation.validateArtifactBounds('input', action.inputs, action.artifactBounds.minInputs, action.artifactBounds.maxInputs, action.category, action.provider)
            .concat(validation.validateArtifactBounds('output', action.outputs, action.artifactBounds.minOutputs, action.artifactBounds.maxOutputs, action.category, action.provider));
    }
    attachActionToPipeline(action) {
        // notify the Pipeline of the new Action
        const actionScope = new cdk.Construct(this.scope, action.actionProperties.actionName);
        return this._pipeline._attachActionToPipeline(this, action, actionScope);
    }
    renderAction(action) {
        const outputArtifacts = this.renderArtifacts(action.outputs);
        const inputArtifacts = this.renderArtifacts(action.inputs);
        return {
            name: action.actionName,
            inputArtifacts,
            outputArtifacts,
            actionTypeId: {
                category: action.category.toString(),
                version: action.version,
                owner: action.owner,
                provider: action.provider,
            },
            configuration: action.configuration,
            runOrder: action.runOrder,
            roleArn: action.role ? action.role.roleArn : undefined,
            region: action.region,
        };
    }
    renderArtifacts(artifacts) {
        return artifacts
            .filter(a => a.artifactName)
            .map(a => ({ name: a.artifactName }));
    }
}
exports.Stage = Stage;
//# sourceMappingURL=data:application/json;base64,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