import s3 = require("@aws-cdk/aws-s3");
/**
 * An output artifact of an action. Artifacts can be used as input by some actions.
 */
export declare class Artifact {
    /**
     * A static factory method used to create instances of the Artifact class.
     * Mainly meant to be used from `decdk`.
     *
     * @param name the (required) name of the Artifact
     */
    static artifact(name: string): Artifact;
    private _artifactName?;
    constructor(artifactName?: string);
    readonly artifactName: string | undefined;
    /**
     * Returns an ArtifactPath for a file within this artifact.
     * CfnOutput is in the form "<artifact-name>::<file-name>"
     * @param fileName The name of the file
     */
    atPath(fileName: string): ArtifactPath;
    /**
     * The artifact attribute for the name of the S3 bucket where the artifact is stored.
     */
    readonly bucketName: string;
    /**
     * The artifact attribute for The name of the .zip file that contains the artifact that is
     * generated by AWS CodePipeline, such as 1ABCyZZ.zip.
     */
    readonly objectKey: string;
    /**
     * The artifact attribute of the Amazon Simple Storage Service (Amazon S3) URL of the artifact,
     * such as https://s3-us-west-2.amazonaws.com/artifactstorebucket-yivczw8jma0c/test/TemplateSo/1ABCyZZ.zip.
     */
    readonly url: string;
    /**
     * Returns a token for a value inside a JSON file within this artifact.
     * @param jsonFile The JSON file name.
     * @param keyName The hash key.
     */
    getParam(jsonFile: string, keyName: string): string;
    /**
     * Returns the location of the .zip file in S3 that this Artifact represents.
     * Used by Lambda's `CfnParametersCode` when being deployed in a CodePipeline.
     */
    readonly s3Location: s3.Location;
    toString(): string | undefined;
}
/**
 * A specific file within an output artifact.
 *
 * The most common use case for this is specifying the template file
 * for a CloudFormation action.
 */
export declare class ArtifactPath {
    readonly artifact: Artifact;
    readonly fileName: string;
    static artifactPath(artifactName: string, fileName: string): ArtifactPath;
    constructor(artifact: Artifact, fileName: string);
    readonly location: string;
}
