# -*- coding: utf-8 -*-
from setuptools import setup

package_dir = \
{'': 'src'}

packages = \
['sequences', 'sequences.migrations']

package_data = \
{'': ['*'],
 'sequences': ['locale/bg/LC_MESSAGES/*',
               'locale/cs/LC_MESSAGES/*',
               'locale/da/LC_MESSAGES/*',
               'locale/de/LC_MESSAGES/*',
               'locale/es/LC_MESSAGES/*',
               'locale/fr/LC_MESSAGES/*',
               'locale/it/LC_MESSAGES/*',
               'locale/ja/LC_MESSAGES/*',
               'locale/lt/LC_MESSAGES/*',
               'locale/nl/LC_MESSAGES/*',
               'locale/pt/LC_MESSAGES/*',
               'locale/rm/LC_MESSAGES/*',
               'locale/ru/LC_MESSAGES/*',
               'locale/sk/LC_MESSAGES/*',
               'locale/sv/LC_MESSAGES/*',
               'locale/th/LC_MESSAGES/*',
               'locale/tr/LC_MESSAGES/*']}

install_requires = \
['django>=3.2']

setup_kwargs = {
    'name': 'django-sequences',
    'version': '2.7',
    'description': 'Generate gapless sequences of integer values.',
    'long_description': 'django-sequences\n################\n\nBy default, Django gives each model an auto-incrementing integer primary key.\nThese primary keys look like they generate a continuous sequence of integers.\n\nHowever, this behavior isn\'t guaranteed.\n\nIf a transaction inserts a row and then is rolled back, the sequence counter\nisn\'t rolled back for performance reasons, creating a gap in primary keys.\n\nSuch gaps may happen on all databases natively supported by Django:\n\n* `PostgreSQL <https://www.postgresql.org/docs/current/datatype-numeric.html#DATATYPE-SERIAL>`_\n* `MariaDB <https://mariadb.com/kb/en/auto_increment/#missing-values>`_ / MySQL\n* `Oracle <https://docs.oracle.com/en/database/oracle/oracle-database/21/sqlrf/CREATE-SEQUENCE.html>`_\n* `SQLite <https://sqlite.org/autoinc.html#the_autoincrement_keyword>`_\n\nThey may also happen on most databases supported via third-party backends.\n\nThis can cause compliance issues for some use cases such as accounting.\n\nThis risk isn\'t well known. Since most transactions succeed, values look\nsequential. Gaps will only be revealed by audits.\n\ndjango-sequences solves this problem with a ``get_next_value`` function\ndesigned to be used as follows:\n\n.. code:: python\n\n    from django.db import transaction\n    from sequences import get_next_value\n    from invoices.models import Invoice\n\n    with transaction.atomic():\n        Invoice.objects.create(number=get_next_value("invoice_numbers"))\n\nOr, if you\'d rather use an object-oriented API:\n\n.. code:: python\n\n    from django.db import transaction\n    from sequences import Sequence\n    from invoices.models import Invoice\n\n    invoice_numbers = Sequence("invoice_numbers")\n\n    with transaction.atomic():\n        Invoice.objects.create(number=next(invoice_numbers))\n\n``get_next_value`` relies on the database\'s transactional integrity to ensure\nthat each value is returned exactly once. As a consequence, **the guarantees\nof django-sequences apply only if you call** ``get_next_value`` **and save its\nreturn value to the database within the same transaction!**\n\nTable of contents\n=================\n\n* `Getting started`_\n* `Database support`_\n* `Multiple databases`_\n* `Isolation levels`_\n* `Contributing`_\n* `Releasing`_\n* `Changelog`_\n\nGetting started\n===============\n\ndjango-sequences is tested with Django 3.2 (LTS) and 4.0.\nIt is also tested with all database backends built-in to Django: MySQL/MariaDB,\nOracle, PostgreSQL and SQLite.\n\nIt is released under the BSD license, like Django itself.\n\nInstall django-sequences:\n\n.. code:: shell-session\n\n    $ pip install django-sequences\n\nAdd it to the list of applications in your project\'s settings:\n\n.. code:: python\n\n    INSTALLED_APPS = [\n        ...,\n        "sequences.apps.SequencesConfig",\n        ...\n    ]\n\nRun migrations:\n\n.. code:: shell-session\n\n    $ django-admin migrate\n\nAPI\n===\n\n``get_next_value``\n------------------\n\n.. code:: pycon\n\n    >>> from sequences import get_next_value\n\nThis function generates a gapless sequence of integer values:\n\n.. code:: pycon\n\n    >>> get_next_value()\n    1\n    >>> get_next_value()\n    2\n    >>> get_next_value()\n    3\n\nIt supports multiple independent sequences:\n\n.. code:: pycon\n\n    >>> get_next_value("cases")\n    1\n    >>> get_next_value("cases")\n    2\n    >>> get_next_value("invoices")\n    1\n    >>> get_next_value("invoices")\n    2\n\nThe first value defaults to 1. It can be customized:\n\n.. code:: pycon\n\n    >>> get_next_value("customers", initial_value=1000)  # pro growth hacking\n\nThe ``initial_value`` parameter only matters when ``get_next_value`` is called\nfor the first time for a given sequence — assuming the corresponding database\ntransaction gets committed; as discussed above, if the transaction is rolled\nback, the generated value isn\'t consumed. It\'s also possible to initialize a\nsequence in a data migration and not use ``initial_value`` in actual code.\n\nSequences can loop:\n\n.. code:: pycon\n\n    >>> get_next_value("seconds", initial_value=0, reset_value=60)\n\nWhen the sequence reaches ``reset_value``, it restarts at ``initial_value``.\nIn other words, it generates ``reset_value - 2``, ``reset_value - 1``,\n``initial_value``, ``initial_value + 1``, etc. In that case, each call to\n``get_next_value`` must provide ``initial_value`` when it isn\'t the default\nand ``reset_value``.\n\n**Database transactions that call** ``get_next_value`` **for a given sequence\nare serialized.** As a consequence, when you call ``get_next_value`` in a\ndatabase transaction, other callers trying to get a value from the same\nsequence block until the transaction completes, either with a commit or a\nrollback. You should keep such transactions short to minimize the impact on\nperformance.\n\nThis is why databases default to a faster behavior that may create gaps.\n\nPassing ``nowait=True`` makes ``get_next_value`` raise an exception instead of\nblocking in this scenario. This is rarely useful. Also it doesn\'t work for the\nfirst call. (This is a bug but it\'s harmless and hard to fix.)\n\nCalls to ``get_next_value`` for distinct sequences don\'t interact with one\nanother.\n\nFinally, passing ``using="..."`` allows selecting the database on which the\ncurrent sequence value is stored. When this parameter isn\'t provided, it\ndefaults to the default database for writing models of the ``sequences``\napplication. See `Multiple databases`_ for details.\n\nTo sum up, the complete signature of ``get_next_value`` is:\n\n.. code:: python\n\n    get_next_value(\n        sequence_name="default",\n        initial_value=1,\n        reset_value=None,\n        *,\n        nowait=False,\n        using=None,\n    )\n\n``get_last_value``\n------------------\n\n.. code:: pycon\n\n    >>> from sequences import get_last_value\n\nThis function returns the last value generated by a sequence:\n\n.. code:: pycon\n\n    >>> get_last_value()\n    None\n    >>> get_next_value()\n    1\n    >>> get_last_value()\n    1\n    >>> get_next_value()\n    2\n    >>> get_last_value()\n    2\n\nIf the sequence hasn\'t generated a value yet, ``get_last_value`` returns\n``None``.\n\nIt supports independent sequences like ``get_next_value``:\n\n.. code:: pycon\n\n    >>> get_next_value("cases")\n    1\n    >>> get_last_value("cases")\n    1\n    >>> get_next_value("invoices")\n    1\n    >>> get_last_value("invoices")\n    1\n\nIt accepts ``using="..."`` for selecting the database on which the current\nsequence value is stored, defaulting to the default database for reading\nmodels of the ``sequences`` application.\n\nThe complete signature of ``get_last_value`` is:\n\n.. code:: python\n\n    get_last_value(\n        sequence_name="default",\n        *,\n        using=None,\n    )\n\n``get_last_value`` **is a convenient and fast way to tell how many values a\nsequence generated but it makes no guarantees.** Concurrent calls to\n``get_next_value`` may produce unexpected results of ``get_last_value``.\n\n``Sequence``\n------------\n\n.. code:: pycon\n\n    >>> from sequences import Sequence\n\n(not to be confused with ``sequences.models.Sequence``, a private API)\n\nThis class stores parameters for a sequence and provides ``get_next_value``\nand ``get_last_value`` methods:\n\n.. code:: pycon\n\n    >>> claim_ids = Sequence("claims")\n    >>> claim_ids.get_next_value()\n    1\n    >>> claim_ids.get_next_value()\n    2\n    >>> claim_ids.get_last_value()\n    2\n\nThis reduces the risk of errors when the same sequence is used in multiple\nplaces.\n\nInstances of ``Sequence`` are also infinite iterators:\n\n.. code:: pycon\n\n    >>> next(claim_ids)\n    3\n    >>> next(claim_ids)\n    4\n\nThe complete API is:\n\n.. code:: python\n\n    Sequence(\n        sequence_name="default",\n        initial_value=1,\n        reset_value=None,\n        *,\n        using=None,\n    )\n\n    Sequence.get_next_value(\n        self,\n        *,\n        nowait=False,\n    )\n\n    Sequence.get_last_value(\n        self,\n    )\n\nAll parameters have the same meaning as in the ``get_next_value`` and\n``get_last_value`` functions.\n\nDatabase support\n================\n\ndjango-sequences is tested on PostgreSQL, MariaDB / MySQL, Oracle, and SQLite.\n\nMySQL only supports the ``nowait`` parameter from version 8.0.1.\nMariaDB only supports ``nowait`` from version 10.3.\n\nMultiple databases\n==================\n\nSince django-sequences relies on the database to guarantee transactional\nintegrity, the current value for a given sequence must be stored in the same\ndatabase as models containing generated values.\n\nIn a project that uses multiple databases, you must write a suitable database\nrouter to create tables for the ``sequences`` application on all databases\nstoring models containing sequential numbers.\n\nEach database has its own namespace: a sequence with the same name stored in\ntwo databases will have independent counters in each database.\n\nIsolation levels\n================\n\nSince django-sequences relies on the database\'s transactional integrity, using\na non-default transaction isolation level requires special care.\n\n* **read uncommitted:** django-sequences cannot work at this isolation level.\n\n  Indeed, concurrent transactions can create gaps, as in this scenario:\n\n  * Transaction A reads N and writes N + 1;\n  * Transaction B reads N + 1 (dirty read) and writes N + 2;\n  * Transaction A is rolled back;\n  * Transaction B is committed;\n  * N + 1 is a gap.\n\n  The read uncommitted isolation level doesn\'t provide sufficient guarantees.\n  It will never be supported.\n\n* **read committed:** django-sequences works best at this isolation level,\n  like Django itself.\n\n* **repeatable read:** django-sequences also works at this isolation level,\n  provided your code handles serialization failures and retries transactions.\n\n  This requirement isn\'t specific to django-sequences. It\'s generally needed\n  when running at the repeatable read isolation level.\n\n  Here\'s a scenario where only one of two concurrent transactions can\n  complete on PostgreSQL:\n\n  * Transaction A reads N and writes N + 1;\n  * Transaction B attemps to read; it must wait until transaction A completes;\n  * Transaction A is committed;\n  * Transaction B is aborted.\n\n  On PostgreSQL, serialization failures are reported as: ``OperationalError:\n  could not serialize access due to concurrent update``.\n\n  On MySQL, they result in: ``OperationalError: (1213, \'Deadlock found when\n  trying to get lock; try restarting transaction\')``.\n\n  Concurrent transactions initializing the same sequence are also vulnerable,\n  although that\'s hardly ever a problem in practice.\n\n  On PostgreSQL, this manifests as ``IntegrityError: duplicate key value\n  violates unique constraint "sequences_sequence_pkey"``.\n\n* **serializable:** the situation is identical to the repeatable read level.\n\n  SQLite always runs at the serializable isolation level. Serialization\n  failures result in: ``OperationalError: database is locked``.\n\nContributing\n============\n\nPrepare a development environment:\n\n* Install Poetry_.\n* Run ``poetry install``.\n* Run ``poetry shell`` to load the development environment.\n\nPrepare testing databases:\n\n* Install PostgreSQL, MariaDB, and Oracle.\n* Create a database called ``sequences``, owned by a user called ``sequences``\n  with password ``sequences``, with permissions to create a ``test_sequences``\n  test database. You may override these values with environment variables; see\n  ``tests/*_settings.py`` for details.\n\nMake changes:\n\n* Make changes to the code, tests, or docs.\n* Run ``make style`` and fix any flake8 violations.\n* Run ``make test`` to run the set suite on all databases.\n\nIterate until you\'re happy.\n\nCheck quality and submit your changes:\n\n* Install tox_.\n* Run ``tox`` to test on all Python and Django versions and all databases.\n* Submit a pull request.\n\n.. _Poetry: https://python-poetry.org/\n.. _tox: https://tox.readthedocs.io/\n\nReleasing\n=========\n\nIncrement version number X.Y in ``pyproject.toml``.\n\nCommit, tag, and push the change:\n\n.. code:: shell-session\n\n    $ git commit -m "Bump version number".\n    $ git tag X.Y\n    $ git push\n    $ git push --tags\n\nBuild and publish the new version:\n\n.. code:: shell-session\n\n    $ poetry build\n    $ poetry publish\n\nChangelog\n=========\n\n2.7\n---\n\n* Sequence values can go up to ``2 ** 63 - 1`` instead of ``2 ** 31 - 1``\n  previously. The exact limit depends on the database backend.\n\n  Migration ``0002_alter_sequence_last.py`` changes the field storing sequence\n  values from ``PositiveIntegerField`` to ``PositiveBigIntegerField``. Running\n  it requires an exclusive lock on the table, which prevents other operations,\n  including reads.\n\n  If you have many distinct sequences, e.g. if you create one sequence per user\n  and you have millions of users, review how the migration will affect your app\n  before running it or skip it with ``migrate --fake``.\n\n2.6\n---\n\n* Improve documentation.\n\n2.5\n---\n\n* Fix Japanese and Turkish translations.\n* Restore compatibility with Python 3.5.\n* Support relabeling the ``sequences`` app with a custom ``AppConfig``.\n\n2.4\n---\n\n* Add the ``get_last_value`` function.\n* Add the ``Sequence`` class.\n\n2.3\n---\n\n* Optimize performance on MySQL.\n* Test on MySQL, SQLite and Oracle.\n\n2.2\n---\n\n* Optimize performance on PostgreSQL ≥ 9.5.\n\n2.1\n---\n\n* Provide looping sequences with ``reset_value``.\n\n2.0\n---\n\n* Add support for multiple databases.\n* Add translations.\n* ``nowait`` becomes keyword-only argument.\n* Drop support for Python 2.\n\n1.0\n---\n\n* Initial stable release.\n',
    'author': 'Aymeric Augustin',
    'author_email': 'aymeric.augustin@m4x.org',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/aaugustin/django-sequences',
    'package_dir': package_dir,
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.6.2',
}


setup(**setup_kwargs)
