"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.StaticWebsiteOrigin = exports.StaticWebsite = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/*! Copyright [Amazon.com](http://amazon.com/), Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0 */
const pdk_nag_1 = require("@aws-prototyping-sdk/pdk-nag");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_cloudfront_1 = require("aws-cdk-lib/aws-cloudfront");
const aws_cloudfront_origins_1 = require("aws-cdk-lib/aws-cloudfront-origins");
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
const aws_s3_1 = require("aws-cdk-lib/aws-s3");
const aws_s3_deployment_1 = require("aws-cdk-lib/aws-s3-deployment");
const cdk_nag_1 = require("cdk-nag");
const constructs_1 = require("constructs");
const cloudfront_web_acl_1 = require("./cloudfront-web-acl");
const DEFAULT_RUNTIME_CONFIG_FILENAME = "runtime-config.json";
/**
 * Deploys a Static Website using by default a private S3 bucket as an origin and Cloudfront as the entrypoint.
 *
 * This construct configures a webAcl containing rules that are generally applicable to web applications. This
 * provides protection against exploitation of a wide range of vulnerabilities, including some of the high risk
 * and commonly occurring vulnerabilities described in OWASP publications such as OWASP Top 10.
 *
 */
class StaticWebsite extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.validateProps = (props) => {
            this.validateEncryptionSettings(props);
            props.runtimeOptions && this.validateRuntimeConfig(props.runtimeOptions);
            props.websiteBucket && this.validateBucketConfig(props.websiteBucket);
        };
        this.validateRuntimeConfig = (config) => {
            if (!config) {
                throw new Error("validateRuntimeConfig only accepts non-null RuntimeOptions.");
            }
            if (config.jsonFileName && !config.jsonFileName.endsWith(".json")) {
                throw new Error("RuntimeOptions.jsonFileName must be a json file.");
            }
        };
        this.validateBucketConfig = (bucket) => {
            if (bucket.isWebsite) {
                throw new Error("Website buckets cannot be configured as websites as this will break Cloudfront hosting!");
            }
        };
        this.validateEncryptionSettings = ({ defaultWebsiteBucketEncryption, defaultWebsiteBucketEncryptionKey, }) => {
            if (defaultWebsiteBucketEncryptionKey &&
                defaultWebsiteBucketEncryption !== aws_s3_1.BucketEncryption.KMS) {
                throw new Error("Bucket encryption should be set to KMS if providing a defaultWebsiteBucketEncryptionKey.");
            }
            if (defaultWebsiteBucketEncryption &&
                defaultWebsiteBucketEncryption !== aws_s3_1.BucketEncryption.KMS &&
                defaultWebsiteBucketEncryption !== aws_s3_1.BucketEncryption.S3_MANAGED) {
                throw new Error("Only KMS and S3_MANAGED encryption are supported on the default bucket.");
            }
        };
        this.suppressCDKNagViolations = (props) => {
            const stack = aws_cdk_lib_1.Stack.of(this);
            !props.distributionProps?.certificate &&
                [
                    "AwsSolutions-CFR4",
                    "AwsPrototyping-CloudFrontDistributionHttpsViewerNoOutdatedSSL",
                ].forEach((RuleId) => {
                    cdk_nag_1.NagSuppressions.addResourceSuppressions(this.cloudFrontDistribution, [
                        {
                            id: RuleId,
                            reason: "Certificate is not mandatory therefore the Cloudfront certificate will be used.",
                        },
                    ]);
                });
            ["AwsSolutions-L1", "AwsPrototyping-LambdaLatestVersion"].forEach((RuleId) => {
                cdk_nag_1.NagSuppressions.addStackSuppressions(stack, [
                    {
                        id: RuleId,
                        reason: "Latest runtime cannot be configured. CDK will need to upgrade the BucketDeployment construct accordingly.",
                    },
                ]);
            });
            ["AwsSolutions-IAM5", "AwsPrototyping-IAMNoWildcardPermissions"].forEach((RuleId) => {
                cdk_nag_1.NagSuppressions.addStackSuppressions(stack, [
                    {
                        id: RuleId,
                        reason: "All Policies have been scoped to a Bucket. Given Buckets can contain arbitrary content, wildcard resources with bucket scope are required.",
                        appliesTo: [
                            {
                                regex: "/^Action::s3:.*$/g",
                            },
                            {
                                regex: `/^Resource::.*$/g`,
                            },
                        ],
                    },
                ]);
            });
            ["AwsSolutions-IAM4", "AwsPrototyping-IAMNoManagedPolicies"].forEach((RuleId) => {
                cdk_nag_1.NagSuppressions.addStackSuppressions(stack, [
                    {
                        id: RuleId,
                        reason: "Buckets can contain arbitrary content, therefore wildcard resources under a bucket are required.",
                        appliesTo: [
                            {
                                regex: `/^Policy::arn:${pdk_nag_1.PDKNag.getStackPartitionRegex(stack)}:iam::aws:policy/service-role/AWSLambdaBasicExecutionRole$/g`,
                            },
                        ],
                    },
                ]);
            });
            ["AwsSolutions-S1", "AwsPrototyping-S3BucketLoggingEnabled"].forEach((RuleId) => {
                cdk_nag_1.NagSuppressions.addStackSuppressions(stack, [
                    {
                        id: RuleId,
                        reason: "Access Log buckets should not have s3 bucket logging",
                    },
                ]);
            });
        };
        this.validateProps(props);
        const accessLogsBucket = new aws_s3_1.Bucket(this, "AccessLogsBucket", {
            versioned: false,
            enforceSSL: true,
            autoDeleteObjects: true,
            removalPolicy: aws_cdk_lib_1.RemovalPolicy.DESTROY,
            encryption: aws_s3_1.BucketEncryption.S3_MANAGED,
            publicReadAccess: false,
            blockPublicAccess: aws_s3_1.BlockPublicAccess.BLOCK_ALL,
        });
        // S3 Bucket to hold website files
        this.websiteBucket =
            props.websiteBucket ??
                new aws_s3_1.Bucket(this, "WebsiteBucket", {
                    versioned: true,
                    enforceSSL: true,
                    autoDeleteObjects: true,
                    removalPolicy: aws_cdk_lib_1.RemovalPolicy.DESTROY,
                    encryption: props.defaultWebsiteBucketEncryption ?? aws_s3_1.BucketEncryption.S3_MANAGED,
                    encryptionKey: props.defaultWebsiteBucketEncryptionKey,
                    publicReadAccess: false,
                    blockPublicAccess: aws_s3_1.BlockPublicAccess.BLOCK_ALL,
                    serverAccessLogsPrefix: "website-access-logs",
                    serverAccessLogsBucket: accessLogsBucket,
                });
        // Web ACL
        const { distributionProps } = props;
        const webAclArn = distributionProps?.webAclId ??
            new cloudfront_web_acl_1.CloudfrontWebAcl(this, "WebsiteAcl", props.webAclProps).webAclArn;
        // Cloudfront Distribution
        const logBucket = props.distributionProps?.logBucket ||
            new aws_s3_1.Bucket(this, "DistributionLogBucket", {
                enforceSSL: true,
                autoDeleteObjects: true,
                removalPolicy: aws_cdk_lib_1.RemovalPolicy.DESTROY,
                encryption: props.defaultWebsiteBucketEncryption ?? aws_s3_1.BucketEncryption.S3_MANAGED,
                encryptionKey: props.defaultWebsiteBucketEncryptionKey,
                publicReadAccess: false,
                blockPublicAccess: aws_s3_1.BlockPublicAccess.BLOCK_ALL,
                serverAccessLogsPrefix: "distribution-access-logs",
                serverAccessLogsBucket: accessLogsBucket,
            });
        const originAccessIdentity = new aws_cloudfront_1.OriginAccessIdentity(this, "OriginAccessIdentity");
        this.websiteBucket.addToResourcePolicy(new aws_iam_1.PolicyStatement({
            resources: [this.websiteBucket.bucketArn],
            actions: ["s3:ListBucket"],
            principals: [originAccessIdentity.grantPrincipal],
        }));
        const defaultRootObject = distributionProps?.defaultRootObject ?? "index.html";
        this.cloudFrontDistribution = new aws_cloudfront_1.Distribution(this, "CloudfrontDistribution", {
            ...distributionProps,
            webAclId: webAclArn,
            enableLogging: true,
            logBucket: logBucket,
            defaultBehavior: {
                ...distributionProps?.defaultBehavior,
                origin: new aws_cloudfront_origins_1.S3Origin(this.websiteBucket, {
                    originAccessIdentity,
                }),
                viewerProtocolPolicy: aws_cloudfront_1.ViewerProtocolPolicy.REDIRECT_TO_HTTPS,
            },
            defaultRootObject,
            errorResponses: distributionProps?.errorResponses ?? [
                {
                    httpStatus: 404,
                    responseHttpStatus: 200,
                    responsePagePath: `/${defaultRootObject}`,
                },
            ],
        });
        // Deploy Website
        this.bucketDeployment = new aws_s3_deployment_1.BucketDeployment(this, "WebsiteDeployment", {
            sources: [
                aws_s3_deployment_1.Source.asset(props.websiteContentPath),
                ...(props.runtimeOptions
                    ? [
                        aws_s3_deployment_1.Source.jsonData(props.runtimeOptions?.jsonFileName ||
                            DEFAULT_RUNTIME_CONFIG_FILENAME, props.runtimeOptions?.jsonPayload),
                    ]
                    : []),
            ],
            destinationBucket: this.websiteBucket,
            // Files in the distribution's edge caches will be invalidated after files are uploaded to the destination bucket.
            distribution: this.cloudFrontDistribution,
        });
        this.suppressCDKNagViolations(props);
    }
}
exports.StaticWebsite = StaticWebsite;
_a = JSII_RTTI_SYMBOL_1;
StaticWebsite[_a] = { fqn: "@aws-prototyping-sdk/static-website.StaticWebsite", version: "0.13.3" };
/**
 * If passing in distributionProps, the default behaviour.origin is a required parameter. An instance of this class can be passed in
 * to make the compiler happy.
 */
class StaticWebsiteOrigin {
    bind(_scope, _options) {
        throw new Error("This should never be called");
    }
}
exports.StaticWebsiteOrigin = StaticWebsiteOrigin;
_b = JSII_RTTI_SYMBOL_1;
StaticWebsiteOrigin[_b] = { fqn: "@aws-prototyping-sdk/static-website.StaticWebsiteOrigin", version: "0.13.3" };
//# sourceMappingURL=data:application/json;base64,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