"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.CloudfrontWebAcl = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/*! Copyright [Amazon.com](http://amazon.com/), Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0 */
const path = require("path");
const pdk_nag_1 = require("@aws-prototyping-sdk/pdk-nag");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
const aws_lambda_1 = require("aws-cdk-lib/aws-lambda");
const custom_resources_1 = require("aws-cdk-lib/custom-resources");
const cdk_nag_1 = require("cdk-nag");
const constructs_1 = require("constructs");
/**
 * This construct creates a WAFv2 Web ACL for cloudfront in the us-east-1 region (required for cloudfront) no matter the
 * region of the parent cdk stack.
 */
class CloudfrontWebAcl extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        const stack = aws_cdk_lib_1.Stack.of(this);
        const aclName = `${stack.stackName}-${id}`; // Unique per stack
        const onEventHandler = this.createOnEventHandler(stack, aclName);
        const customResource = this.createAclCustomResource(stack, aclName, onEventHandler, props);
        this.webAclId = customResource.getAttString("WebAclId");
        this.webAclArn = customResource.getAttString("WebAclArn");
    }
    /**
     * Creates an event handler for managing an ACL in us-east-1.
     *
     * @param stack containing Stack instance.
     * @param aclName name of the ACL to manage.
     * @private
     */
    createOnEventHandler(stack, aclName) {
        const onEventHandlerName = `${pdk_nag_1.PDKNag.getStackPrefix(stack)
            .split("/")
            .join("-")}OnEventHandler`;
        const onEventHandlerRole = new aws_iam_1.Role(this, "OnEventHandlerRole", {
            assumedBy: new aws_iam_1.ServicePrincipal("lambda.amazonaws.com"),
            inlinePolicies: {
                logs: new aws_iam_1.PolicyDocument({
                    statements: [
                        new aws_iam_1.PolicyStatement({
                            effect: aws_iam_1.Effect.ALLOW,
                            actions: [
                                "logs:CreateLogGroup",
                                "logs:CreateLogStream",
                                "logs:PutLogEvents",
                            ],
                            resources: [
                                `arn:aws:logs:${stack.region}:${stack.account}:log-group:/aws/lambda/${onEventHandlerName}`,
                                `arn:aws:logs:${stack.region}:${stack.account}:log-group:/aws/lambda/${onEventHandlerName}:*`,
                            ],
                        }),
                    ],
                }),
                wafv2: new aws_iam_1.PolicyDocument({
                    statements: [
                        new aws_iam_1.PolicyStatement({
                            effect: aws_iam_1.Effect.ALLOW,
                            actions: [
                                "wafv2:CreateWebACL",
                                "wafv2:DeleteWebACL",
                                "wafv2:UpdateWebACL",
                                "wafv2:GetWebACL",
                            ],
                            resources: [
                                `arn:aws:wafv2:us-east-1:${stack.account}:global/ipset/${aclName}-IPSet/*`,
                                `arn:aws:wafv2:us-east-1:${stack.account}:global/webacl/${aclName}/*`,
                                `arn:aws:wafv2:us-east-1:${stack.account}:global/managedruleset/*/*`,
                            ],
                        }),
                        new aws_iam_1.PolicyStatement({
                            effect: aws_iam_1.Effect.ALLOW,
                            actions: [
                                "wafv2:CreateIPSet",
                                "wafv2:DeleteIPSet",
                                "wafv2:UpdateIPSet",
                                "wafv2:GetIPSet",
                            ],
                            resources: [
                                `arn:aws:wafv2:us-east-1:${stack.account}:global/ipset/${aclName}-IPSet/*`,
                            ],
                        }),
                    ],
                }),
            },
        });
        const onEventHandler = new aws_lambda_1.Function(this, "CloudfrontWebAclOnEventHandler", {
            code: aws_lambda_1.Code.fromAsset(path.join(__dirname, "../lib/webacl_event_handler")),
            role: onEventHandlerRole,
            functionName: onEventHandlerName,
            handler: "index.onEvent",
            runtime: aws_lambda_1.Runtime.NODEJS_16_X,
            timeout: aws_cdk_lib_1.Duration.seconds(300),
        });
        ["AwsSolutions-IAM5", "AwsPrototyping-IAMNoWildcardPermissions"].forEach((RuleId) => {
            cdk_nag_1.NagSuppressions.addResourceSuppressions(onEventHandlerRole, [
                {
                    id: RuleId,
                    reason: "WafV2 resources have been scoped down to the ACL/IPSet level, however * is still needed as resource id's are created just in time.",
                    appliesTo: [
                        {
                            regex: `/^Resource::arn:aws:wafv2:us-east-1:${pdk_nag_1.PDKNag.getStackAccountRegex(stack)}:global/(.*)$/g`,
                        },
                    ],
                },
                {
                    id: RuleId,
                    reason: "Cloudwatch resources have been scoped down to the LogGroup level, however * is still needed as stream names are created just in time.",
                    appliesTo: [
                        {
                            regex: `/^Resource::arn:aws:logs:${pdk_nag_1.PDKNag.getStackRegionRegex(stack)}:${pdk_nag_1.PDKNag.getStackAccountRegex(stack)}:log-group:/aws/lambda/${onEventHandlerName}:\*/g`,
                        },
                    ],
                },
            ], true);
        });
        return onEventHandler;
    }
    /**
     * Creates a Custom resource to manage the deployment of the ACL.
     *
     * @param stack containing Stack instance.
     * @param aclName name of the ACL to manage.
     * @param onEventHandler event handler to use for deployment.
     * @param props user provided properties for configuring the ACL.
     * @private
     */
    createAclCustomResource(stack, aclName, onEventHandler, props) {
        const providerFunctionName = `${onEventHandler.functionName}-Provider`;
        const providerRole = new aws_iam_1.Role(this, "CloudfrontWebAclProviderRole", {
            assumedBy: new aws_iam_1.ServicePrincipal("lambda.amazonaws.com"),
            inlinePolicies: {
                logs: new aws_iam_1.PolicyDocument({
                    statements: [
                        new aws_iam_1.PolicyStatement({
                            effect: aws_iam_1.Effect.ALLOW,
                            actions: [
                                "logs:CreateLogGroup",
                                "logs:CreateLogStream",
                                "logs:PutLogEvents",
                            ],
                            resources: [
                                `arn:aws:logs:${stack.region}:${stack.account}:log-group:/aws/lambda/${providerFunctionName}`,
                                `arn:aws:logs:${stack.region}:${stack.account}:log-group:/aws/lambda/${providerFunctionName}:*`,
                            ],
                        }),
                    ],
                }),
            },
        });
        const provider = new custom_resources_1.Provider(this, "CloudfrontWebAclProvider", {
            onEventHandler,
            role: providerRole,
            providerFunctionName,
        });
        ["AwsSolutions-IAM5", "AwsPrototyping-IAMNoWildcardPermissions"].forEach((RuleId) => {
            cdk_nag_1.NagSuppressions.addResourceSuppressions(providerRole, [
                {
                    id: RuleId,
                    reason: "Cloudwatch resources have been scoped down to the LogGroup level, however * is still needed as stream names are created just in time.",
                },
            ], true);
        });
        ["AwsSolutions-L1", "AwsPrototyping-LambdaLatestVersion"].forEach((RuleId) => {
            cdk_nag_1.NagSuppressions.addResourceSuppressions(provider, [
                {
                    id: RuleId,
                    reason: "Latest runtime cannot be configured. CDK will need to upgrade the Provider construct accordingly.",
                },
            ], true);
        });
        return new aws_cdk_lib_1.CustomResource(this, "CFWebAclCustomResource", {
            serviceToken: provider.serviceToken,
            properties: {
                ID: aclName,
                MANAGED_RULES: props?.managedRules ?? [
                    { vendor: "AWS", name: "AWSManagedRulesCommonRuleSet" },
                ],
                CIDR_ALLOW_LIST: props?.cidrAllowList,
            },
        });
    }
}
exports.CloudfrontWebAcl = CloudfrontWebAcl;
_a = JSII_RTTI_SYMBOL_1;
CloudfrontWebAcl[_a] = { fqn: "@aws-prototyping-sdk/static-website.CloudfrontWebAcl", version: "0.13.3" };
//# sourceMappingURL=data:application/json;base64,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