import { Project } from "projen";
import { TypeScriptProject, TypeScriptProjectOptions } from "projen/lib/typescript";
import { SyncpackConfig } from "./syncpack-options";
/**
 * Configuration for nx targetDependencies.
 */
export declare type TargetDependencies = {
    [target: string]: TargetDependency[];
};
/**
 * Configuration for project specific targets.
 */
export declare type ProjectTargets = {
    [target: string]: ProjectTarget;
};
/**
 * Project Target.
 */
export interface ProjectTarget {
    /**
     * List of outputs to cache, relative to the root of the monorepo.
     *
     * note: must start with leading /
     */
    readonly outputs?: string[];
    /**
     * List of Target Dependencies.
     */
    readonly dependsOn: TargetDependency[];
}
/**
 * Implicit Dependencies map.
 */
export declare type ImplicitDependencies = {
    [pkg: string]: string[];
};
/**
 * Supported enums for a TargetDependency.
 */
export declare enum TargetDependencyProject {
    /**
     * Only rely on the package where the target is called.
     *
     * This is usually done for test like targets where you only want to run unit
     * tests on the target packages without testing all dependent packages.
     */
    SELF = "self",
    /**
     * Target relies on executing the target against all dependencies first.
     *
     * This is usually done for build like targets where you want to build all
     * dependant projects first.
     */
    DEPENDENCIES = "dependencies"
}
/**
 * Represents an NX Target Dependency.
 */
export interface TargetDependency {
    /**
     * Projen target i.e: build, test, etc
     */
    readonly target: string;
    /**
     * Target dependencies.
     */
    readonly projects: TargetDependencyProject;
}
/**
 * NX configurations.
 *
 * @link https://nx.dev/configuration/packagejson
 */
export interface NXConfig {
    /**
     * Affected branch.
     *
     * @default mainline
     */
    readonly affectedBranch?: string;
    /**
     * Configuration for Implicit Dependnecies.
     *
     * @link https://nx.dev/configuration/packagejson#implicitdependencies
     */
    readonly implicitDependencies?: ImplicitDependencies;
    /**
     * Configuration for TargetDependencies.
     *
     * @link https://nx.dev/configuration/packagejson#target-dependencies
     */
    readonly targetDependencies?: TargetDependencies;
    /**
     * List of patterns to include in the .nxignore file.
     *
     * @link https://nx.dev/configuration/packagejson#nxignore
     */
    readonly nxIgnore?: string[];
    /**
     * Read only access token if enabling nx cloud.
     */
    readonly nxCloudReadOnlyAccessToken?: string;
}
/**
 * Workspace configurations.
 *
 * @link https://classic.yarnpkg.com/lang/en/docs/workspaces/
 */
export interface WorkspaceConfig {
    /**
     * List of package globs to exclude from hoisting in the workspace.
     *
     * @link https://classic.yarnpkg.com/blog/2018/02/15/nohoist/
     */
    readonly noHoist?: string[];
    /**
     * Disable automatically applying `noHoist` logic for all sub-project "bundledDependencies".
     *
     * @default false
     */
    readonly disableNoHoistBundled?: boolean;
    /**
     * List of additional package globs to include in the workspace.
     *
     * All packages which are parented by the monorepo are automatically added to the workspace, but you can use this
     * property to specify any additional paths to packages which may not be managed by projen.
     */
    readonly additionalPackages?: string[];
}
/**
 * Configuration for Monorepo Upgrade Deps task.
 */
export interface MonorepoUpgradeDepsOptions {
    /**
     * Name of the task to create.
     *
     * @default upgrade-deps
     */
    readonly taskName?: string;
    /**
     * Syncpack configuration.
     *
     * No merging is performed and as such a complete syncpackConfig is required if supplied.
     *
     * @default SyncpackConfig.DEFAULT_CONFIG
     */
    readonly syncpackConfig?: SyncpackConfig;
}
/**
 * Configuration options for the NxMonorepoProject.
 */
export interface NxMonorepoProjectOptions extends TypeScriptProjectOptions {
    /**
     * Configuration for NX.
     */
    readonly nxConfig?: NXConfig;
    /**
     * Configuration for workspace.
     */
    readonly workspaceConfig?: WorkspaceConfig;
    /**
     * Whether to include an upgrade-deps task at the root of the monorepo which will upgrade all dependencies.
     *
     * @default true
     */
    readonly monorepoUpgradeDeps?: boolean;
    /**
     * Monorepo Upgrade Deps options.
     *
     * This is only used if monorepoUpgradeDeps is true.
     *
     * @default undefined
     */
    readonly monorepoUpgradeDepsOptions?: MonorepoUpgradeDepsOptions;
}
/**
 * This project type will bootstrap a NX based monorepo with support for polygot
 * builds, build caching, dependency graph visualization and much more.
 *
 * @pjid nx-monorepo
 */
export declare class NxMonorepoProject extends TypeScriptProject {
    private readonly implicitDependencies;
    private readonly targetOverrides;
    private readonly nxConfig?;
    private readonly workspaceConfig?;
    private readonly workspacePackages;
    private readonly nxJson;
    constructor(options: NxMonorepoProjectOptions);
    /**
     * Create an implicit dependency between two Project's. This is typically
     * used in polygot repos where a Typescript project wants a build dependency
     * on a Python project as an example.
     *
     * @param dependent project you want to have the dependency.
     * @param dependee project you wish to depend on.
     */
    addImplicitDependency(dependent: Project, dependee: Project): void;
    /**
     * Add one or more additional package globs to the workspace.
     * @param packageGlobs paths to the package to include in the workspace (for example packages/my-package)
     */
    addWorkspacePackages(...packageGlobs: string[]): void;
    /**
     * Allow project specific target overrides.
     */
    overrideProjectTargets(project: Project, targets: ProjectTargets): void;
    private get instantiationOrderSubProjects();
    get subProjects(): Project[];
    /**
     * @inheritDoc
     */
    synth(): void;
    /**
     * Ensures subprojects don't have a default task and that all packages use the same package manager.
     */
    private validateSubProjects;
    /**
     * For non-node projects, a package.json is required in order to be discovered by NX.
     */
    private synthesizeNonNodePackageJson;
    /**
     * Add a submodule entry to the appropriate workspace file.
     */
    private updateWorkspace;
    /**
     * Updates the install task for python projects so that they are run serially and in dependency order such that python
     * projects within the monorepo can declare dependencies on one another.
     * @private
     */
    private wirePythonDependencies;
    /**
     * Copies the given step into the given task. Step and Task must be from the given Project
     * @private
     */
    private copyStepIntoTask;
}
