"""
skyview_frame.py

Satview frame class for PyGPSClient application.

This handles a frame containing a 2D plot of satellite visibility.

Created on 13 Sep 2020

:author: semuadmin (Steve Smith)
:copyright: 2020 semuadmin
:license: BSD 3-Clause
"""

# pylint: disable = no-member

from operator import itemgetter
from tkinter import BOTH, YES, Frame

from pygpsclient.canvas_plot import (
    MODE_CEL,
    TAG_DATA,
    TAG_GRID,
    TAG_XLABEL,
    CanvasCompass,
)
from pygpsclient.globals import (
    BGCOL,
    FGCOL,
    GNSS_LIST,
    WIDGETU2,
)
from pygpsclient.helpers import col2contrast, snr2col

OL_WID = 4
FONTSCALE = 30


class SkyviewFrame(Frame):
    """
    Skyview frame class.
    """

    def __init__(self, app, *args, **kwargs):
        """
        Constructor.

        :param Frame app: reference to main tkinter application
        :param args: optional args to pass to Frame parent class
        :param kwargs: optional kwargs to pass to Frame parent class
        """

        self.__app = app  # Reference to main application class
        self.__master = self.__app.appmaster  # Reference to root class (Tk)

        Frame.__init__(self, self.__master, *args, **kwargs)

        def_w, def_h = WIDGETU2
        self.width = kwargs.get("width", def_w)
        self.height = kwargs.get("height", def_h)
        self.bg_col = BGCOL
        self.fg_col = FGCOL
        self._redraw = True
        self._body()
        self._attach_events()

    def _body(self):
        """
        Set up frame and widgets.
        """

        self.grid_columnconfigure(0, weight=1)
        self.grid_rowconfigure(0, weight=1)
        self._canvas = CanvasCompass(
            self.__app,
            self,
            MODE_CEL,
            width=self.width,
            height=self.height,
            bg=self.bg_col,
        )
        self._canvas.pack(fill=BOTH, expand=YES)

    def _attach_events(self):
        """
        Bind events to frame.
        """

        self.bind("<Configure>", self._on_resize)

    def init_frame(self):
        """
        Initialise satellite view
        """

        # only redraw the tags that have changed
        tags = (TAG_GRID, TAG_XLABEL) if self._redraw else ()
        self._canvas.create_compass(
            fontscale=FONTSCALE,
            tags=tags,
        )
        self._redraw = False

    def update_frame(self):
        """
        Plot satellites' elevation and azimuth position.
        """

        data = self.__app.gnss_status.gsv_data
        self.init_frame()

        for d in sorted(data.values(), key=itemgetter(4)):  # sort by ascending snr
            try:
                gnssId, prn, ele, azi, snr = d
                x, y = self._canvas.d2xy(int(azi), int(ele))
                snr = 0 if snr == "" else int(snr)
                (_, ol_col) = GNSS_LIST[gnssId]
                prn = f"{int(prn):02}"
                bg_col = snr2col(snr)
                self._canvas.create_circle(
                    x,
                    y,
                    self._canvas.maxr / 10,
                    outline=ol_col,
                    fill=bg_col,
                    width=OL_WID,
                    tags=TAG_DATA,
                )
                self._canvas.create_text(
                    x,
                    y,
                    text=prn,
                    fill=col2contrast(bg_col),
                    font=self._canvas.font,
                    tags=TAG_DATA,
                )
            except ValueError:
                pass

        self._canvas.update_idletasks()

    def _on_resize(self, event):  # pylint: disable=unused-argument
        """
        Resize frame

        :param event event: resize event
        """

        self.width, self.height = self.get_size()
        self._redraw = True

    def get_size(self):
        """
        Get current canvas size.

        :return: window size (width, height)
        :rtype: tuple
        """

        self.update_idletasks()  # Make sure we know about any resizing
        return self._canvas.winfo_width(), self._canvas.winfo_height()
