"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Protocol = exports.Service = exports.ServiceType = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cdk8s = require("cdk8s");
const base_1 = require("./base");
const k8s = require("./imports/k8s");
const ingress_1 = require("./ingress");
/**
 * For some parts of your application (for example, frontends) you may want to expose a Service onto an external IP address, that's outside of your cluster.
 *
 * Kubernetes ServiceTypes allow you to specify what kind of Service you want.
 * The default is ClusterIP.
 *
 * @stability stable
 */
var ServiceType;
(function (ServiceType) {
    ServiceType["CLUSTER_IP"] = "ClusterIP";
    ServiceType["NODE_PORT"] = "NodePort";
    ServiceType["LOAD_BALANCER"] = "LoadBalancer";
    ServiceType["EXTERNAL_NAME"] = "ExternalName";
})(ServiceType = exports.ServiceType || (exports.ServiceType = {}));
/**
 * An abstract way to expose an application running on a set of Pods as a network service.
 *
 * With Kubernetes you don't need to modify your application to use an unfamiliar service discovery mechanism.
 * Kubernetes gives Pods their own IP addresses and a single DNS name for a set of Pods, and can load-balance across them.
 *
 * For example, consider a stateless image-processing backend which is running with 3 replicas. Those replicas are fungible—frontends do not care which backend they use.
 * While the actual Pods that compose the backend set may change, the frontend clients should not need to be aware of that,
 * nor should they need to keep track of the set of backends themselves.
 * The Service abstraction enables this decoupling.
 *
 * If you're able to use Kubernetes APIs for service discovery in your application, you can query the API server for Endpoints,
 * that get updated whenever the set of Pods in a Service changes. For non-native applications, Kubernetes offers ways to place a network port
 * or load balancer in between your application and the backend Pods.
 *
 * @stability stable
 */
class Service extends base_1.Resource {
    /**
     * @stability stable
     */
    constructor(scope, id, props = {}) {
        var _b, _c, _d;
        super(scope, id);
        this.apiObject = new k8s.KubeService(this, 'Resource', {
            metadata: props.metadata,
            spec: cdk8s.Lazy.any({ produce: () => this._toKube() }),
        });
        this.clusterIP = props.clusterIP;
        this.externalName = props.externalName;
        if (props.externalName !== undefined) {
            this.type = ServiceType.EXTERNAL_NAME;
        }
        else {
            this.type = (_b = props.type) !== null && _b !== void 0 ? _b : ServiceType.CLUSTER_IP;
        }
        this._externalIPs = (_c = props.externalIPs) !== null && _c !== void 0 ? _c : [];
        this._ports = [];
        this._selector = {};
        this._loadBalancerSourceRanges = props.loadBalancerSourceRanges;
        for (const portAndOptions of (_d = props.ports) !== null && _d !== void 0 ? _d : []) {
            this.serve(portAndOptions.port, portAndOptions);
        }
    }
    /**
     * Expose a service via an ingress using the specified path.
     *
     * @param path The path to expose the service under.
     * @param options Additional options.
     * @returns The `Ingress` resource that was used.
     * @stability stable
     */
    exposeViaIngress(path, options = {}) {
        var _b;
        const ingress = (_b = options.ingress) !== null && _b !== void 0 ? _b : new ingress_1.Ingress(this, 'Ingress');
        ingress.addRule(path, ingress_1.IngressBackend.fromService(this), options.pathType);
        return ingress;
    }
    /**
     * Returns the labels which are used to select pods for this service.
     *
     * @stability stable
     */
    get selector() {
        return this._selector;
    }
    /**
     * Ports for this service.
     *
     * Use `serve()` to expose additional service ports.
     *
     * @stability stable
     */
    get ports() {
        return [...this._ports];
    }
    /**
     * Associate a deployment to this service.
     *
     * If not targetPort is specific in the portOptions, then requests will be routed
     * to the port exposed by the first container in the deployment's pods.
     * The deployment's `labelSelector` will be used to select pods.
     *
     * @param deployment The deployment to expose.
     * @param options Optional settings for the port.
     * @stability stable
     */
    addDeployment(deployment, options = {}) {
        var _b, _c;
        const containers = deployment.containers;
        if (containers.length === 0) {
            throw new Error('Cannot expose a deployment without containers');
        }
        // just a PoC, we assume the first container is the main one.
        // TODO: figure out what the correct thing to do here.
        const container = containers[0];
        const port = (_b = options.port) !== null && _b !== void 0 ? _b : container.port;
        const targetPort = (_c = options.targetPort) !== null && _c !== void 0 ? _c : containers[0].port;
        if (!port) {
            throw new Error('Cannot determine port. Either pass `port` in options or configure a port on the first container of the deployment');
        }
        const selector = Object.entries(deployment.labelSelector);
        if (selector.length === 0) {
            throw new Error('deployment does not have a label selector');
        }
        if (Object.keys(this.selector).length > 0) {
            throw new Error('a selector is already defined for this service. cannot add a deployment');
        }
        for (const [k, v] of selector) {
            this.addSelector(k, v);
        }
        this.serve(port, { ...options, targetPort });
    }
    /**
     * Services defined using this spec will select pods according the provided label.
     *
     * @param label The label key.
     * @param value The label value.
     * @stability stable
     */
    addSelector(label, value) {
        this._selector[label] = value;
    }
    /**
     * Configure a port the service will bind to.
     *
     * This method can be called multiple times.
     *
     * @param port The port definition.
     * @stability stable
     */
    serve(port, options = {}) {
        this._ports.push({ ...options, port });
    }
    /**
     * @internal
     */
    _toKube() {
        if (this._ports.length === 0 && this.type !== ServiceType.EXTERNAL_NAME) {
            throw new Error('A service must be configured with a port');
        }
        if (this.type === ServiceType.EXTERNAL_NAME && this.externalName === undefined) {
            throw new Error('A service with type EXTERNAL_NAME requires an externalName prop');
        }
        const ports = [];
        for (const port of this._ports) {
            ports.push({
                name: port.name,
                port: port.port,
                targetPort: port.targetPort ? k8s.IntOrString.fromNumber(port.targetPort) : undefined,
                nodePort: port.nodePort,
                protocol: port.protocol,
            });
        }
        return this.type !== ServiceType.EXTERNAL_NAME ? {
            clusterIp: this.clusterIP,
            externalIPs: this._externalIPs,
            externalName: this.externalName,
            type: this.type,
            selector: this._selector,
            ports: ports,
            loadBalancerSourceRanges: this._loadBalancerSourceRanges,
        } : {
            type: this.type,
            externalName: this.externalName,
        };
    }
}
exports.Service = Service;
_a = JSII_RTTI_SYMBOL_1;
Service[_a] = { fqn: "cdk8s-plus-22.Service", version: "1.0.0-beta.57" };
/**
 * @stability stable
 */
var Protocol;
(function (Protocol) {
    Protocol["TCP"] = "TCP";
    Protocol["UDP"] = "UDP";
    Protocol["SCTP"] = "SCTP";
})(Protocol = exports.Protocol || (exports.Protocol = {}));
//# sourceMappingURL=data:application/json;base64,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