import { ApiObject, ApiObjectMetadata, ApiObjectMetadataDefinition } from 'cdk8s';
import { Construct } from 'constructs';
import { ResourceProps, Resource } from './base';
import { Container, ContainerProps } from './container';
import * as k8s from './imports/k8s';
import { IServiceAccount } from './service-account';
import { Volume } from './volume';
/**
 * Represents a resource that can be configured with a kuberenets pod spec. (e.g `Deployment`, `Job`, `Pod`, ...).
 *
 * Use the `PodSpec` class as an implementation helper.
 *
 * @stability stable
 */
export interface IPodSpec {
    /**
     * The containers belonging to the pod.
     *
     * Use `addContainer` to add containers.
     *
     * @stability stable
     */
    readonly containers: Container[];
    /**
     * The volumes associated with this pod.
     *
     * Use `addVolume` to add volumes.
     *
     * @stability stable
     */
    readonly volumes: Volume[];
    /**
     * Restart policy for all containers within the pod.
     *
     * @stability stable
     */
    readonly restartPolicy?: RestartPolicy;
    /**
     * The service account used to run this pod.
     *
     * @stability stable
     */
    readonly serviceAccount?: IServiceAccount;
    /**
     * Add a container to the pod.
     *
     * @param container The container.
     * @stability stable
     */
    addContainer(container: ContainerProps): Container;
    /**
     * Add a volume to the pod.
     *
     * @param volume The volume.
     * @stability stable
     */
    addVolume(volume: Volume): void;
}
/**
 * Represents a resource that can be configured with a kuberenets pod template. (e.g `Deployment`, `Job`, ...).
 *
 * Use the `PodTemplate` class as an implementation helper.
 *
 * @stability stable
 */
export interface IPodTemplate extends IPodSpec {
    /**
     * Provides read/write access to the underlying pod metadata of the resource.
     *
     * @stability stable
     */
    readonly podMetadata: ApiObjectMetadataDefinition;
}
/**
 * Provides read/write capabilities ontop of a `PodSpecProps`.
 *
 * @stability stable
 */
export declare class PodSpec implements IPodSpec {
    /**
     * Restart policy for all containers within the pod.
     *
     * @stability stable
     */
    readonly restartPolicy?: RestartPolicy;
    /**
     * The service account used to run this pod.
     *
     * @stability stable
     */
    readonly serviceAccount?: IServiceAccount;
    private readonly _containers;
    private readonly _volumes;
    /**
     * @stability stable
     */
    constructor(props?: PodSpecProps);
    /**
     * The containers belonging to the pod.
     *
     * Use `addContainer` to add containers.
     *
     * @stability stable
     */
    get containers(): Container[];
    /**
     * The volumes associated with this pod.
     *
     * Use `addVolume` to add volumes.
     *
     * @stability stable
     */
    get volumes(): Volume[];
    /**
     * Add a container to the pod.
     *
     * @stability stable
     */
    addContainer(container: ContainerProps): Container;
    /**
     * Add a volume to the pod.
     *
     * @stability stable
     */
    addVolume(volume: Volume): void;
    /**
     * @internal
     */
    _toPodSpec(): k8s.PodSpec;
}
/**
 * Properties of a `PodTemplate`.
 *
 * Adds metadata information on top of the spec.
 *
 * @stability stable
 */
export interface PodTemplateProps extends PodSpecProps {
    /**
     * The pod metadata.
     *
     * @stability stable
     */
    readonly podMetadata?: ApiObjectMetadata;
}
/**
 * Provides read/write capabilities ontop of a `PodTemplateProps`.
 *
 * @stability stable
 */
export declare class PodTemplate extends PodSpec implements IPodTemplate {
    /**
     * Provides read/write access to the underlying pod metadata of the resource.
     *
     * @stability stable
     */
    readonly podMetadata: ApiObjectMetadataDefinition;
    /**
     * @stability stable
     */
    constructor(props?: PodTemplateProps);
    /**
     * @internal
     */
    _toPodTemplateSpec(): k8s.PodTemplateSpec;
}
/**
 * Properties for initialization of `Pod`.
 *
 * @stability stable
 */
export interface PodProps extends ResourceProps, PodSpecProps {
}
/**
 * Properties of a `PodSpec`.
 *
 * @stability stable
 */
export interface PodSpecProps {
    /**
     * List of containers belonging to the pod.
     *
     * Containers cannot currently be
     * added or removed. There must be at least one container in a Pod.
     *
     * You can add additionnal containers using `podSpec.addContainer()`
     *
     * @default - No containers. Note that a pod spec must include at least one container.
     * @stability stable
     */
    readonly containers?: ContainerProps[];
    /**
     * List of volumes that can be mounted by containers belonging to the pod.
     *
     * You can also add volumes later using `podSpec.addVolume()`
     *
     * @default - No volumes.
     * @see https://kubernetes.io/docs/concepts/storage/volumes
     * @stability stable
     */
    readonly volumes?: Volume[];
    /**
     * Restart policy for all containers within the pod.
     *
     * @default RestartPolicy.ALWAYS
     * @see https://kubernetes.io/docs/concepts/workloads/pods/pod-lifecycle/#restart-policy
     * @stability stable
     */
    readonly restartPolicy?: RestartPolicy;
    /**
     * A service account provides an identity for processes that run in a Pod.
     *
     * When you (a human) access the cluster (for example, using kubectl), you are
     * authenticated by the apiserver as a particular User Account (currently this
     * is usually admin, unless your cluster administrator has customized your
     * cluster). Processes in containers inside pods can also contact the
     * apiserver. When they do, they are authenticated as a particular Service
     * Account (for example, default).
     *
     * @default - No service account.
     * @see https://kubernetes.io/docs/tasks/configure-pod-container/configure-service-account/
     * @stability stable
     */
    readonly serviceAccount?: IServiceAccount;
}
/**
 * Pod is a collection of containers that can run on a host.
 *
 * This resource is
 * created by clients and scheduled onto hosts.
 *
 * @stability stable
 */
export declare class Pod extends Resource implements IPodSpec {
    /**
     * The underlying cdk8s API object.
     *
     * @see base.Resource.apiObject
     * @stability stable
     */
    protected readonly apiObject: ApiObject;
    private readonly _spec;
    /**
     * @stability stable
     */
    constructor(scope: Construct, id: string, props?: PodProps);
    /**
     * The containers belonging to the pod.
     *
     * Use `addContainer` to add containers.
     *
     * @stability stable
     */
    get containers(): Container[];
    /**
     * The volumes associated with this pod.
     *
     * Use `addVolume` to add volumes.
     *
     * @stability stable
     */
    get volumes(): Volume[];
    /**
     * Restart policy for all containers within the pod.
     *
     * @stability stable
     */
    get restartPolicy(): RestartPolicy | undefined;
    /**
     * The service account used to run this pod.
     *
     * @stability stable
     */
    get serviceAccount(): IServiceAccount | undefined;
    /**
     * Add a container to the pod.
     *
     * @stability stable
     */
    addContainer(container: ContainerProps): Container;
    /**
     * Add a volume to the pod.
     *
     * @stability stable
     */
    addVolume(volume: Volume): void;
}
/**
 * Restart policy for all containers within the pod.
 *
 * @stability stable
 */
export declare enum RestartPolicy {
    /**
     * Always restart the pod after it exits.
     *
     * @stability stable
     */
    ALWAYS = "Always",
    /**
     * Only restart if the pod exits with a non-zero exit code.
     *
     * @stability stable
     */
    ON_FAILURE = "OnFailure",
    /**
     * Never restart the pod.
     *
     * @stability stable
     */
    NEVER = "Never"
}
