"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const child_process_1 = require("child_process");
const fs = require("fs");
const path = require("path");
const util_1 = require("./util");
/**
 * Builder
 */
class Builder {
    constructor(options) {
        this.options = options;
        // Original package.json
        const pkgPath = util_1.findPkgPath();
        if (!pkgPath) {
            throw new Error('Cannot find a `package.json` in this project.');
        }
        this.pkgPath = path.join(pkgPath, 'package.json');
        this.originalPkg = fs.readFileSync(this.pkgPath);
        this.originalPkgJson = JSON.parse(this.originalPkg.toString());
    }
    /**
     * Build with parcel in a Docker container
     */
    build() {
        var _a, _b, _c, _d;
        try {
            this.updatePkg();
            const dockerBuildArgs = [
                'build',
                '--build-arg', `NODE_TAG=${this.options.nodeDockerTag}`,
                '-t', 'parcel-bundler',
                path.join(__dirname, '../parcel-bundler'),
            ];
            const build = child_process_1.spawnSync('docker', dockerBuildArgs);
            if (build.error) {
                throw build.error;
            }
            if (build.status !== 0) {
                throw new Error(`[Status ${build.status}] stdout: ${(_a = build.stdout) === null || _a === void 0 ? void 0 : _a.toString().trim()}\n\n\nstderr: ${(_b = build.stderr) === null || _b === void 0 ? void 0 : _b.toString().trim()}`);
            }
            const containerProjectRoot = '/project';
            const containerOutDir = '/out';
            const containerCacheDir = '/cache';
            const containerEntryPath = path.join(containerProjectRoot, path.relative(this.options.projectRoot, path.resolve(this.options.entry)));
            const dockerRunArgs = [
                'run', '--rm',
                '-v', `${this.options.projectRoot}:${containerProjectRoot}`,
                '-v', `${path.resolve(this.options.outDir)}:${containerOutDir}`,
                ...(this.options.cacheDir ? ['-v', `${path.resolve(this.options.cacheDir)}:${containerCacheDir}`] : []),
                '-w', path.dirname(containerEntryPath),
                'parcel-bundler',
            ];
            const parcelArgs = [
                'parcel', 'build', containerEntryPath,
                '--out-dir', containerOutDir,
                '--out-file', 'index.js',
                '--global', this.options.global,
                '--target', 'node',
                '--bundle-node-modules',
                '--log-level', '2',
                !this.options.minify && '--no-minify',
                !this.options.sourceMaps && '--no-source-maps',
                ...(this.options.cacheDir ? ['--cache-dir', containerCacheDir] : []),
            ].filter(Boolean);
            const parcel = child_process_1.spawnSync('docker', [...dockerRunArgs, ...parcelArgs]);
            if (parcel.error) {
                throw parcel.error;
            }
            if (parcel.status !== 0) {
                throw new Error(`[Status ${parcel.status}] stdout: ${(_c = parcel.stdout) === null || _c === void 0 ? void 0 : _c.toString().trim()}\n\n\nstderr: ${(_d = parcel.stderr) === null || _d === void 0 ? void 0 : _d.toString().trim()}`);
            }
        }
        catch (err) {
            throw new Error(`Failed to build file at ${this.options.entry}: ${err}`);
        }
        finally { // Always restore package.json to original
            this.restorePkg();
        }
    }
    /**
     * Updates the package.json to configure Parcel
     */
    updatePkg() {
        const updateData = {};
        // Update engines.node (Babel target)
        updateData.engines = { node: `>= ${this.options.nodeVersion}` };
        // Write new package.json
        if (Object.keys(updateData).length !== 0) {
            fs.writeFileSync(this.pkgPath, JSON.stringify({
                ...this.originalPkgJson,
                ...updateData,
            }, null, 2));
        }
    }
    restorePkg() {
        fs.writeFileSync(this.pkgPath, this.originalPkg);
    }
}
exports.Builder = Builder;
//# sourceMappingURL=data:application/json;base64,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