# -*- coding: utf-8 -*-
from setuptools import setup

package_dir = \
{'': 'src'}

packages = \
['omnipy',
 'omnipy.api',
 'omnipy.compute',
 'omnipy.compute.mixins',
 'omnipy.config',
 'omnipy.data',
 'omnipy.engine',
 'omnipy.hub',
 'omnipy.log',
 'omnipy.modules',
 'omnipy.modules.fairtracks',
 'omnipy.modules.general',
 'omnipy.modules.json',
 'omnipy.modules.pandas',
 'omnipy.modules.prefect',
 'omnipy.modules.prefect.engine',
 'omnipy.modules.raw',
 'omnipy.modules.tables',
 'omnipy.util']

package_data = \
{'': ['*'], 'omnipy.modules.prefect': ['settings/*']}

install_requires = \
['aiostream>=0.4.5,<0.5.0',
 'greenlet>=2.0.1,<3.0.0',
 'inflection>=0.5.1,<0.6.0',
 'pandas>=1.2.4,<2.0.0',
 'prefect>=2.7.3,<3.0.0',
 'pydantic>=1.10.2,<2.0.0',
 'pytest-asyncio>=0.20.2,<0.21.0',
 'pytest-cases>=3.6.13,<4.0.0',
 'requests>=2.25.1,<3.0.0',
 'typing-inspect>=0.8.0,<0.9.0']

setup_kwargs = {
    'name': 'omnipy',
    'version': '0.9.5',
    'description': 'Omnipy is a high level Python library for type-driven data wrangling and scalable workflow orchestration (under development)',
    'long_description': '\n![Omnypy logo](https://fairtracks.net/_nuxt/img/9a84303.webp)\n\nOmnipy is the new name of the Python package formerly known as uniFAIR.\n\nWe are very grateful to Dr. Jamin Chen, who gracefully transferred ownership of the (mostly unused) "omnipy" name in PyPI to us!\n\n--\n\nUpdate Feb 3, 2023: Documentation of the Omnipy API is still sparse. However, for running examples,\nplease check out the [omnipy-examples repo](https://github.com/fairtracks/omnipy_examples) and its\nrelated [PYPI package](https://pypi.org/project/omnipy-examples/)!\n\n(NOTE: Read the section [Transformation on the FAIRtracks.net website](https://fairtracks.net/fair/#fair-07-transformation)\nfor a more detailed and better formatted version of the following description!)\n\n## Generic functionality\n\nOmnipy is designed primarily to simplify development and deployment of (meta)data transformation \nprocesses in the context of FAIRification and data brokering efforts. However, the functionality is \nvery generic and can also be used to support research data (and metadata) transformations in a range \nof fields and contexts beyond life science, including day-to-day research scenarios:\n\n![Conceptual overview of Omnipy](https://fairtracks.net/_nuxt/img/aef0283-1440.png)\n\n**Data wrangling in day-to-day research:** Researchers in life science and other data-centric fields\noften need to extract, manipulate and integrate data and/or metadata from different sources, such as\nrepositories, databases or flat files. Much research time is spent on trivial and not-so-trivial\ndetails of such ["data wrangling"](https://en.wikipedia.org/wiki/Data_wrangling):\n\n- reformat data structures\n- clean up errors\n- remove duplicate data\n- map and integrate dataset fields\n- etc.\n\nGeneral software for data wrangling and analysis, such as [Pandas](https://pandas.pydata.org/),\n[R](https://www.r-project.org/) or [Frictionless](https://frictionlessdata.io/), are useful, but\nresearchers still regularly end up with hard-to-reuse scripts, often with manual steps.\n\n**Step-wise data model transformations:** With the Omnipy\nPython package, researchers can import (meta)data in almost any shape or form: _nested JSON; tabular\n(relational) data; binary streams; or other data structures_. Through a step-by-step process, data\nis continuously parsed and reshaped according to a series of data model transformations.\n\n<ui-quote-text quote=\'Omnipy tasks (single steps) and flows (workflows) are defined as transformations from specific input data models to specific output data models.\'>\n</ui-quote-text>\n\n**"Parse, don\'t validate":** Omnipy follows the principles of "Type-driven design" (read \n_Technical note #2: "Parse, don\'t validate"_ on the \n[FAIRtracks.net website](https://fairtracks.net/fair/#fair-07-transformation) for more info). It \nmakes use of cutting-edge [Python type hints](https://peps.python.org/pep-0484/) and the popular\n[pydantic](https://pydantic-docs.helpmanual.io/) package to "pour" data into precisely defined data\nmodels that can range from very general (e.g. _"any kind of JSON data", "any kind of tabular data"_,\netc.) to very specific (e.g. _"follow the FAIRtracks JSON Schema for track files with the extra\nrestriction of only allowing BigBED files"_).\n\n**Data types as contracts:** Omnipy _tasks_ (single steps) or _flows_ (workflows) are defined as\ntransformations from specific _input_ data models to specific _output_ data models.\n[pydantic](https://pydantic-docs.helpmanual.io/)-based parsing guarantees that the input and output\ndata always follows the data models (i.e. data types). Thus, the data models defines "contracts"\nthat simplifies reuse of tasks and flows in a _mix-and-match_ fashion.\n\n**Catalog of common processing steps:** Omnipy is built from the ground up to be modular. We aim \nto provide a catalog of commonly useful functionality ranging from:\n\n- data import from REST API endpoints, common flat file formats, database dumps, etc.\n- flattening of complex, nested JSON structures\n- standardization of relational tabular data (i.e. removing redundancy)\n- mapping of tabular data between schemas\n- lookup and mapping of ontology terms\n- semi-automatic data cleaning (through e.g. [Open Refine](https://openrefine.org/))\n- support for common data manipulation software and libraries, such as\n  [Pandas](https://pandas.pydata.org/), [R](https://www.r-project.org/),\n  [Frictionless](https://frictionlessdata.io/), etc.\n\nIn particular, we will provide a _FAIRtracks_ module that contains data models and processing steps\nto transform metadata to follow the [FAIRtracks standard](/standards/#standards-01-fairtracks).\n\n![Catalog of commonly useful processing steps, data modules and tool integrations](https://fairtracks.net/_nuxt/img/7101c5f-1280.png)\n\n**Refine and apply templates:** An Omnipy module typically consists of a set of generic _task_ and\n_flow templates_ with related data models, (de)serializers, and utility functions. The user can then\npick task and flow templates from this extensible, modular catalog, further refine them in the\ncontext of a custom, use case-specific flow, and apply them to the desired compute engine to carry\nout the transformations needed to wrangle data into the required shape.\n\n**Rerun only when needed:** When piecing together a custom flow in Omnipy, the user has persistent\naccess to the state of the data at every step of the process. Persistent intermediate data allows\nfor caching of tasks based on the input data and parameters. Hence, if the input data and parameters\nof a task does not change between runs, the task is not rerun. This is particularly useful for\nimporting from REST API endpoints, as a flow can be continuously rerun without taxing the remote\nserver; data import will only carried out in the initial iteration or when the REST API signals that\nthe data has changed.\n\n**Scale up with external compute resources:** In the case of large datasets, the researcher can set\nup a flow based on a representative sample of the full dataset, in a size that is suited for running\nlocally on, say, a laptop. Once the flow has produced the correct output on the sample data, the\noperation can be seamlessly scaled up to the full dataset and sent off in\n[software containers](https://www.docker.com/resources/what-container/) to run on external compute\nresources, using e.g. [Kubernetes](https://kubernetes.io/). Such offloaded flows\ncan be easily monitored using a web GUI.\n\n![Working with Omnipy directly from an Integrated Development Environment (IDE)](https://fairtracks.net/_nuxt/img/f9be071-1440.png)\n\n**Industry-standard ETL backbone:** Offloading of flows to external compute resources is provided by\nthe integration of Omnipy with a workflow engine based on the [Prefect](https://www.prefect.io/)\nPython package. Prefect is an industry-leading platform for dataflow automation and orchestration\nthat brings a [series of powerful features](https://www.prefect.io/opensource/) to Omnipy:\n\n- Predefined integrations with a range of compute infrastructure solutions\n- Predefined integration with various services to support extraction, transformation, and loading\n  (ETL) of data and metadata\n- Code as workflow ("If Python can write it, Prefect can run it")\n- Dynamic workflows: no predefined Direct Acyclic Graphs (DAGs) needed!\n- Command line and web GUI-based visibility and control of jobs\n- Trigger jobs from external events such as GitHub commits, file uploads, etc.\n- Define continuously running workflows that still respond to external events\n- Run tasks concurrently through support for asynchronous tasks\n\n![Overview of the compute and storage infrastructure integrations that comes built-in with Prefect](https://fairtracks.net/_nuxt/img/ccc322a-1440.png)\n\n**Pluggable workflow engines:** It is also possible to integrate Omnipy with other workflow\nbackends by implementing new workflow engine plugins. This is relatively easy to do, as the core\narchitecture of Omnipy allows the user to easily switch the workflow engine at runtime. Omnipy\nsupports both traditional DAG-based and the more _avant garde_ code-based definition of flows. Two\nworkflow engines are currently supported: _local_ and _prefect_.\n\n## Scenarios\nAs initial use cases, we will consider the following two scenarios:\n* Transforming ENCODE metadata into FAIRtracks format\n* Transforming TCGA metadata into FAIRtracks format\n\n## Nomenclature:\n* Omnipy is designed to work with content which could be classified both as data and metadata in their original context. For simplicity, we will refer to all such content as "data".\n\n## Overview of the proposed FAIRification process:\n\n* ### Step 1: Import data from original source:\n  * #### 1A: From API endpoints\n    * _Input:_ API endpoint producing JSON data\n    * _Output:_ JSON files (possibly with embedded JSON objects/lists [as strings])\n    * _Description:_ General interface to support various API endpoints. Import all data by crawling API endpoints providing JSON content\n    * _Generalizable:_ Partly (at least reuse of utility functions)\n    * _Manual/automatic:_ Automatic\n    * _Details:_\n      * GDC/TCGA substeps (implemented as Step objects with file input/output)\n        * 1A. Filtering step:\n          * Input: parameters defining how to filter, e.g.:\n            * For all endpoints (projects, cases, files, annotations), support:\n              * Filter on list of IDs\n              * Specific number of items\n              * All\n            * Example config:\n              * projects: 2 items\n              * cases: 2 items\n              * files: all\n              * annotations: all\n            *  Define standard configurations, e.g.:\n                * Default: limited extraction (3 projects * 3 cases * 5 files? (+ annotations?))\n                * All TCGA\n                * List of projects\n            * Hierarchical for loop through endpoints to create filter definitions\n          * Output: Filter definitions as four files, e.g. as JSON, \n            as they should be input to the filter parameter to the API:\n             ```\n            projects_filter.json:\n            {\n                "op": "in",\n                "content": {\n                    "field": "project_id",\n                    "value": [\'TCGA_ABCD\', \'TCGA_BCDE\']\n                }\n            }\n            \n            cases_filter.json:\n            {\n                "op": "in",\n                "content": {\n                    "field": "case_id",\n                    "value": [\'1234556\', \'234567\', \'3456789\', \'4567890\']\n                }\n            }\n            \n            files_filter.json:\n            {\n                "op": "in",\n                "content": {\n                    "field": "file_id",\n                    "value": [\'1234556\', \'234567\', \'3456789\', \'4567890\']\n                }\n            }\n            \n            annotations.json\n            {\n                "op": "in",\n                "content": {\n                    "field": "annotation_id",\n                    "value": [\'1234556\', \'234567\', \'3456789\', \'4567890\']\n                }\n            }\n            ```\n            \n        * 1B. Fetch and divide all fields step:\n          * Input: None\n          * Output: JSON files specifying all the fields of an endpoint fetched from the `mapping` API. \n              The fields should be divided into chunks of a size that is small enough for the endpoints to \n              handle. The JSON output should also specify the primary_key field, that needs to be added to \n              all the API calls in order for the results to be joinable.\n              \n              Example JSON files:\n             ```\n            projects_fields.json:\n            {\n                "primary_key": "project_id",\n                "fields_divided": [\n                    ["field_a", "field_b"],\n                    ["field_c.subfield_a", "field_c.subfield_b", "field_d"]      \n                ]\n            }\n            \n            (...) # For all endpoints\n            ```\n        * 1C. Download from all endpoints according to the filters and the field divisions.\n              If there is a limitation on the number of hits that the endpoint is able to return, divide into smaller\n              API calls for a certain number of hits and concatenate the results. Make sure that proper waiting time\n              (1 second?) is added between the calls (to not overload the endpoint).\n        * 1D. Extract identifiers from nested objects (when present) and insert into parents objects\n      * ENCODE:\n        * Identify where to start (Cart? Experiment?)\n        * To get all data for a table (double-check this): `https://www.encodeproject.org/experiments/@@listing?format=json&frame=object`\n        * Download all tables directly.\n  * #### 1b: From JSON files\n    * _Input:_ JSON content as files\n    * _Output:_ Pandas DataFrames (possibly with embedded JSON objects/lists)\n    * _Description:_ Import data from files. Requires specific parsers to be implemented.\n    * _Generalizable:_ Fully\n    * _Manual/automatic:_ Automatic\n  * #### 1c: From non-JSON files\n    * _Input:_ File content in some supported format (e.g. GSuite)\n    * _Output:_ Pandas DataFrames (possibly containing lists of identifiers as Pandas Series) + reference metadata\n    * _Description:_ Import data from files. Requires specific parsers to be implemented.\n    * _Generalizable:_ Partly (generating reference metadata might be tricky)\n    * _Manual/automatic:_ Automatic\n  * #### 1d: From database\n    * _Input:_ Direct access to relational database\n    * _Output:_ Pandas DataFrames (possibly containing lists of identifiers as Pandas Series) + reference metadata\n    * _Description:_ Import data from database\n    * _Generalizable:_ Fully\n    * _Manual/automatic:_ Automatic\n* ### Step 2: JSON cleanup\n    * _Input:_ Pandas DataFrames (possibly with embedded JSON objects/lists)\n    * _Output:_ Pandas DataFrames (possibly containing lists of identifiers as Pandas Series) + reference metadata\n    * _Description:_ Replace embedded objects with identifiers (possibly as lists)\n    * _Generalizable:_ Partly (generating reference metadata might be tricky)\n    * _Manual/automatic:_ Depending on original input\n    * _Details:_\n      * If there are embedded objects from other tables:\n        * ENCODE update: \n          * By using the `frame=object` parameter, we will not get any embedded objects from the APIs for the main tables. There is, however, some "auditing" fields that contain JSON objects. We can ignore these in the first iteration.\n        * If the original table of the embedded objects can be retrieved directly from an API, replace such embedded objects with unique identifiers to the object in another table (maintaining a reference to the name of the table, if needed)\n          * Record the reference metadata `(table_from, attr_from) -> (table_to, attr_to)` for joins:\n            * Example: `(table: "experiment", column: "replicates") -> (table: "replicate", column: "@id")`\n        * If the original table of the embedded objects are not directly available from an API, one needs to fill out the other table with the content that is embedded in the current object, creating the table if needed.\n      * For all fields with identifiers that reference other tables:\n        * Record the reference metadata `(table_from, attr_from) -> (table_to, attr_to)` for joins.\n        * If the field contains a list of identifiers\n          * Convert into Pandas Series\n* ### Step 3: Create reference tables to satisfy 1NF\n    * _Input:_ Pandas DataFrames (possibly containing lists of identifiers as Pandas Series) + reference metadata\n    * _Output:_ Pandas DataFrames (original tables without reference column) [1NF] + reference tables + reference metadata\n    * _Description:_ Move references into separate tables, transforming the tables in first normal form ([1NF](https://en.wikipedia.org/wiki/Database_normalization#Satisfying_1NF))\n    * _Generalizable:_ Fully\n    * _Manual/automatic:_ Automatic\n    * _Details:_\n      * For each reference pair: \n        * Create a reference table\n        * For each item in the "from"-reference column:\n          * Add new rows in the reference table for each "to"-identifier, using the same "from"-identifier\n            * Example: Table "experiment-replicate" with columns "experiment.@id", "replicate.@id"\n        * Delete the complete column from the original table\n* ### Step 4: Satisfy 2NF\n    * _Input:_ Pandas DataFrames (original tables without reference column) [1NF] + reference tables\n    * _Output:_ Pandas DataFrames (original tables without reference column) [2NF] + reference tables\n    * _Description:_ Automatic transformation of original tables into second normal form ([2NF](https://en.wikipedia.org/wiki/Database_normalization#Satisfying_2NF)):\n    * _Generalizable:_ Fully (if not, we skip it)\n    * _Manual/automatic:_ Automatic\n    * _Details:_\n      * Use existing library.\n* ### Step 5: Satisfy 3NF\n    * _Input:_ Pandas DataFrames (original tables without reference column) [2NF] + reference tables\n    * _Output:_ Pandas DataFrames (original tables without reference column) [3NF] + reference tables\n    * _Description:_ Automatic transformation of original tables into third normal form ([3NF](https://en.wikipedia.org/wiki/Database_normalization#Satisfying_3NF)):\n    * _Generalizable:_ Fully (if not, we skip it)\n    * _Manual/automatic:_ Automatic\n    * _Details:_\n      * Use existing library.\n* ### Step 6: Create model map\n    * _Input:_ Pandas DataFrames (original tables without reference column) [Any NF] + reference tables + FAIRtracks JSON schemas\n    * _Output:_ Model map [some data structure (to be defined) mapping FAIRtracks objects and attributes to tables/columns in the original data]\n    * _Description:_ Manual mapping of FAIRtracks objects and attributes to corresponding tables and columns in the original data.\n    * _Generalizable:_ Fully\n    * _Manual/automatic:_ Manual\n    * Details:\n      * For each FAIRtracks object:\n        * Define a start table in the original data\n        * For each FAIRtracks attribute:\n          * Manually find the path (or paths) to the original table/column that this maps to\n            * _Example:_ `Experiments:organism (FAIRtracks) -> Experiments.Biosamples.Organism.scientific_name`\n* ### Step 7: Apply model map to generate initial FAIRtracks tables\n  * _Input:_ Pandas DataFrames (original tables without reference column) [Any NF] + reference tables + Model map \n  * _Output:_ Pandas DataFrames (initial FAIRtracks tables, possibly with multimapped attributes)\n    * Example: `Experiment.target_from_origcolumn1` and `Experimentl.target_from_origcolumn2` contain content from two different attributes from the original data that both corresponds to `Experiment.target`\n  * _Description:_ Generate initial FAIRtracks tables by applying the model map, mapping FAIRtracks attributes with one or more attributes (columns) in the original table.\n  * _Generalizable:_ Fully\n  * _Manual/automatic:_ Automatic\n  * _Details:_\n    * For every FAIRtracks object:\n      * Create a new pandas DataFrame\n      * For every FAIRtracks attribute:\n        * From the model map, get the path to the corresponding original table/column, or a list of such paths in case of multimapping\n        * For each path:\n          * Automatically join tables to get primary keys and attribute value in the same table:\n            * _Example:_ `experiment-biosample JOIN biosample-organism JOIN organism` will create mapping table with two columns: `Experiments.local_id` and `Organism.scientific_name`\n          * Add column to FAIRtracks DataFrame\n          * In case of multimodeling, record the relation between FAIRtracks attribute and corresponding multimapped attributes, e.g. by generating unique attribute names for each path, such as `Experiment.target_from_origcolumn1` and `Experiment.target_from_origcolumn2`, which one can derive directly from the model map.\n* ### Step 8: Harmonize multimapped attributes\n  * _Input:_ Pandas DataFrames (initial FAIRtracks tables, possibly with multimapped attributes) + model map\n  * _Output:_ Pandas DataFrames (initial FAIRtracks tables)\n  * _Description:_ Harmonize multimapped attributes manually, or possibly by applying scripts\n  * _Generalizable:_ Limited (mostly by reusing util functions)\n  * _Manual/automatic:_ Mixed (possibly scriptable)\n  * _Details:_\n    * For all multimapped attributes:\n      * Manually review values (in batch mode) and generate a single output value for each combination:\n        * Hopefully Open Refine can be used for this. If so, one needs to implement data input/output mechanisms.\n\n* ### Further steps to be detailed:\n  * For all FAIRtracks attributes with ontology terms: Convert terms using required ontologies\n  * Other FAIRtracks specific value conversion \n  * Manual batch correction of values (possibly with errors), probably using Open Refine\n  * Validation of FAIRtracks document\n\nSuggestion: we will use Pandas DataFrames as the core data structure for tables, given that the library provides the required features (specifically Foreign key and Join capabilities)\n\n\n',
    'author': 'Sveinung Gundersen',
    'author_email': 'sveinugu@gmail.com',
    'maintainer': 'None',
    'maintainer_email': 'None',
    'url': 'https://fairtracks.net/fair/#fair-07-transformation',
    'package_dir': package_dir,
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.10,<4.0',
}


setup(**setup_kwargs)
