from typing import Dict, Any, List
from datetime import datetime, timedelta
from .api_client import PersianDateAPI

class DateConverter:
    """Convert between different date formats"""
    
    def __init__(self, api_client=None):
        self.api = api_client or PersianDateAPI()
    
    def get_all_formats(self) -> Dict[str, str]:
        """Get date in all available formats"""
        data = self.api.get_current_time()
        date_info = data.get('date', {})
        
        return {
            'persian_official_iso': date_info.get('full', {}).get('official', {}).get('iso', {}).get('fa'),
            'persian_official_usual': date_info.get('full', {}).get('official', {}).get('usual', {}).get('fa'),
            'persian_unofficial_iso': date_info.get('full', {}).get('unofficial', {}).get('iso', {}).get('fa'),
            'persian_unofficial_usual': date_info.get('full', {}).get('unofficial', {}).get('usual', {}).get('fa'),
            'gregorian_iso': date_info.get('other', {}).get('gregorian', {}).get('iso', {}).get('fa'),
            'gregorian_usual': date_info.get('other', {}).get('gregorian', {}).get('usual', {}).get('fa'),
            'ghamari_iso': date_info.get('other', {}).get('ghamari', {}).get('iso', {}).get('fa'),
            'ghamari_usual': date_info.get('other', {}).get('ghamari', {}).get('usual', {}).get('fa')
        }
    
    def get_time_in_all_formats(self) -> Dict[str, str]:
        """Get time in all available formats"""
        data = self.api.get_current_time()
        
        return {
            '12h_short': data.get('time12', {}).get('full', {}).get('short', {}).get('fa'),
            '12h_full': data.get('time12', {}).get('full', {}).get('full', {}).get('fa'),
            '24h_full': data.get('time24', {}).get('full', {}).get('fa'),
            'components_12h': {
                'hour': data.get('time12', {}).get('hour', {}).get('fa'),
                'minute': data.get('time12', {}).get('minute', {}).get('fa'),
                'second': data.get('time12', {}).get('second', {}).get('fa'),
                'shift': data.get('time12', {}).get('shift', {})
            },
            'components_24h': {
                'hour': data.get('time24', {}).get('hour', {}).get('fa'),
                'minute': data.get('time24', {}).get('minute', {}).get('fa'),
                'second': data.get('time24', {}).get('second', {}).get('fa')
            }
        }

class HolidayChecker:
    """Check and manage holiday information"""
    
    def __init__(self, api_client=None):
        self.api = api_client or PersianDateAPI()
    
    def get_holiday_status(self) -> Dict[str, Any]:
        """Get complete holiday status"""
        is_holiday = self.api.is_holiday()
        holiday_info = self.api.get_holiday_info()
        
        return {
            'is_holiday': is_holiday,
            'holiday_info': holiday_info,
            'message': 'امروز تعطیل است 🎉' if is_holiday else 'امروز روز کاری است 💼'
        }
    
    def get_upcoming_holidays(self, days_ahead: int = 30) -> List[Dict[str, Any]]:
        """Get upcoming holidays (mock implementation - in real scenario would need historical data)"""
        # This is a simplified version - in production you'd need a holidays database
        current_holiday = self.get_holiday_status()
        
        if current_holiday['is_holiday']:
            return [{
                'date': self.api.get_full_persian_date(),
                'event': current_holiday['holiday_info']['text'] if current_holiday['holiday_info'] else 'تعطیلی',
                'is_today': True
            }]
        
        return []