import json
from typing import Dict, Any, List
from datetime import datetime
from .date_parser import DateParser
from .api_client import PersianDateAPI

class DateFormatter:
    """Format date information in various output formats"""
    
    def __init__(self, api_client=None):
        self.parser = DateParser(api_client)
    
    def format_as_text(self) -> str:
        """Format complete date info as readable text"""
        calendar_info = self.parser.parse_calendar_info()
        time_info = self.parser.parse_time_components()
        events_info = self.parser.parse_events_info()
        
        text = f"""
📅 تاریخ شمسی:
{calendar_info['weekday']['name']}، {calendar_info['day']['number']} {calendar_info['month']['name']} {calendar_info['year']['number']}

⏰ زمان:
{time_info['time_12h']['full']} ({time_info['time_24h']['full']})

📊 اطلاعات سال:
سال {calendar_info['year']['name']} - حیوان سال: {calendar_info['year']['animal']}
{calendar_info['year']['progress']['passed_percent']}% از سال گذشت ({calendar_info['year']['progress']['passed_days']} روز)

🎯 اطلاعات ماه:
ماه {calendar_info['month']['name']} - برج: {calendar_info['month']['asterism']}

"""
        
        if events_info['is_holiday']:
            holy_event = events_info['holy_events']
            text += f"🎉 امروز تعطیل است:\n{holy_event['text']}\n"
        
        return text.strip()
    
    def format_as_json(self, pretty: bool = True) -> str:
        """Format complete date info as JSON"""
        complete_data = {
            'timestamp': datetime.now().isoformat(),
            'time': self.parser.parse_time_components(),
            'date': self.parser.parse_date_components(),
            'calendar': self.parser.parse_calendar_info(),
            'season': self.parser.parse_season_info(),
            'events': self.parser.parse_events_info()
        }
        
        if pretty:
            return json.dumps(complete_data, ensure_ascii=False, indent=2)
        else:
            return json.dumps(complete_data, ensure_ascii=False)
    
    def format_as_html(self, title: str = "اطلاعات تاریخ شمسی") -> str:
        """Format date info as HTML table"""
        calendar_info = self.parser.parse_calendar_info()
        time_info = self.parser.parse_time_components()
        
        html = f"""
<!DOCTYPE html>
<html dir="rtl" lang="fa">
<head>
    <meta charset="UTF-8">
    <title>{title}</title>
    <style>
        body {{ font-family: Tahoma, Arial, sans-serif; direction: rtl; padding: 20px; background: #f5f5f5; }}
        .container {{ max-width: 800px; margin: 0 auto; background: white; padding: 30px; border-radius: 10px; box-shadow: 0 2px 10px rgba(0,0,0,0.1); }}
        .header {{ text-align: center; margin-bottom: 30px; border-bottom: 2px solid #4CAF50; padding-bottom: 20px; }}
        .section {{ margin-bottom: 25px; padding: 20px; background: #f9f9f9; border-radius: 8px; }}
        .section-title {{ color: #4CAF50; margin-bottom: 15px; font-size: 18px; }}
        .info-grid {{ display: grid; grid-template-columns: 1fr 1fr; gap: 15px; }}
        .info-item {{ padding: 10px; background: white; border-radius: 5px; border-right: 3px solid #4CAF50; }}
        .holiday {{ background: #FFF3CD; border-color: #FFC107; }}
        .progress-bar {{ background: #e0e0e0; border-radius: 10px; overflow: hidden; margin: 10px 0; }}
        .progress-fill {{ background: #4CAF50; height: 20px; }}
    </style>
</head>
<body>
    <div class="container">
        <div class="header">
            <h1>{title}</h1>
            <p>تاریخ و زمان فعلی به وقت ایران</p>
        </div>
        
        <div class="section">
            <div class="section-title">📅 تاریخ امروز</div>
            <div class="info-grid">
                <div class="info-item">
                    <strong>تاریخ کامل:</strong><br>
                    {calendar_info['weekday']['name']}، {calendar_info['day']['number']} {calendar_info['month']['name']} {calendar_info['year']['number']}
                </div>
                <div class="info-item">
                    <strong>روز هفته:</strong><br>
                    {calendar_info['weekday']['name']} (روز {calendar_info['weekday']['number']})
                </div>
            </div>
        </div>
        
        <div class="section">
            <div class="section-title">⏰ زمان فعلی</div>
            <div class="info-grid">
                <div class="info-item">
                    <strong>فرم 12 ساعته:</strong><br>
                    {time_info['time_12h']['full']}
                </div>
                <div class="info-item">
                    <strong>فرم 24 ساعته:</strong><br>
                    {time_info['time_24h']['full']}
                </div>
            </div>
        </div>
        
        <div class="section">
            <div class="section-title">📊 اطلاعات سال</div>
            <div class="info-grid">
                <div class="info-item">
                    <strong>نام سال:</strong><br>
                    {calendar_info['year']['name']}
                </div>
                <div class="info-item">
                    <strong>حیوان سال:</strong><br>
                    {calendar_info['year']['animal']}
                </div>
                <div class="info-item">
                    <strong>روزهای گذشته:</strong><br>
                    {calendar_info['year']['progress']['passed_days']} روز ({calendar_info['year']['progress']['passed_percent']}%)
                </div>
                <div class="info-item">
                    <strong>روزهای باقی‌مانده:</strong><br>
                    {calendar_info['year']['progress']['remaining_days']} روز ({calendar_info['year']['progress']['remaining_percent']}%)
                </div>
            </div>
        </div>
"""
        
        # Add holiday section if today is holiday
        events_info = self.parser.parse_events_info()
        if events_info['is_holiday']:
            holy_event = events_info['holy_events']
            html += f"""
        <div class="section">
            <div class="section-title">🎉 مناسبت امروز</div>
            <div class="info-item holiday">
                <strong>{holy_event['text']}</strong><br>
                امروز تعطیل رسمی است
            </div>
        </div>
"""
        
        html += """
    </div>
</body>
</html>
"""
        return html

class CalendarGenerator:
    """Generate calendar-related outputs"""
    
    def __init__(self, api_client=None):
        self.api = api_client or PersianDateAPI()
    
    def get_year_progress_bar(self, width: int = 20) -> str:
        """Generate ASCII progress bar for year progress"""
        progress = self.api.get_year_progress()
        passed_percent = progress['passed_percent']
        
        filled = int(width * passed_percent / 100)
        empty = width - filled
        
        bar = '█' * filled + '░' * empty
        return f"[{bar}] {passed_percent:.1f}%"
    
    def get_season_emoji(self) -> str:
        """Get emoji for current season"""
        season_info = self.api.get_season_info()
        season_name = season_info.get('name', '')
        
        emoji_map = {
            'بهار': '🌸',
            'تابستان': '☀️',
            'پاییز': '🍂',
            'زمستان': '⛄'
        }
        
        return emoji_map.get(season_name, '📅')