import re
from typing import Dict, Any, Optional
from datetime import datetime
from .api_client import PersianDateAPI

class DateParser:
    """
    Parse and extract specific information from date API response
    """
    
    def __init__(self, api_client=None):
        self.api = api_client or PersianDateAPI()
    
    def parse_time_components(self) -> Dict[str, Any]:
        """Parse all time components"""
        data = self.api.get_current_time()
        
        return {
            'unix_timestamp': {
                'persian': data.get('unix', {}).get('fa'),
                'english': data.get('unix', {}).get('en')
            },
            'time_12h': {
                'short': data.get('time12', {}).get('full', {}).get('short', {}).get('fa'),
                'full': data.get('time12', {}).get('full', {}).get('full', {}).get('fa'),
                'hour': data.get('time12', {}).get('hour', {}).get('fa'),
                'minute': data.get('time12', {}).get('minute', {}).get('fa'),
                'second': data.get('time12', {}).get('second', {}).get('fa'),
                'shift': data.get('time12', {}).get('shift', {})
            },
            'time_24h': {
                'full': data.get('time24', {}).get('full', {}).get('fa'),
                'hour': data.get('time24', {}).get('hour', {}).get('fa'),
                'minute': data.get('time24', {}).get('minute', {}).get('fa'),
                'second': data.get('time24', {}).get('second', {}).get('fa')
            }
        }
    
    def parse_date_components(self) -> Dict[str, Any]:
        """Parse all date components"""
        data = self.api.get_current_time()
        date_info = data.get('date', {})
        
        return {
            'persian_date': {
                'official_iso': date_info.get('full', {}).get('official', {}).get('iso', {}).get('fa'),
                'official_usual': date_info.get('full', {}).get('official', {}).get('usual', {}).get('fa'),
                'unofficial_iso': date_info.get('full', {}).get('unofficial', {}).get('iso', {}).get('fa'),
                'unofficial_usual': date_info.get('full', {}).get('unofficial', {}).get('usual', {}).get('fa')
            },
            'gregorian_date': {
                'iso': date_info.get('other', {}).get('gregorian', {}).get('iso', {}).get('fa'),
                'usual': date_info.get('other', {}).get('gregorian', {}).get('usual', {}).get('fa')
            },
            'ghamari_date': {
                'iso': date_info.get('other', {}).get('ghamari', {}).get('iso', {}).get('fa'),
                'usual': date_info.get('other', {}).get('ghamari', {}).get('usual', {}).get('fa')
            }
        }
    
    def parse_calendar_info(self) -> Dict[str, Any]:
        """Parse calendar information"""
        data = self.api.get_current_time()
        date_info = data.get('date', {})
        
        return {
            'year': {
                'number': date_info.get('year', {}).get('number', {}).get('fa'),
                'name': date_info.get('year', {}).get('name'),
                'animal': date_info.get('year', {}).get('animal'),
                'leap_year': date_info.get('year', {}).get('leapyear') == 'کبیسه',
                'progress': {
                    'passed_days': date_info.get('year', {}).get('agone', {}).get('days', {}).get('fa'),
                    'passed_percent': date_info.get('year', {}).get('agone', {}).get('percent', {}).get('fa'),
                    'remaining_days': date_info.get('year', {}).get('left', {}).get('days', {}).get('fa'),
                    'remaining_percent': date_info.get('year', {}).get('left', {}).get('percent', {}).get('fa')
                }
            },
            'month': {
                'number': date_info.get('month', {}).get('number', {}).get('fa'),
                'name': date_info.get('month', {}).get('name'),
                'asterism': date_info.get('month', {}).get('asterism')
            },
            'day': {
                'number': date_info.get('day', {}).get('number', {}).get('fa'),
                'name': date_info.get('day', {}).get('name'),
                'is_holiday': self.api.is_holiday()
            },
            'weekday': {
                'number': date_info.get('weekday', {}).get('number', {}).get('fa'),
                'name': date_info.get('weekday', {}).get('name'),
                'champ': date_info.get('weekday', {}).get('champ')
            }
        }
    
    def parse_season_info(self) -> Dict[str, Any]:
        """Parse season information"""
        data = self.api.get_current_time()
        season_info = data.get('season', {})
        
        return {
            'name': season_info.get('name'),
            'number': {
                'persian': season_info.get('number', {}).get('fa'),
                'english': season_info.get('number', {}).get('en')
            }
        }
    
    def parse_events_info(self) -> Dict[str, Any]:
        """Parse events and holidays"""
        data = self.api.get_current_time()
        events = data.get('date', {}).get('day', {}).get('events', {})
        
        return {
            'holy_events': events.get('holy'),
            'local_events': events.get('local'),
            'global_events': events.get('global'),
            'is_holiday': self.api.is_holiday()
        }

class DateValidator:
    """Validate Persian date formats"""
    
    @staticmethod
    def validate_persian_date(date_string: str) -> bool:
        """Validate Persian date string format"""
        patterns = [
            r'^\d{4}/\d{2}/\d{2}$',  # 1404/09/03
            r'^\d{4}-\d{2}-\d{2}$',  # 1404-09-03
            r'^\d{4}/\d{1,2}/\d{1,2}$',  # 1404/9/3
            r'^\d{4}-\d{1,2}-\d{1,2}$'   # 1404-9-3
        ]
        
        return any(re.match(pattern, date_string) for pattern in patterns)
    
    @staticmethod
    def extract_date_components(date_string: str) -> Dict[str, str]:
        """Extract year, month, day from date string"""
        # Remove any non-digit characters except / and -
        cleaned = re.sub(r'[^\d/\-]', '', date_string)
        
        if '/' in cleaned:
            parts = cleaned.split('/')
        elif '-' in cleaned:
            parts = cleaned.split('-')
        else:
            return {}
        
        if len(parts) == 3:
            return {
                'year': parts[0],
                'month': parts[1],
                'day': parts[2]
            }
        
        return {}