#!/usr/bin/env python3
"""
Command Line Interface for Persian Date Manager
"""

import argparse
import sys
import json
from .api_client import PersianDateAPI
from .formatter import DateFormatter
from .utilities import HolidayChecker, DateConverter

def main():
    parser = argparse.ArgumentParser(
        description='مدیریت تاریخ شمسی - Persian Date Manager',
        formatter_class=argparse.RawDescriptionHelpFormatter
    )
    
    # Main commands
    parser.add_argument('--current', action='store_true', help='نمایش تاریخ و زمان فعلی')
    parser.add_argument('--full', action='store_true', help='نمایش اطلاعات کامل')
    parser.add_argument('--holiday', action='store_true', help='بررسی تعطیلی امروز')
    parser.add_argument('--progress', action='store_true', help='نمایش پیشرفت سال')
    parser.add_argument('--zodiac', action='store_true', help='نمایش حیوان سال')
    
    # Output formats
    parser.add_argument('--format', choices=['text', 'json', 'html'], default='text', help='فرمت خروجی')
    parser.add_argument('--output', help='ذخیره در فایل')
    
    args = parser.parse_args()
    
    # If no arguments provided, show current date
    if not any(vars(args).values()):
        args.current = True
    
    try:
        api = PersianDateAPI()
        formatter = DateFormatter(api)
        
        output_content = ""
        
        if args.full:
            if args.format == 'text':
                output_content = formatter.format_as_text()
            elif args.format == 'json':
                output_content = formatter.format_as_json()
            elif args.format == 'html':
                output_content = formatter.format_as_html()
                
        elif args.holiday:
            checker = HolidayChecker(api)
            status = checker.get_holiday_status()
            
            if args.format == 'json':
                output_content = json.dumps(status, ensure_ascii=False, indent=2)
            else:
                output_content = status['message']
                
        elif args.progress:
            progress = api.get_year_progress()
            if args.format == 'json':
                output_content = json.dumps(progress, ensure_ascii=False, indent=2)
            else:
                output_content = f"📊 پیشرفت سال: {progress['passed_percent']:.1f}% ({progress['passed_days']} روز گذشته)"
                
        elif args.zodiac:
            animal = api.get_zodiac_animal()
            if args.format == 'json':
                output_content = json.dumps({'zodiac_animal': animal}, ensure_ascii=False, indent=2)
            else:
                output_content = f"🐲 حیوان سال: {animal}"
                
        elif args.current:
            full_date = api.get_full_persian_date()
            time_24h = api.get_24h_time().get('full', {}).get('fa', '')
            
            if args.format == 'json':
                converter = DateConverter(api)
                all_data = converter.get_all_formats()
                output_content = json.dumps(all_data, ensure_ascii=False, indent=2)
            else:
                output_content = f"📅 {full_date} - 🕒 {time_24h}"
        
        # Output results
        if args.output:
            with open(args.output, 'w', encoding='utf-8') as f:
                f.write(output_content)
            print(f"نتایج در {args.output} ذخیره شد")
        else:
            print(output_content)
            
    except Exception as e:
        print(f"خطا: {e}", file=sys.stderr)
        sys.exit(1)

if __name__ == '__main__':
    main()