import requests
import json
from typing import Dict, Any, Optional
from datetime import datetime

class PersianDateAPI:
    """
    Client for Keybit.ir Time API with comprehensive data parsing
    """
    
    BASE_URL = "https://api.keybit.ir/time/"
    
    def __init__(self, timeout: int = 10):
        self.timeout = timeout
        self._last_response = None
    
    def get_current_time(self) -> Dict[str, Any]:
        """
        Get complete current Persian date/time information
        
        Returns:
            Dict with complete time data
        """
        try:
            response = requests.get(self.BASE_URL, timeout=self.timeout)
            response.raise_for_status()
            self._last_response = response.json()
            return self._last_response
        except requests.exceptions.RequestException as e:
            raise ConnectionError(f"Failed to fetch time data: {e}")
    
    def get_unix_timestamp(self) -> Dict[str, str]:
        """Get Unix timestamp in both Persian and English"""
        data = self.get_current_time()
        return data.get('unix', {})
    
    def get_timezone_info(self) -> Dict[str, Any]:
        """Get timezone information"""
        data = self.get_current_time()
        return data.get('timezone', {})
    
    def get_season_info(self) -> Dict[str, Any]:
        """Get current season information"""
        data = self.get_current_time()
        return data.get('season', {})
    
    def get_12h_time(self) -> Dict[str, Any]:
        """Get 12-hour format time"""
        data = self.get_current_time()
        return data.get('time12', {})
    
    def get_24h_time(self) -> Dict[str, Any]:
        """Get 24-hour format time"""
        data = self.get_current_time()
        return data.get('time24', {})
    
    def get_date_info(self) -> Dict[str, Any]:
        """Get complete date information"""
        data = self.get_current_time()
        return data.get('date', {})
    
    def get_year_info(self) -> Dict[str, Any]:
        """Get detailed year information"""
        data = self.get_current_time()
        return data.get('date', {}).get('year', {})
    
    def get_month_info(self) -> Dict[str, Any]:
        """Get month information"""
        data = self.get_current_time()
        return data.get('date', {}).get('month', {})
    
    def get_day_info(self) -> Dict[str, Any]:
        """Get detailed day information"""
        data = self.get_current_time()
        return data.get('date', {}).get('day', {})
    
    def get_weekday_info(self) -> Dict[str, Any]:
        """Get weekday information"""
        data = self.get_current_time()
        return data.get('date', {}).get('weekday', {})
    
    def get_full_persian_date(self) -> str:
        """Get full Persian date string"""
        data = self.get_current_time()
        date = data.get('date', {})
        year = date.get('year', {}).get('number', {}).get('fa', '')
        month = date.get('month', {}).get('name', '')
        day = date.get('day', {}).get('number', {}).get('fa', '')
        weekday = date.get('weekday', {}).get('name', '')
        
        return f"{weekday}، {day} {month} {year}"
    
    def get_gregorian_date(self) -> Dict[str, str]:
        """Get Gregorian date equivalent"""
        data = self.get_current_time()
        return data.get('date', {}).get('other', {}).get('gregorian', {})
    
    def get_ghamari_date(self) -> Dict[str, str]:
        """Get Ghamari (Islamic) date equivalent"""
        data = self.get_current_time()
        return data.get('date', {}).get('other', {}).get('ghamari', {})
    
    def is_holiday(self) -> bool:
        """Check if current day is a holiday"""
        data = self.get_current_time()
        events = data.get('date', {}).get('day', {}).get('events', {})
        holy_events = events.get('holy', {})
        return holy_events.get('holiday', False) if holy_events else False
    
    def get_holiday_info(self) -> Optional[Dict[str, Any]]:
        """Get holiday information if today is holiday"""
        data = self.get_current_time()
        events = data.get('date', {}).get('day', {}).get('events', {})
        holy_events = events.get('holy', {})
        return holy_events if holy_events and holy_events.get('holiday') else None
    
    def get_year_progress(self) -> Dict[str, float]:
        """Get year progress percentage"""
        data = self.get_current_time()
        year_info = data.get('date', {}).get('year', {})
        agone_percent = float(year_info.get('agone', {}).get('percent', {}).get('en', 0))
        left_percent = float(year_info.get('left', {}).get('percent', {}).get('en', 0))
        
        return {
            'passed_percent': agone_percent,
            'remaining_percent': left_percent,
            'passed_days': int(year_info.get('agone', {}).get('days', {}).get('en', 0)),
            'remaining_days': int(year_info.get('left', {}).get('days', {}).get('en', 0))
        }
    
    def get_zodiac_animal(self) -> str:
        """Get Chinese zodiac animal for current year"""
        data = self.get_current_time()
        return data.get('date', {}).get('year', {}).get('animal', '')
    
    def get_asterism(self) -> str:
        """Get current month's asterism (برج فلکی)"""
        data = self.get_current_time()
        return data.get('date', {}).get('month', {}).get('asterism', '')
    
    def is_leap_year(self) -> bool:
        """Check if current year is leap year"""
        data = self.get_current_time()
        leapyear = data.get('date', {}).get('year', {}).get('leapyear', '')
        return leapyear == 'کبیسه'
    
    def get_last_response(self) -> Optional[Dict[str, Any]]:
        """Get the last API response"""
        return self._last_response