import unittest
from unittest import mock
import pytest
import datetime
import aiohttp
import pandas.api.types as ptypes
from pydantic import ValidationError, AnyUrl, BaseModel
from typing import List
from copy import deepcopy

from pyprediktormapclient.opc_ua import OPC_UA, Variables
from pyprediktormapclient.auth_client import AUTH_CLIENT, Token

URL = "http://someserver.somedomain.com/v1/"
OPC_URL = "opc.tcp://nosuchserver.nosuchdomain.com"
username = "some@user.com"
password = "somepassword"
auth_id = "0b518533-fb09-4bb7-a51f-166d3453685e"
auth_session_id = "qlZULxcaNc6xVdXQfqPxwix5v3tuCLaO"
auth_expires_at = "2022-12-04T07:31:28.767407252Z"

list_of_ids = [
    {"Id": "SOMEID1", "Namespace": "1", "IdType": "2"},
]

list_of_write_values = [
    {
        "NodeId": {
            'Id': 'SOMEID',
            'Namespace': 1,
            'IdType': 2
        },
        "Value": {
            "Value": {
                "Type": 10,
                "Body": 1.2
            },
            "SourceTimestamp": "2022-11-03T12:00:00Z",
            "StatusCode": {
                "Code": 0,
                "Symbol": "Good"
            }
        }
    }
]

list_of_write_historical_values = [
        {
            "NodeId": {
                "Id": "SOMEID",
                "Namespace": 1,
                "IdType": 2
            },
            "PerformInsertReplace": 1,
            "UpdateValues": [
                {
                    "Value": {
                        "Type": 10,
                        "Body": 1.1
                    },
                    "SourceTimestamp": "2022-11-03T12:00:00Z",
                    "StatusCode": {
                        "Code": 0,
                        "Symbol": "Good"
                    }
                },
                {
                    "Value": {
                        "Type": 10,
                        "Body": 2.1
                    },
                    "SourceTimestamp": "2022-11-03T13:00:00Z",
                    "StatusCode": {
                        "Code": 0,
                        "Symbol": "Good"
                    }
                }
            ]
        }
    ]

list_of_write_historical_values_in_wrong_order = [
            {
                "NodeId": {
                    "Id": "SOMEID",
                    "Namespace": 4,
                    "IdType": 1
                },
                "PerformInsertReplace": 1,
                "UpdateValues": [
                    {
                        "Value": {
                            "Type": 10,
                            "Body": 1.1
                        },
                        "SourceTimestamp": "2022-11-03T14:00:00Z",
                        "StatusCode": {
                            "Code": 0,
                            "Symbol": "Good"
                        }
                    },
                    {
                        "Value": {
                            "Type": 10,
                            "Body": 2.1
                        },
                        "SourceTimestamp": "2022-11-03T13:00:00Z",
                        "StatusCode": {
                            "Code": 0,
                            "Symbol": "Good"
                        }
                    }
                ]
            }
        ]

list_of_historical_values_wrong_type_and_value = [
            {
                "NodeId": {
                    "Id": "SSO.JO-GL.321321321",
                    "Namespace": 4,
                    "IdType": 1
                },
                "PerformInsertReplace": 1,
                "UpdateValues": [
                    {
                        "Value": {
                            "Type": 10,
                            "Body": 1.1
                        },
                        "SourceTimestamp": "2022-11-03T14:00:00Z",
                        "StatusCode": {
                            "Code": 0,
                            "Symbol": "Good"
                        }
                    },
                    {
                        "Value": {
                            "Type": 1,
                            "Body": "2.1"
                        },
                        "SourceTimestamp": "2022-11-03T15:00:00Z",
                        "StatusCode": {
                            "Code": 0,
                            "Symbol": "Good"
                        }
                    }
                ]
            }
        ]

successful_live_response = [
        {
            "Success": True,
            "Values": [
                {
                    "NodeId": {"Id": "SOMEID", "Namespace": 0, "IdType": 0},
                    "Value": {"Type": 10, "Body": 1.2},
                    "ServerTimestamp": "2022-01-01T12:00:00Z",
                    "StatusCode": {"Code": 0, "Symbol": "Good"}
                },
                {
                    "NodeId": {"Id": "SOMEID2", "Namespace": 0, "IdType": 0},
                    "Value": {"Type": 11, "Body": 2.3},
                    "ServerTimestamp": "2022-01-01T12:05:00Z",
                    "StatusCode": {"Code": 1, "Symbol": "Uncertain"}
                },
            ],
        }
    ]

empty_live_response = [
    {
        "Success": True,
        "Values": [
            {
                "SourceTimestamp": "2022-09-21T13:13:38.183Z",
                "ServerTimestamp": "2022-09-21T13:13:38.183Z",
            },
            {
                "SourceTimestamp": "2023-09-21T13:13:38.183Z",
                "ServerTimestamp": "2023-09-21T13:13:38.183Z",
            },
        ],
    }
]

successful_historical_result = {
    "Success": True,
    "ErrorMessage": "",
    "ErrorCode": 0,
    "ServerNamespaces": ["string"],
    "HistoryReadResults": [
        {
            "NodeId": {
                "IdType": 2,
                "Id": "SOMEID",
                "Namespace": 1,
            },
            "StatusCode": {"Code": 0, "Symbol": "Good"},
            "DataValues": [
                {
                    "Value": {"Type": 11, "Body": 34.28500000000003},
                    "StatusCode": {"Code": 1, "Symbol": "Good"},
                    "SourceTimestamp": "2022-09-13T13:39:51Z",
                },
                {
                    "Value": {"Type": 11, "Body": 6.441666666666666},
                    "StatusCode": {"Code": 1, "Symbol": "Good"},
                    "SourceTimestamp": "2022-09-13T14:39:51Z",
                },
            ],
        },
{
            "NodeId": {
                "IdType": 2,
                "Id": "SOMEID2",
                "Namespace": 1,
            },
            "StatusCode": {"Code": 0, "Symbol": "Good"},
            "DataValues": [
                {
                    "Value": {"Type": 11, "Body": 34.28500000000003},
                    "StatusCode": {"Code": 1, "Symbol": "Good"},
                    "SourceTimestamp": "2022-09-13T13:39:51Z",
                },
                {
                    "Value": {"Type": 11, "Body": 6.441666666666666},
                    "StatusCode": {"Code": 1, "Symbol": "Good"},
                    "SourceTimestamp": "2022-09-13T14:39:51Z",
                },
            ],
        }
    ]
}

successful_raw_historical_result = {
    "Success": True,
    "ErrorMessage": "",
    "ErrorCode": 0,
    "ServerNamespaces": ["string"],
    "HistoryReadResults": [
        {
            "NodeId": {
                "IdType": 2,
                "Id": "SOMEID",
                "Namespace": 1,
            },
            "StatusCode": {"Code": 0, "Symbol": "Good"},
            "DataValues": [
                {
                    "Value": {"Type": 11, "Body": 34.28500000000003},
                    "SourceTimestamp": "2022-09-13T13:39:51Z",
                },
                {
                    "Value": {"Type": 11, "Body": 35.12345678901234},
                    "SourceTimestamp": "2022-09-13T13:40:51Z",
                },
                {
                    "Value": {"Type": 11, "Body": 33.98765432109876},
                    "SourceTimestamp": "2022-09-13T13:41:51Z",
                },
            ],
        },
        {
            "NodeId": {
                "IdType": 2,
                "Id": "SOMEID2",
                "Namespace": 1,
            },
            "StatusCode": {"Code": 0, "Symbol": "Good"},
            "DataValues": [
                {
                    "Value": {"Type": 11, "Body": 6.441666666666666},
                    "SourceTimestamp": "2022-09-13T13:39:51Z",
                },
                {
                    "Value": {"Type": 11, "Body": 6.523456789012345},
                    "SourceTimestamp": "2022-09-13T13:40:51Z",
                },
                {
                    "Value": {"Type": 11, "Body": 6.345678901234567},
                    "SourceTimestamp": "2022-09-13T13:41:51Z",
                },
            ],
        }
    ]
}

successful_write_live_response = {
    "Success": True,
    "ErrorMessage": "string",
    "ErrorCode": 0,
    "ServerNamespaces": [
      "string"
    ],
    "StatusCodes": [
      {
        "Code": 0,
        "Symbol": "Good"
      }
    ]
  }
  

empty_write_live_response = {
    "Success": True,
    "ErrorMessage": "string",
    "ErrorCode": 0,
    "ServerNamespaces": [
      "string"
    ],
    "StatusCodes": [
      {

      }
    ]
  }

successful_write_historical_response = {
  "Success": True,
  "ErrorMessage": "string",
  "ErrorCode": 0,
  "ServerNamespaces": [
    "string"
  ],
  "HistoryUpdateResults": [
    {
    }
  ]
}

unsuccessful_write_historical_response = {
  "Success": True,
  "ErrorMessage": "string",
  "ErrorCode": 0,
  "ServerNamespaces": [
    "string"
  ]
}

successfull_write_historical_response_with_errors = {
  "Success": True,
  "ErrorMessage": "string",
  "ErrorCode": 0,
  "ServerNamespaces": [
    "string"
  ],
  "HistoryUpdateResults": [
      {
        "StatusCode": {
            'Code': 2158690304,
            'Symbol': 'BadInvalidArgument'
        }
      }
  ]
}

class SubValue(BaseModel):
    Type: int
    Body: float

class StatusCode(BaseModel):
    Code: int
    Symbol: str

class Value(BaseModel):
    Value: SubValue
    SourceTimestamp: str
    StatusCode: StatusCode

class Variables(BaseModel):
    Id: str
    Namespace: int
    IdType: int

class WriteHistoricalVariables(BaseModel):
    NodeId: Variables
    PerformInsertReplace: int
    UpdateValues: List[Value]

class MockResponse:
    def __init__(self, json_data, status_code):
        self.json_data = json_data
        self.status_code = status_code
        self.raise_for_status = mock.Mock(return_value=False)
        self.headers = {'Content-Type': 'application/json'}

    def json(self):
        return self.json_data


# This method will be used by the mock to replace requests
def successful_mocked_requests(*args, **kwargs):
    status_code = 200 if args[0] == f"{URL}values/get" else 404
    json_data = successful_live_response  
    response = MockResponse(json_data=json_data, status_code=status_code)
    response.json_data = json_data
    return response

def empty_values_mocked_requests(*args, **kwargs):
    if args[0] == f"{URL}values/get":
        return MockResponse(empty_live_response, 200)

    return MockResponse(None, 404)

def successful_write_mocked_requests(*args, **kwargs):
    if args[0] == f"{URL}values/set":
        return MockResponse(successful_write_live_response, 200)

    return MockResponse(None, 404)


def empty_write_values_mocked_requests(*args, **kwargs):
    if args[0] == f"{URL}values/set":
        return MockResponse(empty_write_live_response, 200)

    return MockResponse(None, 404)

def no_write_mocked_requests(*args, **kwargs):
    if args[0] == f"{URL}values/set":
        # Set Success to False
        response = ""
        return MockResponse(response, 200)

    return MockResponse(None, 404)


def unsuccessful_write_mocked_requests(*args, **kwargs):
    if args[0] == f"{URL}values/set":
        # Set Success to False
        unsuc = deepcopy(successful_write_live_response)
        unsuc["Success"] = False
        return MockResponse(unsuc, 200)

    return MockResponse(None, 404)


def no_status_code_write_mocked_requests(*args, **kwargs):
    if args[0] == f"{URL}values/set":
        # Set Success to False
        nostats = deepcopy(successful_write_live_response)
        nostats["StatusCodes"][0].pop("Code")
        return MockResponse(nostats, 200)

    return MockResponse(None, 404)


def empty_write_mocked_requests(*args, **kwargs):
    if args[0] == f"{URL}values/set":
        # Remove values from the dict
        empty = deepcopy(successful_write_live_response)
        empty.pop("StatusCodes")
        return MockResponse(empty, 200)

    return MockResponse(None, 404)


def successful_write_historical_mocked_requests(*args, **kwargs):
    if args[0] == f"{URL}values/historicalwrite":
        return MockResponse(successful_write_historical_response, 200)

    return MockResponse(None, 404)


def no_write_mocked_historical_requests(*args, **kwargs):
    if args[0] == f"{URL}values/historicalwrite":
        # Set Success to False
        response = ""
        return MockResponse(response, 200)

    return MockResponse(None, 404)

def successful_write_historical_with_errors_mocked_requests(*args, **kwargs):
    if args[0] == f"{URL}values/historicalwrite":
        suce = deepcopy(successfull_write_historical_response_with_errors)
        return MockResponse(suce, 200)

    return MockResponse(None, 404)

def unsuccessful_write_historical_mocked_requests(*args, **kwargs):
    if args[0] == f"{URL}values/historicalwrite":
        # Set Success to False
        unsuc = deepcopy(successful_write_historical_response)
        unsuc["Success"] = False
        return MockResponse(unsuc, 200)

    return MockResponse(None, 404)


def no_status_code_write_historical_mocked_requests(*args, **kwargs):
    if args[0] == f"{URL}values/historicalwrite":
        # Set Success to False
        nostats = deepcopy(successful_write_historical_response)
        nostats["StatusCodes"][0].pop("Code")
        return MockResponse(nostats, 200)

    return MockResponse(None, 404)


def empty_write_historical_mocked_requests(*args, **kwargs):
    if args[0] == f"{URL}values/historicalwrite":
        # Remove values from the dict
        empty = deepcopy(successful_write_historical_response)
        empty.pop("HistoryUpdateResults")
        return MockResponse(empty, 200)

    return MockResponse(None, 404)


def no_mocked_requests(*args, **kwargs):
    if args[0] == f"{URL}values/get":
        # Set Success to False
        response = ""
        return MockResponse(response, 200)

    return MockResponse(None, 404)


def unsuccessful_mocked_requests(*args, **kwargs):
    if args[0] == f"{URL}values/get":
        # Set Success to False
        unsuc = deepcopy(successful_live_response[0])
        unsuc["Success"] = False
        return MockResponse([unsuc], 200)

    return MockResponse(None, 404)


def no_status_code_mocked_requests(*args, **kwargs):
    if args[0] == f"{URL}values/get":
        # Set Success to False
        nostats = deepcopy(successful_live_response[0])
        nostats["Values"][0].pop("StatusCode")
        return MockResponse([nostats], 200)

    return MockResponse(None, 404)


def empty_mocked_requests(*args, **kwargs):
    if args[0] == f"{URL}values/get":
        # Remove values from the dict
        empty = deepcopy(successful_live_response[0])
        empty.pop("Values")
        return MockResponse([empty], 200)

    return MockResponse(None, 404)

class AnyUrlModel(BaseModel):
    url: AnyUrl

class TestOPCUA(unittest.TestCase):
    def test_malformed_rest_url(self):
        with pytest.raises(ValidationError):
            AnyUrlModel(rest_url="not_an_url", opcua_url=OPC_URL)

    def test_malformed_opcua_url(self):
        with pytest.raises(ValidationError):
            AnyUrlModel(rest_url=URL, opcua_url="not_an_url")

    def test_namespaces(self):
        opc = OPC_UA(rest_url=URL, opcua_url=OPC_URL, namespaces=["1", "2"])
        assert "ClientNamespaces" in opc.body
        opc = OPC_UA(rest_url=URL, opcua_url=OPC_URL)
        assert "ClientNamespaces" not in opc.body

    def test_get_value_type(self):
        opc = OPC_UA(rest_url=URL, opcua_url=OPC_URL)
        result_none = opc._get_value_type(100000)
        assert result_none["id"] == None
        result = opc._get_value_type(1)
        assert "id" in result
        assert "type" in result
        assert "description" in result
        assert result["type"] == "Boolean"

    def test_get_variable_list_as_list(self):
        opc = OPC_UA(rest_url=URL, opcua_url=OPC_URL)
        var = Variables(Id="ID", Namespace=1, IdType=2)
        list = [var]
        result = opc._get_variable_list_as_list(list)
        assert "Id" in result[0]
        assert result[0]["Id"] == "ID"

    def test_check_auth_client_is_none(self):
        opc = OPC_UA(rest_url=URL, opcua_url=OPC_URL, auth_client=None)
        with pytest.raises(Exception):
            opc.check_auth_client()


    @mock.patch("requests.post", side_effect=successful_mocked_requests)
    def test_get_live_values_successful(self, mock_get):
        tsdata = OPC_UA(rest_url=URL, opcua_url=OPC_URL)
        result = tsdata.get_values(list_of_ids)
        if list_of_ids:
            for num, row in enumerate(list_of_ids):
                assert result[num]["Id"] == list_of_ids[num]["Id"]
                assert (
                    result[num]["Timestamp"]
                    == successful_live_response[0]["Values"][num]["ServerTimestamp"]
                )
                assert (
                    result[num]["Value"]
                    == successful_live_response[0]["Values"][num]["Value"]["Body"]
                )
                assert (
                    result[num]["ValueType"]
                    == tsdata._get_value_type(
                        successful_live_response[0]["Values"][num]["Value"]["Type"]
                    )["type"]
                )
                assert (
                    result[num]["StatusCode"]
                    == successful_live_response[0]["Values"][num]["StatusCode"]["Code"]
                )
                assert (
                    result[num]["StatusSymbol"]
                    == successful_live_response[0]["Values"][num]["StatusCode"]["Symbol"]
                )


    @mock.patch("requests.post", side_effect=successful_mocked_requests)
    def test_get_live_values_successful_with_auth(self, mock_get):
        auth_client = AUTH_CLIENT(rest_url=URL, username=username, password=password)
        auth_client.token = Token(session_token=auth_session_id, expires_at=auth_expires_at)
        tsdata = OPC_UA(rest_url=URL, opcua_url=OPC_URL, auth_client=auth_client)
        result = tsdata.get_values(list_of_ids)
        for num, row in enumerate(list_of_ids):
            assert result[num]["Id"] == list_of_ids[num]["Id"]
            assert (
                result[num]["Timestamp"]
                == successful_live_response[0]["Values"][num]["ServerTimestamp"]
            )
            assert (
                result[num]["Value"]
                == successful_live_response[0]["Values"][num]["Value"]["Body"]
            )
            assert (
                result[num]["ValueType"]
                == tsdata._get_value_type(
                    successful_live_response[0]["Values"][num]["Value"]["Type"]
                )["type"]
            )
            assert (
                result[num]["StatusCode"]
                == successful_live_response[0]["Values"][num]["StatusCode"]["Code"]
            )
            assert (
                result[num]["StatusSymbol"]
                == successful_live_response[0]["Values"][num]["StatusCode"]["Symbol"]
            )
            

    @mock.patch("requests.post", side_effect=empty_values_mocked_requests)
    def test_get_live_values_with_missing_value_and_statuscode(self, mock_get):
        tsdata = OPC_UA(rest_url=URL, opcua_url=OPC_URL)
        result = tsdata.get_values(list_of_ids)
        for num, row in enumerate(list_of_ids):
            if num < len(result):  
                assert result[num]["Id"] == list_of_ids[num]["Id"]
                assert (
                    result[num]["Timestamp"]
                    == empty_live_response[0]["Values"][num]["ServerTimestamp"]
                )
                assert result[num]["Value"] is None
                assert result[num]["ValueType"] is None
                assert result[num]["StatusCode"] is None
                assert result[num]["StatusSymbol"] is None

    @mock.patch("requests.post", side_effect=no_mocked_requests)
    def test_get_live_values_no_response(self, mock_get):
        tsdata = OPC_UA(rest_url=URL, opcua_url=OPC_URL)
        result = tsdata.get_values(list_of_ids)
        assert result[0]["Timestamp"] == None

    @mock.patch("requests.post", side_effect=unsuccessful_mocked_requests)
    def test_get_live_values_unsuccessful(self, mock_post):
        tsdata = OPC_UA(rest_url=URL, opcua_url=OPC_URL)
        with pytest.raises(RuntimeError):
            tsdata.get_values(list_of_ids)

    @mock.patch("requests.post", side_effect=empty_mocked_requests)
    def test_get_live_values_empty(self, mock_get):
        tsdata = OPC_UA(rest_url=URL, opcua_url=OPC_URL)
        result = tsdata.get_values(list_of_ids)
        assert result[0]["Timestamp"] == None

    @mock.patch("requests.post", side_effect=no_status_code_mocked_requests)
    def test_get_live_values_no_status_code(self, mock_get):
        tsdata = OPC_UA(rest_url=URL, opcua_url=OPC_URL)
        result = tsdata.get_values(list_of_ids)
        assert result[0]["StatusCode"] == None

    @mock.patch("requests.post", side_effect=successful_write_mocked_requests)
    def test_write_live_values_successful(self, mock_get):
        tsdata = OPC_UA(rest_url=URL, opcua_url=OPC_URL)
        result = tsdata.write_values(list_of_write_values)
        for num, row in enumerate(list_of_write_values):
            assert (
                result[num]["Value"]["StatusCode"]["Code"]
                == successful_write_live_response["StatusCodes"][num]["Code"]
            )
            assert (
                result[num]["Value"]["StatusCode"]["Symbol"]
                == successful_write_live_response["StatusCodes"][num]["Symbol"]
            )
            assert result[num]["WriteSuccess"] is True

    @mock.patch("requests.post", side_effect=empty_write_values_mocked_requests)
    def test_write_live_values_with_missing_value_and_statuscode(self, mock_get):
        tsdata = OPC_UA(rest_url=URL, opcua_url=OPC_URL) 
        result = tsdata.write_values(list_of_write_values)
        for num, row in enumerate(list_of_write_values):
            assert result[num]["WriteSuccess"] is False

    @mock.patch("requests.post", side_effect=no_write_mocked_requests)
    def test_get_write_live_values_no_response(self, mock_get):
        tsdata = OPC_UA(rest_url=URL, opcua_url=OPC_URL)
        result = tsdata.write_values(list_of_write_values)
        assert result is None

    @mock.patch("requests.post", side_effect=unsuccessful_write_mocked_requests)
    def test_get_write_live_values_unsuccessful(self, mock_get):
        tsdata = OPC_UA(rest_url=URL, opcua_url=OPC_URL)
        with pytest.raises(RuntimeError):
           result = tsdata.write_values(list_of_write_values) 

    @mock.patch("requests.post", side_effect=empty_write_mocked_requests)
    def test_get_write_live_values_empty(self, mock_get):
        tsdata = OPC_UA(rest_url=URL, opcua_url=OPC_URL)
        with pytest.raises(ValueError):
            result = tsdata.write_values(list_of_write_values)

    @mock.patch("requests.post", side_effect=successful_write_historical_mocked_requests)
    def test_write_historical_values_successful(self, mock_get):
        tsdata = OPC_UA(rest_url=URL, opcua_url=OPC_URL)
        converted_data = [WriteHistoricalVariables(**item) for item in list_of_write_historical_values]
        result = tsdata.write_historical_values(converted_data)
        for num, row in enumerate(list_of_write_values):
            assert result[0]["WriteSuccess"] == True

    @mock.patch("requests.post", side_effect=successful_write_historical_mocked_requests)
    def test_write_wrong_order_historical_values_successful(self, mock_get):
        tsdata = OPC_UA(rest_url=URL, opcua_url=OPC_URL)
        converted_data = [WriteHistoricalVariables(**item) for item in list_of_write_historical_values_in_wrong_order]
        with pytest.raises(ValueError):
            result = tsdata.write_historical_values(converted_data)

    @mock.patch("requests.post", side_effect=empty_write_historical_mocked_requests)
    def test_write_historical_values_with_missing_value_and_statuscode(self, mock_get):
        tsdata = OPC_UA(rest_url=URL, opcua_url=OPC_URL)
        converted_data = [WriteHistoricalVariables(**item) for item in list_of_write_historical_values]
        with pytest.raises(ValueError):
            result = tsdata.write_historical_values(converted_data)

    @mock.patch("requests.post", side_effect=no_write_mocked_historical_requests)
    def test_get_write_historical_values_no_response(self, mock_get):
        tsdata = OPC_UA(rest_url=URL, opcua_url=OPC_URL)
        converted_data = [WriteHistoricalVariables(**item) for item in list_of_write_historical_values]
        result = tsdata.write_historical_values(converted_data)
        assert result is None

    @mock.patch("requests.post", side_effect=unsuccessful_write_historical_mocked_requests)
    def test_get_write_historical_values_unsuccessful(self, mock_get):
        tsdata = OPC_UA(rest_url=URL, opcua_url=OPC_URL)
        converted_data = [WriteHistoricalVariables(**item) for item in list_of_write_historical_values]
        with pytest.raises(RuntimeError):
            result = tsdata.write_historical_values(converted_data) 

    @mock.patch("requests.post", side_effect=successful_write_historical_with_errors_mocked_requests)
    def test_get_write_historical_values_successful_with_error_codes(self, mock_get):
        tsdata = OPC_UA(rest_url=URL, opcua_url=OPC_URL)
        converted_data = [WriteHistoricalVariables(**item) for item in list_of_historical_values_wrong_type_and_value]
        result = tsdata.write_historical_values(converted_data)
        assert result[0]["WriteError"]["Code"] == successfull_write_historical_response_with_errors["HistoryUpdateResults"][0]["StatusCode"]["Code"]
        assert result[0]["WriteError"]["Symbol"] == successfull_write_historical_response_with_errors["HistoryUpdateResults"][0]["StatusCode"]["Symbol"]

class AsyncMockResponse:
    def __init__(self, json_data, status_code):
        self.json_data = json_data
        self.status = status_code
        self.headers = {'Content-Type': 'application/json'}

    async def __aenter__(self):
        return self

    async def __aexit__(self, exc_type, exc, tb):
        pass

    async def json(self):
        return self.json_data
    
    async def raise_for_status(self):
        if self.status >= 400:
            raise aiohttp.ClientResponseError(
                request_info=aiohttp.RequestInfo(
                    url=URL,
                    method="POST",
                    headers={},
                    real_url=OPC_URL
                ),
                history=(),
                status=self.status,
                message="Mocked error",
                headers=self.headers
            )

def unsuccessful_async_mock_response(*args, **kwargs):
    return AsyncMockResponse(
        json_data=None,
        status_code=400
    )

async def make_historical_request():
    tsdata = OPC_UA(rest_url=URL, opcua_url=OPC_URL)
    return await tsdata.get_historical_aggregated_values_asyn(
                start_time=(datetime.datetime.now() - datetime.timedelta(30)),
                end_time=(datetime.datetime.now() - datetime.timedelta(29)),
                pro_interval=3600000,
                agg_name="Average",
                variable_list=list_of_ids,
            )

async def make_raw_historical_request():
    tsdata = OPC_UA(rest_url=URL, opcua_url=OPC_URL)
    return await tsdata.get_historical_raw_values_asyn(
        start_time=(datetime.datetime.now() - datetime.timedelta(30)),
        end_time=(datetime.datetime.now() - datetime.timedelta(29)),
        variable_list=list_of_ids,
    )

@pytest.mark.asyncio
class TestAsyncOPCUA:

    @mock.patch("aiohttp.ClientSession.post")
    async def test_historical_values_success(self, mock_post):
        mock_post.return_value = AsyncMockResponse(
            json_data=successful_historical_result,
            status_code=200
        )
        result = await make_historical_request()
        cols_to_check = ["Value"]
        assert all(ptypes.is_numeric_dtype(result[col]) for col in cols_to_check)
        assert result['Value'].tolist() == [34.28500000000003, 6.441666666666666, 34.28500000000003, 6.441666666666666]
        assert result['ValueType'].tolist() == ["Double", "Double", "Double", "Double"]

    @mock.patch("aiohttp.ClientSession.post")
    async def test_historical_values_no_dict(self, mock_post):
        with pytest.raises(RuntimeError):
            await make_historical_request()

    @mock.patch("aiohttp.ClientSession.post")
    async def test_historical_values_unsuccess(self, mock_post):
        mock_post.return_value = unsuccessful_async_mock_response()
        with pytest.raises(RuntimeError):
            await make_historical_request()

    @mock.patch("aiohttp.ClientSession.post")
    async def test_historical_values_no_hist(self, mock_post):
        with pytest.raises(RuntimeError):
            await make_historical_request()

    @mock.patch("aiohttp.ClientSession.post")
    async def test_raw_historical_values_success(self, mock_post):
        mock_post.return_value = AsyncMockResponse(
            json_data=successful_raw_historical_result,
            status_code=200
        )
        result = await make_raw_historical_request()
        cols_to_check = ["Value"]
        assert all(ptypes.is_numeric_dtype(result[col]) for col in cols_to_check)

    @mock.patch("aiohttp.ClientSession.post")
    async def test_raw_historical_values_no_dict(self, mock_post):
        with pytest.raises(RuntimeError):
            await make_raw_historical_request()

    @mock.patch("aiohttp.ClientSession.post")
    async def test_raw_historical_values_unsuccess(self, mock_post):
        with pytest.raises(RuntimeError):
            await make_raw_historical_request()

    @mock.patch("aiohttp.ClientSession.post")
    async def test_raw_historical_values_no_hist(self, mock_post):
        with pytest.raises(RuntimeError):
            await make_raw_historical_request()

if __name__ == "__main__":
    unittest.main()
