// Copyright 2016 ETH Zurich and University of Bologna.
// Copyright and related rights are licensed under the Solderpad Hardware
// License, Version 0.51 (the "License"); you may not use this file except in
// compliance with the License.  You may obtain a copy of the License at
// http://solderpad.org/licenses/SHL-0.51. Unless required by applicable law
// or agreed to in writing, software, hardware and materials distributed under
// this License is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
// CONDITIONS OF ANY KIND, either express or implied. See the License for the
// specific language governing permissions and limitations under the License.

/**
 * True Dual-Port BRAM Array for Xilinx 7 Series Devices
 *
 * This module contains a two-dimensional array of Xilinx' 7 Series True Dual-Port BRAM cells.  The
 * array is
 *    NUM_PAR_BRAMS = ceil(DATA_BITW/32)
 * BRAMs wide and
 *    NUM_SER_BRAMS = ceil(NUM_ENTRIES/1024)
 * BRAMs deep, as each BRAM is 32 bit wide and 1024 entries deep.
 *
 * This module is addressed byte-wise!  Be careful when addressing a BRAM array that has more than
 * one BRAM in parallel (i.e., when `DATA_BITW > 32`): the
 *    WORD_OFFSET = ceil(log2(DATA_BITW/8))
 * least-significant bits of the address are used to address the bytes within a word.  The reason
 * for byte-wise addressing is to enable instances with a width that is a power of two (and >= 32)
 * to directly connect to a Xilinx AXI 4 BRAM Controller.  If you want to connect a BRAM that does
 * not fulfill the direct connection criterion, use the Data Width Converter `BramDwc`.
 *
 * Even though addressing is byte-wise, accesses that are not aligned to words are not supported.
 *
 * Both ports can be operated independently and asynchronously; the behavior on access collisions is
 * specified in the Xilinx Block Memory Generator Product Guide (PG058).
 *
 * Current Maintainers:
 * - Andreas Kurth  <akurth@iis.ee.ethz.ch>
 * - Pirmin Vogel   <vogelpi@iis.ee.ethz.ch>
 */

import CfMath::ceil_div, CfMath::log2;

module TdpBramArray

  // Parameters {{{
  #(

    // Width (in bits) of the read/write data ports.  Should be a multiple of 32 (for optimal
    // resource usage) and is upper-bound by the available memory resources.
    parameter DATA_BITW   =   96,

    // Number of entries (each entry is `DATA_BITW` bits wide) in the BRAM array.  Should be
    // a multiple of 1024 (for optimal resource usage) and is upper-bound by the available memory
    // resources.
    parameter NUM_ENTRIES = 8192
  )
  // }}}

  // Ports {{{
  (
    BramPort.Slave  A_PS,
    BramPort.Slave  B_PS
  );
  // }}}

  // Module-Wide Constants {{{

  // Properties of the employed BRAM cells
  localparam integer BRAM_BITW      = 32;
  localparam integer BRAM_BYTEW     = BRAM_BITW / 8;
  localparam integer NUM_BRAM_WORDS = 1024;

  // Properties of the resulting memory array
  localparam integer NUM_PAR_BRAMS  = ceil_div(DATA_BITW, BRAM_BITW);
  localparam integer NUM_SER_BRAMS  = ceil_div(NUM_ENTRIES, NUM_BRAM_WORDS);
  localparam integer ARR_BITW       = BRAM_BITW       * NUM_PAR_BRAMS;
  localparam integer ARR_BYTEW      = BRAM_BYTEW      * NUM_PAR_BRAMS;
  localparam integer NUM_ARR_WORDS  = NUM_BRAM_WORDS  * NUM_SER_BRAMS;

  // Offset (in bits) of words in the external addresses
  localparam integer ADDR_WORD_BITO = log2(ARR_BYTEW);

  localparam integer WORD_ADDR_BITW = log2(NUM_ARR_WORDS);
  localparam integer SER_IDX_BITW   = log2(NUM_SER_BRAMS);
  localparam integer WORD_IDX_BITW  = log2(NUM_BRAM_WORDS);

  // }}}

  // Signal Declarations {{{

  // Output signals after multiplexing of parallel BRAM cells
  logic [NUM_SER_BRAMS-1:0] [ARR_BITW       -1:0]   ARd_D,              BRd_D;

  // Word part of external address
  logic                     [WORD_ADDR_BITW -1:0]   WordAddrA_S,        WordAddrB_S;

  // Serial index of BRAM cell in array
  logic                     [SER_IDX_BITW   -1:0]   SerIdxA_S,          SerIdxB_S;
  logic                                             SerIdxAOverflow_S,  SerIdxBOverflow_S;

  // Word index in BRAM cell
  logic                     [WORD_IDX_BITW  -1:0]   WordIdxA_S,         WordIdxB_S;

  // }}}

  // Resolve (Linear) Address to Serial (BRAM), Word Index and Address of RAMs {{{

  assign WordAddrA_S  = A_PS.Addr_S[ADDR_WORD_BITO+(WORD_ADDR_BITW-1):ADDR_WORD_BITO];
  assign WordAddrB_S  = B_PS.Addr_S[ADDR_WORD_BITO+(WORD_ADDR_BITW-1):ADDR_WORD_BITO];

  always_comb begin
    SerIdxAOverflow_S = 0;
    SerIdxA_S         = WordAddrA_S / NUM_BRAM_WORDS;
    if (SerIdxA_S >= NUM_SER_BRAMS) begin
      SerIdxAOverflow_S = 1;
      SerIdxA_S         = 0;
    end
  end
  always_comb begin
    SerIdxBOverflow_S = 0;
    SerIdxB_S         = WordAddrB_S / NUM_BRAM_WORDS;
    if (SerIdxB_S >= NUM_SER_BRAMS) begin
      SerIdxBOverflow_S = 1;
      SerIdxB_S         = 0;
    end
  end

  assign WordIdxA_S = WordAddrA_S % NUM_BRAM_WORDS;
  assign WordIdxB_S = WordAddrB_S % NUM_BRAM_WORDS;

  always @ (posedge A_PS.Clk_C) begin
    if (A_PS.Rst_R == 0) begin
      assert (~SerIdxAOverflow_S)           else $warning("Serial index on port A out of bounds!");
      assert (WordIdxA_S < NUM_BRAM_WORDS)  else $error("Word index on port A out of bounds!");
    end
  end
  always @ (posedge B_PS.Clk_C) begin
    if (B_PS.Rst_R == 0) begin
      assert (~SerIdxBOverflow_S)           else $warning("Serial index on port B out of bounds!");
      assert (WordIdxB_S  < NUM_BRAM_WORDS) else $error("Word index on port B out of bounds!");
    end
  end

  // }}}

  // BRAM Instantiation, Signal Resolution, and Port Assignment {{{
  genvar s, p;
  for (s = 0; s < NUM_SER_BRAMS; s++) begin
    for (p = 0; p < NUM_PAR_BRAMS; p++) begin

      // Instance-Specific Constants {{{
      localparam integer WORD_BIT_LOW   = BRAM_BITW *p;
      localparam integer WORD_BIT_HIGH  = WORD_BIT_LOW  + (BRAM_BITW -1);
      localparam integer WORD_BYTE_LOW  = BRAM_BYTEW*p;
      localparam integer WORD_BYTE_HIGH = WORD_BYTE_LOW + (BRAM_BYTEW-1);
      // }}}

      // Write-Enable Resolution {{{
      logic [BRAM_BYTEW-1:0] WrEnA_S, WrEnB_S;
      always_comb begin
        WrEnA_S = '0;
        WrEnB_S = '0;
        if (SerIdxA_S == s && ~SerIdxAOverflow_S) begin
          WrEnA_S = A_PS.WrEn_S[WORD_BYTE_HIGH:WORD_BYTE_LOW];
        end
        if (SerIdxB_S == s && ~SerIdxBOverflow_S) begin
          WrEnB_S = B_PS.WrEn_S[WORD_BYTE_HIGH:WORD_BYTE_LOW];
        end
      end
      // }}}

      // BRAM_TDP_MACRO Parameters and Initial Values {{{
      // BRAM_TDP_MACRO: True Dual Port RAM
      //                 Virtex-7
      // Xilinx HDL Language Template, version 2016.1
      BRAM_TDP_MACRO #(
        // Target BRAM: {"18Kb", "36Kb"}
        .BRAM_SIZE("36Kb"),
        // Target Device: {"7SERIES"}
        .DEVICE("7SERIES"),
        // Optional Port A/B Output Registers: {0, 1}
        .DOA_REG(0),
        .DOB_REG(0),
        // Initial Value of Port A/B Output
        .INIT_A(36'h000000000),
        .INIT_B(36'h000000000),
        // RAM Initialization File
        .INIT_FILE("NONE"),
        // Width of Port A/B Output: 1..36 (19..36 only if BRAM_SIZE="36Kb")
        .READ_WIDTH_A(BRAM_BITW),
        .READ_WIDTH_B(BRAM_BITW),
        // Enable Collision Check in Simulation: {"ALL", "WARNING_ONLY", "GENERATE_X_ONLY", "NONE"}
        .SIM_COLLISION_CHECK ("ALL"),
        // Set/Reset Value of Port A/B Output
        .SRVAL_A(36'h00000000),
        .SRVAL_B(36'h00000000),
        // Write Mode of Port A/B {"WRITE_FIRST", "READ_FIRST", "NO_CHANGE"}
        .WRITE_MODE_A("WRITE_FIRST"),
        .WRITE_MODE_B("WRITE_FIRST"),
        // Width of Port A/B Input: 1..36 (19..36 only if BRAM_SIZE="36Kb")
        .WRITE_WIDTH_A(BRAM_BITW),
        .WRITE_WIDTH_B(BRAM_BITW),

        // Initialization of Data Bits in Lower 16 Kibit {{{
        .INIT_00(256'h0000000000000000000000000000000000000000000000000000000000000000),
        .INIT_01(256'h0000000000000000000000000000000000000000000000000000000000000000),
        .INIT_02(256'h0000000000000000000000000000000000000000000000000000000000000000),
        .INIT_03(256'h0000000000000000000000000000000000000000000000000000000000000000),
        .INIT_04(256'h0000000000000000000000000000000000000000000000000000000000000000),
        .INIT_05(256'h0000000000000000000000000000000000000000000000000000000000000000),
        .INIT_06(256'h0000000000000000000000000000000000000000000000000000000000000000),
        .INIT_07(256'h0000000000000000000000000000000000000000000000000000000000000000),
        .INIT_08(256'h0000000000000000000000000000000000000000000000000000000000000000),
        .INIT_09(256'h0000000000000000000000000000000000000000000000000000000000000000),
        .INIT_0A(256'h0000000000000000000000000000000000000000000000000000000000000000),
        .INIT_0B(256'h0000000000000000000000000000000000000000000000000000000000000000),
        .INIT_0C(256'h0000000000000000000000000000000000000000000000000000000000000000),
        .INIT_0D(256'h0000000000000000000000000000000000000000000000000000000000000000),
        .INIT_0E(256'h0000000000000000000000000000000000000000000000000000000000000000),
        .INIT_0F(256'h0000000000000000000000000000000000000000000000000000000000000000),
        .INIT_10(256'h0000000000000000000000000000000000000000000000000000000000000000),
        .INIT_11(256'h0000000000000000000000000000000000000000000000000000000000000000),
        .INIT_12(256'h0000000000000000000000000000000000000000000000000000000000000000),
        .INIT_13(256'h0000000000000000000000000000000000000000000000000000000000000000),
        .INIT_14(256'h0000000000000000000000000000000000000000000000000000000000000000),
        .INIT_15(256'h0000000000000000000000000000000000000000000000000000000000000000),
        .INIT_16(256'h0000000000000000000000000000000000000000000000000000000000000000),
        .INIT_17(256'h0000000000000000000000000000000000000000000000000000000000000000),
        .INIT_18(256'h0000000000000000000000000000000000000000000000000000000000000000),
        .INIT_19(256'h0000000000000000000000000000000000000000000000000000000000000000),
        .INIT_1A(256'h0000000000000000000000000000000000000000000000000000000000000000),
        .INIT_1B(256'h0000000000000000000000000000000000000000000000000000000000000000),
        .INIT_1C(256'h0000000000000000000000000000000000000000000000000000000000000000),
        .INIT_1D(256'h0000000000000000000000000000000000000000000000000000000000000000),
        .INIT_1E(256'h0000000000000000000000000000000000000000000000000000000000000000),
        .INIT_1F(256'h0000000000000000000000000000000000000000000000000000000000000000),
        .INIT_20(256'h0000000000000000000000000000000000000000000000000000000000000000),
        .INIT_21(256'h0000000000000000000000000000000000000000000000000000000000000000),
        .INIT_22(256'h0000000000000000000000000000000000000000000000000000000000000000),
        .INIT_23(256'h0000000000000000000000000000000000000000000000000000000000000000),
        .INIT_24(256'h0000000000000000000000000000000000000000000000000000000000000000),
        .INIT_25(256'h0000000000000000000000000000000000000000000000000000000000000000),
        .INIT_26(256'h0000000000000000000000000000000000000000000000000000000000000000),
        .INIT_27(256'h0000000000000000000000000000000000000000000000000000000000000000),
        .INIT_28(256'h0000000000000000000000000000000000000000000000000000000000000000),
        .INIT_29(256'h0000000000000000000000000000000000000000000000000000000000000000),
        .INIT_2A(256'h0000000000000000000000000000000000000000000000000000000000000000),
        .INIT_2B(256'h0000000000000000000000000000000000000000000000000000000000000000),
        .INIT_2C(256'h0000000000000000000000000000000000000000000000000000000000000000),
        .INIT_2D(256'h0000000000000000000000000000000000000000000000000000000000000000),
        .INIT_2E(256'h0000000000000000000000000000000000000000000000000000000000000000),
        .INIT_2F(256'h0000000000000000000000000000000000000000000000000000000000000000),
        .INIT_30(256'h0000000000000000000000000000000000000000000000000000000000000000),
        .INIT_31(256'h0000000000000000000000000000000000000000000000000000000000000000),
        .INIT_32(256'h0000000000000000000000000000000000000000000000000000000000000000),
        .INIT_33(256'h0000000000000000000000000000000000000000000000000000000000000000),
        .INIT_34(256'h0000000000000000000000000000000000000000000000000000000000000000),
        .INIT_35(256'h0000000000000000000000000000000000000000000000000000000000000000),
        .INIT_36(256'h0000000000000000000000000000000000000000000000000000000000000000),
        .INIT_37(256'h0000000000000000000000000000000000000000000000000000000000000000),
        .INIT_38(256'h0000000000000000000000000000000000000000000000000000000000000000),
        .INIT_39(256'h0000000000000000000000000000000000000000000000000000000000000000),
        .INIT_3A(256'h0000000000000000000000000000000000000000000000000000000000000000),
        .INIT_3B(256'h0000000000000000000000000000000000000000000000000000000000000000),
        .INIT_3C(256'h0000000000000000000000000000000000000000000000000000000000000000),
        .INIT_3D(256'h0000000000000000000000000000000000000000000000000000000000000000),
        .INIT_3E(256'h0000000000000000000000000000000000000000000000000000000000000000),
        .INIT_3F(256'h0000000000000000000000000000000000000000000000000000000000000000),
        // }}}

        // Initialization of Data Bits in Higher 16 Kibit {{{
        // The next set of INIT_xx are valid when configured as 36Kb
        .INIT_40(256'h0000000000000000000000000000000000000000000000000000000000000000),
        .INIT_41(256'h0000000000000000000000000000000000000000000000000000000000000000),
        .INIT_42(256'h0000000000000000000000000000000000000000000000000000000000000000),
        .INIT_43(256'h0000000000000000000000000000000000000000000000000000000000000000),
        .INIT_44(256'h0000000000000000000000000000000000000000000000000000000000000000),
        .INIT_45(256'h0000000000000000000000000000000000000000000000000000000000000000),
        .INIT_46(256'h0000000000000000000000000000000000000000000000000000000000000000),
        .INIT_47(256'h0000000000000000000000000000000000000000000000000000000000000000),
        .INIT_48(256'h0000000000000000000000000000000000000000000000000000000000000000),
        .INIT_49(256'h0000000000000000000000000000000000000000000000000000000000000000),
        .INIT_4A(256'h0000000000000000000000000000000000000000000000000000000000000000),
        .INIT_4B(256'h0000000000000000000000000000000000000000000000000000000000000000),
        .INIT_4C(256'h0000000000000000000000000000000000000000000000000000000000000000),
        .INIT_4D(256'h0000000000000000000000000000000000000000000000000000000000000000),
        .INIT_4E(256'h0000000000000000000000000000000000000000000000000000000000000000),
        .INIT_4F(256'h0000000000000000000000000000000000000000000000000000000000000000),
        .INIT_50(256'h0000000000000000000000000000000000000000000000000000000000000000),
        .INIT_51(256'h0000000000000000000000000000000000000000000000000000000000000000),
        .INIT_52(256'h0000000000000000000000000000000000000000000000000000000000000000),
        .INIT_53(256'h0000000000000000000000000000000000000000000000000000000000000000),
        .INIT_54(256'h0000000000000000000000000000000000000000000000000000000000000000),
        .INIT_55(256'h0000000000000000000000000000000000000000000000000000000000000000),
        .INIT_56(256'h0000000000000000000000000000000000000000000000000000000000000000),
        .INIT_57(256'h0000000000000000000000000000000000000000000000000000000000000000),
        .INIT_58(256'h0000000000000000000000000000000000000000000000000000000000000000),
        .INIT_59(256'h0000000000000000000000000000000000000000000000000000000000000000),
        .INIT_5A(256'h0000000000000000000000000000000000000000000000000000000000000000),
        .INIT_5B(256'h0000000000000000000000000000000000000000000000000000000000000000),
        .INIT_5C(256'h0000000000000000000000000000000000000000000000000000000000000000),
        .INIT_5D(256'h0000000000000000000000000000000000000000000000000000000000000000),
        .INIT_5E(256'h0000000000000000000000000000000000000000000000000000000000000000),
        .INIT_5F(256'h0000000000000000000000000000000000000000000000000000000000000000),
        .INIT_60(256'h0000000000000000000000000000000000000000000000000000000000000000),
        .INIT_61(256'h0000000000000000000000000000000000000000000000000000000000000000),
        .INIT_62(256'h0000000000000000000000000000000000000000000000000000000000000000),
        .INIT_63(256'h0000000000000000000000000000000000000000000000000000000000000000),
        .INIT_64(256'h0000000000000000000000000000000000000000000000000000000000000000),
        .INIT_65(256'h0000000000000000000000000000000000000000000000000000000000000000),
        .INIT_66(256'h0000000000000000000000000000000000000000000000000000000000000000),
        .INIT_67(256'h0000000000000000000000000000000000000000000000000000000000000000),
        .INIT_68(256'h0000000000000000000000000000000000000000000000000000000000000000),
        .INIT_69(256'h0000000000000000000000000000000000000000000000000000000000000000),
        .INIT_6A(256'h0000000000000000000000000000000000000000000000000000000000000000),
        .INIT_6B(256'h0000000000000000000000000000000000000000000000000000000000000000),
        .INIT_6C(256'h0000000000000000000000000000000000000000000000000000000000000000),
        .INIT_6D(256'h0000000000000000000000000000000000000000000000000000000000000000),
        .INIT_6E(256'h0000000000000000000000000000000000000000000000000000000000000000),
        .INIT_6F(256'h0000000000000000000000000000000000000000000000000000000000000000),
        .INIT_70(256'h0000000000000000000000000000000000000000000000000000000000000000),
        .INIT_71(256'h0000000000000000000000000000000000000000000000000000000000000000),
        .INIT_72(256'h0000000000000000000000000000000000000000000000000000000000000000),
        .INIT_73(256'h0000000000000000000000000000000000000000000000000000000000000000),
        .INIT_74(256'h0000000000000000000000000000000000000000000000000000000000000000),
        .INIT_75(256'h0000000000000000000000000000000000000000000000000000000000000000),
        .INIT_76(256'h0000000000000000000000000000000000000000000000000000000000000000),
        .INIT_77(256'h0000000000000000000000000000000000000000000000000000000000000000),
        .INIT_78(256'h0000000000000000000000000000000000000000000000000000000000000000),
        .INIT_79(256'h0000000000000000000000000000000000000000000000000000000000000000),
        .INIT_7A(256'h0000000000000000000000000000000000000000000000000000000000000000),
        .INIT_7B(256'h0000000000000000000000000000000000000000000000000000000000000000),
        .INIT_7C(256'h0000000000000000000000000000000000000000000000000000000000000000),
        .INIT_7D(256'h0000000000000000000000000000000000000000000000000000000000000000),
        .INIT_7E(256'h0000000000000000000000000000000000000000000000000000000000000000),
        .INIT_7F(256'h0000000000000000000000000000000000000000000000000000000000000000),
        // }}}

        // Initialization of Parity Bits in Lower 16 Kibit {{{
        // The next set of INITP_xx are for the parity bits
        .INITP_00(256'h0000000000000000000000000000000000000000000000000000000000000000),
        .INITP_01(256'h0000000000000000000000000000000000000000000000000000000000000000),
        .INITP_02(256'h0000000000000000000000000000000000000000000000000000000000000000),
        .INITP_03(256'h0000000000000000000000000000000000000000000000000000000000000000),
        .INITP_04(256'h0000000000000000000000000000000000000000000000000000000000000000),
        .INITP_05(256'h0000000000000000000000000000000000000000000000000000000000000000),
        .INITP_06(256'h0000000000000000000000000000000000000000000000000000000000000000),
        .INITP_07(256'h0000000000000000000000000000000000000000000000000000000000000000),
        // }}}

        // Initialization of Parity Bits in Higher 16 Kibit {{{
        // The next set of INITP_xx are valid when configured as 36Kb
        .INITP_08(256'h0000000000000000000000000000000000000000000000000000000000000000),
        .INITP_09(256'h0000000000000000000000000000000000000000000000000000000000000000),
        .INITP_0A(256'h0000000000000000000000000000000000000000000000000000000000000000),
        .INITP_0B(256'h0000000000000000000000000000000000000000000000000000000000000000),
        .INITP_0C(256'h0000000000000000000000000000000000000000000000000000000000000000),
        .INITP_0D(256'h0000000000000000000000000000000000000000000000000000000000000000),
        .INITP_0E(256'h0000000000000000000000000000000000000000000000000000000000000000),
        .INITP_0F(256'h0000000000000000000000000000000000000000000000000000000000000000)
        // }}}

      )
      // }}}

      // BRAM_TDP_MACRO Instantation {{{
      BRAM_TDP_MACRO_inst (

        // Port A {{{
        .CLKA   (A_PS.Clk_C                             ),  //  1-bit inp: clock
        .RSTA   (A_PS.Rst_R                             ),  //  1-bit inp: reset (active high)
        .ENA    (A_PS.En_S                              ),  //  1-bit inp: enable
        .REGCEA (1'b0                                   ),  //  1-bit inp: output register enable
        .ADDRA  (WordIdxA_S                             ),  // 10-bit inp: word-wise address
        .DOA    (ARd_D[s] [WORD_BIT_HIGH:WORD_BIT_LOW]  ),  // 32-bit oup: data output
        .DIA    (A_PS.Wr_D[WORD_BIT_HIGH:WORD_BIT_LOW]  ),  // 32-bit inp: data input
        .WEA    (WrEnA_S                                ),  //  4-bit inp: byte-wise write enable
        // }}}

        // Port B {{{
        .CLKB   (B_PS.Clk_C                             ),  //  1-bit inp: clock
        .RSTB   (B_PS.Rst_R                             ),  //  1-bit inp: reset (active high)
        .ENB    (B_PS.En_S                              ),  //  1-bit inp: enable
        .REGCEB (1'b0                                   ),  //  1-bit inp: output register enable
        .ADDRB  (WordIdxB_S                             ),  // 10-bit inp: word-wise address
        .DOB    (BRd_D[s] [WORD_BIT_HIGH:WORD_BIT_LOW]  ),  // 32-bit oup: data output
        .DIB    (B_PS.Wr_D[WORD_BIT_HIGH:WORD_BIT_LOW]  ),  // 32-bit inp: data input
        .WEB    (WrEnB_S                                )   //  4-bit inp: byte-wise write enable
        // }}}

      );
      // }}}

    end
  end
  // }}}

  // Output Multiplexer {{{
  assign A_PS.Rd_D = ARd_D[SerIdxA_S];
  assign B_PS.Rd_D = BRd_D[SerIdxB_S];
  // }}}

endmodule

// vim: nosmartindent autoindent foldmethod=marker
