#!/usr/bin/env python
# -*- coding: utf-8 -*-
"""
阶段5功能验证脚本

验证题目检索服务的基本功能：
- 语义检索
- 关键词检索
- 混合检索

注意：此脚本需要有效的 Embedding API 密钥才能运行完整测试。
"""

import sys
import os

# 修复Windows控制台编码问题
if sys.platform == 'win32':
    # 设置控制台代码页为UTF-8
    import ctypes
    try:
        ctypes.windll.kernel32.SetConsoleOutputCP(65001)
        ctypes.windll.kernel32.SetConsoleCP(65001)
    except Exception:
        pass

    # 设置标准输出为UTF-8
    if sys.stdout.encoding != 'utf-8':
        try:
            sys.stdout.reconfigure(encoding='utf-8')
            sys.stderr.reconfigure(encoding='utf-8')
        except Exception:
            pass

# 将项目根目录添加到 sys.path
project_root = os.path.abspath(os.path.join(os.path.dirname(__file__), '.'))
if project_root not in sys.path:
    sys.path.insert(0, project_root)

from src.core.config import get_config
from src.core.logger import setup_logger
from src.database.sqlite_dao import SQLiteDAO
from src.database.chroma_dao import ChromaDAO
from src.database.database_manager import DatabaseManager
from src.services.embedding_service import EmbeddingService
from src.services.search_service import SearchService


def main():
    print("=" * 80)
    print("阶段5: 题目检索服务功能验证")
    print("=" * 80)
    print()

    try:
        # 1. 初始化配置和日志
        print("1. 初始化配置和日志...")
        config = get_config()
        logger = setup_logger(config)
        print("   [OK] 配置加载成功")
        print()

        # 2. 初始化数据库访问对象
        print("2. 初始化数据库访问对象...")
        sqlite_dao = SQLiteDAO(config.sqlite_db_path, logger)
        chroma_dao = ChromaDAO(
            persist_dir=config.chromadb_persist_dir,
            collection_name=config.chromadb_collection_name,
            logger=logger
        )
        print("   [OK] 数据库DAO初始化成功")
        print()

        # 3. 初始化Embedding服务
        print("3. 初始化Embedding服务...")
        try:
            embedding_service = EmbeddingService(config_manager=config)
            print("   [OK] Embedding服务初始化成功")
        except Exception as e:
            print(f"   [WARN] Embedding服务初始化失败: {e}")
            print("   跳过需要Embedding的功能测试")
            embedding_service = None
        print()

        # 4. 初始化数据库管理器
        print("4. 初始化数据库管理器...")
        db_manager = DatabaseManager(
            sqlite_dao=sqlite_dao,
            chroma_dao=chroma_dao,
            embedding_service=embedding_service,
            logger=logger
        )
        print("   [OK] 数据库管理器创建成功")
        print()

        # 5. 初始化数据库
        print("5. 初始化数据库...")
        db_manager.initialize_databases()
        print("   [OK] 数据库初始化完成")
        print()

        # 6. 初始化搜索服务
        print("6. 初始化搜索服务...")
        if embedding_service:
            search_service = SearchService(
                db_manager=db_manager,
                embedding_service=embedding_service,
                logger=logger
            )
            print("   [OK] 搜索服务初始化成功")
            print()

            # 7. 测试关键词检索
            print("7. 测试关键词检索...")
            try:
                # 注意：这里只是测试API是否可以正常调用
                # 实际的检索需要数据库中有数据
                result = search_service.search_by_keyword(
                    query="测试",
                    top_k=5
                )
                print("   [OK] 关键词检索API调用成功")
                print(f"   - 检索类型: {result.get('search_type')}")
                print(f"   - 查询: {result.get('query')}")
                print(f"   - 结果数量: {result.get('total_results', 0)}")
            except Exception as e:
                print(f"   [WARN] 关键词检索测试失败: {e}")
            print()

            # 8. 测试语义检索
            print("8. 测试语义检索...")
            try:
                result = search_service.search_by_semantic(
                    query="数据结构",
                    top_k=5
                )
                print("   [OK] 语义检索API调用成功")
                print(f"   - 检索类型: {result.get('search_type')}")
                print(f"   - 查询: {result.get('query')}")
                print(f"   - 结果数量: {result.get('total_results', 0)}")
            except Exception as e:
                print(f"   [WARN] 语义检索测试失败: {e}")
            print()

            # 9. 测试混合检索
            print("9. 测试混合检索...")
            try:
                result = search_service.search_hybrid(
                    query="二叉树遍历",
                    top_k=5
                )
                print("   [OK] 混合检索API调用成功")
                print(f"   - 检索类型: {result.get('search_type')}")
                print(f"   - 查询: {result.get('query')}")
                print(f"   - 结果数量: {result.get('total_results', 0)}")
                print(f"   - 语义权重: {result.get('metadata', {}).get('semantic_weight')}")
                print(f"   - 关键词权重: {result.get('metadata', {}).get('keyword_weight')}")
            except Exception as e:
                print(f"   [WARN] 混合检索测试失败: {e}")
            print()

            # 10. 获取搜索统计信息
            print("10. 获取搜索统计信息...")
            try:
                stats = search_service.get_search_statistics()
                print("   [OK] 统计信息获取成功")
                print(f"   - 缓存命中: {stats.get('embedding_cache_stats', {}).get('hits', 0)}")
                print(f"   - 缓存未命中: {stats.get('embedding_cache_stats', {}).get('misses', 0)}")
            except Exception as e:
                print(f"   [WARN] 统计信息获取失败: {e}")
            print()

        else:
            print("   [WARN] 跳过搜索服务测试（Embedding服务未初始化）")
            print()

        # 11. 验证数据库表结构
        print("11. 验证FTS表是否存在...")
        try:
            with sqlite_dao.get_connection() as conn:
                cursor = conn.execute(
                    "SELECT name FROM sqlite_master WHERE type='table' AND name='questions_fts'"
                )
                if cursor.fetchone():
                    print("   [OK] FTS表存在")
                else:
                    print("   [WARN] FTS表不存在")
        except Exception as e:
            print(f"   [WARN] FTS表验证失败: {e}")
        print()

        print("=" * 80)
        print("阶段5功能验证完成")
        print("=" * 80)
        print()
        print("已实现的功能:")
        print("  [OK] 语义检索 (基于ChromaDB向量相似度)")
        print("  [OK] 关键词检索 (基于SQLite FTS5)")
        print("  [OK] 混合检索 (语义+关键词)")
        print()
        print("MCP Tools:")
        print("  [OK] search_questions - 支持语义/关键词/混合检索")
        print("  [OK] list_questions - 分页查询题目列表")
        print()

    except Exception as e:
        print(f"[ERROR] 验证失败: {e}")
        import traceback
        traceback.print_exc()
        sys.exit(1)


if __name__ == "__main__":
    main()
