# CLAUDE.md

This file provides guidance to Claude Code (claude.ai/code) when working with code in this repository.

## 项目概述

这是一个基于 FastMCP 框架的 MCP Server，用于题目的全生命周期管理、智能检索和统计分析。系统采用**双数据库架构**：
- **SQLite**: 存储完整的题目元数据，支持复杂查询和统计分析
- **ChromaDB**: 存储向量数据，支持语义相似度检索

**当前进度**: 8个阶段中已完成5个阶段（阶段1-5），约62.5%完成度
- ✅ 阶段1: 环境准备与基础框架搭建
- ✅ 阶段2: 数据库设计与初始化
- ✅ 阶段3: Embedding API集成
- ✅ 阶段4: 题目管理服务
- ✅ 阶段5: 题目检索服务 ⭐ (新增)
- ⏳ 阶段6-8: 批量导入、统计分析、集成测试

## 核心架构

### 项目结构
```
questions_mcp_server/
├── config/                  # 配置文件
│   └── config.yaml         # YAML配置文件
├── src/
│   ├── core/               # 核心模块
│   │   ├── config.py       # 配置管理（支持YAML+环境变量）
│   │   └── logger.py       # 日志管理（多输出、脱敏、request_id追踪）
│   ├── database/           # 数据层
│   │   ├── models.py       # 数据模型（Pydantic）
│   │   ├── sqlite_dao.py   # SQLite DAO
│   │   ├── chroma_dao.py   # ChromaDB DAO
│   │   └── database_manager.py  # 数据库管理器（两阶段提交）
│   ├── services/           # 业务服务
│   │   ├── embedding_service.py  # Embedding服务（多API支持、熔断器）
│   │   ├── management_service.py # 题目管理服务（CRUD+批量操作）
│   │   └── search_service.py     # 题目检索服务（语义+关键词+混合检索）
│   ├── utils/              # 工具模块
│   │   ├── helpers.py      # 通用工具函数
│   │   └── cache.py        # LRU缓存（线程安全、TTL）
│   └── main.py             # 主程序入口
├── tests/                  # 单元测试
└── .env.example           # 环境变量示例
```

### 关键技术栈
- **FastMCP** (>=0.2.0): MCP Server框架
- **ChromaDB** (>=0.4.0): 向量数据库
- **SQLite**: 元数据库
- **Pydantic** (>=2.5.0): 数据验证
- **PyYAML**: 配置解析
- **python-dotenv**: 环境变量

## 常用开发命令

### 安装依赖
```bash
pip install -r requirements.txt
```

### 配置环境
```bash
# 复制环境变量示例文件
cp .env.example .env

# 编辑.env文件，设置必要的API密钥
# 必需: EMBEDDING_API_KEY
```

### 运行MCP Server
```bash
python src/main.py
```

### 运行测试
```bash
# 运行所有测试
python -m pytest tests/ -v

# 运行特定测试文件
python -m pytest tests/test_database.py -v
python -m pytest tests/test_embedding_service.py -v
python -m pytest tests/test_management_service.py -v
python -m pytest tests/test_search_service.py -v

# 运行验证脚本（阶段3功能验证）
python verify_stage3.py

# 运行验证脚本（阶段5检索功能验证）
python verify_stage5.py
```

## 核心功能模块

### 1. 配置管理 (src/core/config.py)
- **加载顺序**: 默认配置 → YAML文件 → 环境变量
- **支持嵌套配置**: 使用 `get()` 方法访问，如 `config.get("database.sqlite.db_path")`
- **便捷属性**: 提供属性访问，如 `config.sqlite_db_path`
- **配置验证**: 启动时自动验证必需配置项

### 2. 日志管理 (src/core/logger.py)
- **多输出目标**: 文件、控制台、错误日志单独存储
- **敏感信息脱敏**: API密钥等自动脱敏
- **request_id追踪**: 线程本地存储，追踪请求链路
- **日志轮转**: 基于文件大小自动轮转
- **使用**: 通过 `get_logger()` 获取 logger 实例

### 3. 双数据库架构

#### SQLite (src/database/sqlite_dao.py)
- **表结构**: questions, tags, question_tags
- **特性**: WAL模式、触发器、索引优化
- **事务管理**: 支持事务和并发控制
- **功能**: CRUD操作、标签管理、统计分析

#### ChromaDB (src/database/chroma_dao.py)
- **Collection**: questions_collection
- **距离度量**: 余弦相似度
- **索引**: HNSW
- **功能**: 向量存储、相似度检索、元数据过滤、批量操作

#### 数据库管理器 (src/database/database_manager.py)
- **两阶段提交**: 确保SQLite和ChromaDB数据一致性
- **失败回滚**: 任一数据库失败时自动回滚
- **一致性检查**: 定期检查并修复数据不一致
- **统计功能**: 统一的统计数据获取

### 4. Embedding服务 (src/services/embedding_service.py)
- **多API支持**: OpenAI、通义千问、智谱AI（自动检测）
- **批量向量化**: 支持单文本和批量处理
- **缓存机制**: LRU缓存减少API调用
- **熔断器**: CLOSED/OPEN/HALF_OPEN三状态管理
- **重试机制**: 指数退避重试

### 5. 题目管理服务 (src/services/management_service.py)
- **CRUD操作**: 创建、获取、更新、删除题目
- **批量操作**: 批量创建、更新、删除（ThreadPoolExecutor并发）
- **自动向量化**: 创建/更新时自动调用Embedding服务
- **数据一致性**: 通过DatabaseManager保证
- **分页查询**: 灵活的分页和过滤支持

### 6. 缓存系统 (src/utils/cache.py)
- **LRU缓存**: 线程安全的LRU实现
- **TTL支持**: 缓存项生存时间
- **统计功能**: 缓存命中统计
- **专用缓存**: EmbeddingCache、QueryCache

## 数据模型

### QuestionModel (src/database/models.py)
```python
- question_id: UUID - 题目唯一标识
- content: 文本 - 题目内容（题干+选项）
- title: 字符串 - 题目标题
- question_type: 字符串 - 题型（单选/多选/判断/填空/简答）
- category: 字符串 - 题目分类
- difficulty: 字符串 - 难度等级
- tags: List[str] - 题目标签
- answer: 文本 - 参考答案
- explanation: 文本 - 答案解析
- status: 字符串 - 状态（草稿/已发布/已归档）
- created_at/updated_at: 时间戳
```

### DTO模式
- **QuestionCreateDTO**: 创建题目数据
- **QuestionUpdateDTO**: 更新题目数据
- **QuestionSearchFilter**: 检索过滤条件

## MCP Tools

当前已实现的MCP Tools（在src/main.py中注册）:

1. **health_check**: 健康检查
   - 返回系统状态和各组件健康状况

2. **get_server_info**: 获取服务器信息
   - 返回详细的配置、特性、状态信息

3. **test_configuration**: 测试配置
   - 验证系统配置是否正确

4. **search_questions**: 题目检索工具 ⭐ (阶段5新增)
   - 支持语义检索（基于ChromaDB向量相似度）
   - 支持关键词检索（基于SQLite FTS5）
   - 支持混合检索（语义+关键词智能融合）
   - 可配置权重、相似度阈值、过滤条件
   - 示例: search_questions(query="二叉树", search_type="hybrid")

5. **list_questions**: 分页查询题目列表 ⭐ (阶段5新增)
   - 支持多维度过滤（分类、难度、题型、状态）
   - 分页信息返回
   - 示例: list_questions(page=1, page_size=20, category="数学")

**后续阶段将实现**:
- create_question, update_question, delete_question, get_question
- import_questions (批量导入)
- 5个统计分析Tools

## 关键配置项

### 必需环境变量
```bash
EMBEDDING_API_KEY=your_embedding_api_key_here
```

### 重要配置 (config/config.yaml)
```yaml
database:
  sqlite:
    db_path: "./data/questions.db"  # SQLite数据库路径
    journal_mode: "WAL"             # WAL模式提高并发
  chromadb:
    persist_dir: "./data/chroma"    # ChromaDB持久化目录
    collection_name: "questions_collection"
    distance_metric: "cosine"       # 距离度量

embedding:
  api_endpoint: "https://api.example.com/v1/embeddings"
  model: "text-embedding-v2"
  batch_size: 20                    # 批处理大小
  max_retries: 3                    # 最大重试次数
  circuit_breaker:                  # 熔断器配置
    failure_threshold: 5
    timeout: 60

cache:
  embedding_cache:
    max_size: 1000                  # 向量缓存大小
    ttl: 3600                       # 缓存时间（秒）
```

## 开发规范

### 代码质量要求
- ✅ 完整的类型注解 (Type Hints)
- ✅ 详细的文档字符串 (Docstrings)
- ✅ 遵循 PEP 8 编码规范
- ✅ 单一职责原则
- ✅ 适当的异常处理
- ✅ 为所有新功能编写单元测试

### 目录规范
- **core/**: 核心模块（配置、日志等基础设施）
- **database/**: 数据访问层（DAO、模型、数据库管理）
- **services/**: 业务服务层（管理、检索、统计等）
- **utils/**: 工具函数和通用组件

### 数据一致性保证
1. 所有数据库操作通过 `DatabaseManager` 协调
2. 使用两阶段提交确保SQLite和ChromaDB同步
3. 失败时自动回滚或补偿
4. 定期执行一致性检查

## 扩展指南

### 添加新的MCP Tool
1. 在相应的Service中实现业务逻辑
2. 在 `src/main.py` 中使用 `@server.tool()` 装饰器注册
3. 添加完整的类型注解和文档字符串
4. 编写单元测试

### 添加新的API提供商
1. 在 `EmbeddingProvider` 枚举中添加新提供商
2. 在 `EmbeddingService` 中实现对应的API调用逻辑
3. 更新自动检测逻辑
4. 添加测试用例

### 性能优化建议
1. **向量缓存**: 合理设置 `embedding_cache.max_size` 和 `ttl`
2. **批处理**: 使用 `batch_size` 减少API调用次数
3. **数据库索引**: 在SQLite上创建必要的复合索引
4. **并发控制**: 批量操作使用 `ThreadPoolExecutor`

## 测试策略

### 现有测试
- **test_database.py**: 数据库层测试（CRUD、事务）
- **test_embedding_service.py**: Embedding服务测试（缓存、熔断、API调用）
- **test_management_service.py**: 管理服务测试（CRUD、批量操作、异常处理）
- **verify_stage3.py**: 阶段3功能验证脚本

### 测试命令
```bash
# 运行所有测试
pytest tests/ -v

# 运行特定测试
pytest tests/test_database.py::TestSQLiteDAO -v

# 生成覆盖率报告
pytest tests/ --cov=src --cov-report=html
```

## 故障排查

### 常见问题

1. **配置加载失败**
   - 检查 `.env` 文件是否存在且包含必需的环境变量
   - 验证 `config/config.yaml` 路径和格式
   - 运行 `test_configuration` Tool检查配置

2. **数据库连接失败**
   - 确保数据目录存在且有写权限
   - 检查SQLite和ChromaDB路径配置
   - 查看 `logs/error.log` 获取详细错误

3. **Embedding API调用失败**
   - 验证 `EMBEDDING_API_KEY` 是否正确
   - 检查API端点和模型名称
   - 查看熔断器状态（可能需要等待恢复）

4. **数据不一致**
   - 运行一致性检查功能
   - 查看数据库管理器日志
   - 手动触发数据同步

### 日志位置
- 应用日志: `./logs/app.log`
- 错误日志: `./logs/error.log`
- 日志级别可通过 `LOG_LEVEL` 环境变量或配置文件调整

## 后续开发计划

基于 DEVELOPMENT_STATUS.md，项目正在按以下阶段推进:

**阶段5: 题目检索服务** (✅ 已完成)
- ✅ 语义检索（基于ChromaDB向量相似度）
- ✅ 关键词检索（基于SQLite FTS5）
- ✅ 混合检索支持（语义+关键词智能融合）
- ✅ MCP Tools集成（search_questions, list_questions）
- ✅ 单元测试（14个测试全部通过）

**阶段6: 批量导入功能** (未开始)
- 外部API对接
- 重复检测
- 断点续传

**阶段7: 统计分析服务** (未开始)
- 多维度统计
- 质量报告
- 时间序列分析

**阶段8: 集成测试与优化** (未开始)
- 端到端测试
- 性能优化
- 文档完善

## 重要文档

- **设计文档**: `.qoder/quests/mcps-server-design.md` - 完整的系统架构和设计说明
- **开发状态**: `DEVELOPMENT_STATUS.md` - 当前开发进度和已完成功能
- **开发规则**: `.qoder/rules/questions_mcp_server.md` - 开发规范和最佳实践
- **配置示例**: `.env.example` - 所有可配置项的详细说明

## 注意事项

1. **永远不要将包含真实密钥的 `.env` 文件提交到版本控制**
2. **所有数据库操作必须通过 `DatabaseManager` 以保证一致性**
3. **更新 content 字段会触发重新向量化，注意API调用成本**
4. **批量操作默认使用4个线程，可通过配置调整**
5. **ChromaDB和SQLite数据通过question_id保持同步，请勿直接修改**

### 注意运行python请使用uv python!!!