import { ApiObject } from 'cdk8s';
import { Construct } from 'constructs';
import * as base from './base';
import * as container from './container';
import * as k8s from './imports/k8s';
import * as secret from './secret';
import * as serviceaccount from './service-account';
import * as volume from './volume';
export declare abstract class AbstractPod extends base.Resource {
    readonly restartPolicy?: RestartPolicy;
    readonly serviceAccount?: serviceaccount.IServiceAccount;
    readonly securityContext: PodSecurityContext;
    readonly dns: PodDns;
    readonly dockerRegistryAuth?: secret.DockerConfigSecret;
    readonly automountServiceAccountToken: boolean;
    private readonly _containers;
    private readonly _initContainers;
    private readonly _hostAliases;
    private readonly _volumes;
    constructor(scope: Construct, id: string, props?: AbstractPodProps);
    get containers(): container.Container[];
    get initContainers(): container.Container[];
    get volumes(): volume.Volume[];
    get hostAliases(): HostAlias[];
    addContainer(cont: container.ContainerProps): container.Container;
    addInitContainer(cont: container.ContainerProps): container.Container;
    addHostAlias(hostAlias: HostAlias): void;
    addVolume(vol: volume.Volume): void;
    /**
     * @internal
     */
    _toPodSpec(): k8s.PodSpec;
}
/**
 * Sysctl defines a kernel parameter to be set
 */
export interface Sysctl {
    /**
     * Name of a property to set
     */
    readonly name: string;
    /**
     * Value of a property to set
     */
    readonly value: string;
}
/**
 * Properties for `PodSecurityContext`
 */
export interface PodSecurityContextProps {
    /**
     * Modify the ownership and permissions of pod volumes to this GID.
     *
     * @default - Volume ownership is not changed.
     */
    readonly fsGroup?: number;
    /**
     * Defines behavior of changing ownership and permission of the volume before being exposed inside Pod.
     * This field will only apply to volume types which support fsGroup based ownership(and permissions).
     * It will have no effect on ephemeral volume types such as: secret, configmaps and emptydir.
     *
     * @default FsGroupChangePolicy.ALWAYS
     */
    readonly fsGroupChangePolicy?: FsGroupChangePolicy;
    /**
     * The UID to run the entrypoint of the container process.
     *
     * @default - User specified in image metadata
     */
    readonly user?: number;
    /**
     * The GID to run the entrypoint of the container process.
     *
     * @default - Group configured by container runtime
     */
    readonly group?: number;
    /**
     * Indicates that the container must run as a non-root user.
     * If true, the Kubelet will validate the image at runtime to ensure that it does
     * not run as UID 0 (root) and fail to start the container if it does.
     *
     * @default false
     */
    readonly ensureNonRoot?: boolean;
    /**
     * Sysctls hold a list of namespaced sysctls used for the pod.
     * Pods with unsupported sysctls (by the container runtime) might fail to launch.
     *
     * @default - No sysctls
     */
    readonly sysctls?: Sysctl[];
}
/**
 * Properties for `AbstractPod`.
 */
export interface AbstractPodProps extends base.ResourceProps {
    /**
     * List of containers belonging to the pod. Containers cannot currently be
     * added or removed. There must be at least one container in a Pod.
     *
     * You can add additionnal containers using `podSpec.addContainer()`
     *
     * @default - No containers. Note that a pod spec must include at least one container.
     */
    readonly containers?: container.ContainerProps[];
    /**
     * List of initialization containers belonging to the pod.
     * Init containers are executed in order prior to containers being started.
     * If any init container fails, the pod is considered to have failed and is handled according to its restartPolicy.
     * The name for an init container or normal container must be unique among all containers.
     * Init containers may not have Lifecycle actions, Readiness probes, Liveness probes, or Startup probes.
     * The resourceRequirements of an init container are taken into account during scheduling by finding the highest request/limit
     * for each resource type, and then using the max of of that value or the sum of the normal containers.
     * Limits are applied to init containers in a similar fashion.
     *
     * Init containers cannot currently be added ,removed or updated.
     *
     * @see https://kubernetes.io/docs/concepts/workloads/pods/init-containers/
     * @default - No init containers.
     */
    readonly initContainers?: container.ContainerProps[];
    /**
     * List of volumes that can be mounted by containers belonging to the pod.
     *
     * You can also add volumes later using `podSpec.addVolume()`
     *
     * @see https://kubernetes.io/docs/concepts/storage/volumes
     *
     * @default - No volumes.
     */
    readonly volumes?: volume.Volume[];
    /**
     * Restart policy for all containers within the pod.
     *
     * @see https://kubernetes.io/docs/concepts/workloads/pods/pod-lifecycle/#restart-policy
     *
     * @default RestartPolicy.ALWAYS
     */
    readonly restartPolicy?: RestartPolicy;
    /**
     * A service account provides an identity for processes that run in a Pod.
     *
     * When you (a human) access the cluster (for example, using kubectl), you are
     * authenticated by the apiserver as a particular User Account (currently this
     * is usually admin, unless your cluster administrator has customized your
     * cluster). Processes in containers inside pods can also contact the
     * apiserver. When they do, they are authenticated as a particular Service
     * Account (for example, default).
     *
     * @see https://kubernetes.io/docs/tasks/configure-pod-container/configure-service-account/
     *
     * @default - No service account.
     */
    readonly serviceAccount?: serviceaccount.IServiceAccount;
    /**
     * SecurityContext holds pod-level security attributes and common container settings.
     *
     * @default
     *
     *   fsGroupChangePolicy: FsGroupChangePolicy.FsGroupChangePolicy.ALWAYS
     *   ensureNonRoot: false
     */
    readonly securityContext?: PodSecurityContextProps;
    /**
     * HostAlias holds the mapping between IP and hostnames that will be injected as an entry in the pod's hosts file.
     *
     * @schema io.k8s.api.core.v1.HostAlias
     */
    readonly hostAliases?: HostAlias[];
    /**
     * DNS settings for the pod.
     *
     * @see https://kubernetes.io/docs/concepts/services-networking/dns-pod-service/
     *
     * @default
     *
     *  policy: DnsPolicy.CLUSTER_FIRST
     *  hostnameAsFQDN: false
     */
    readonly dns?: PodDnsProps;
    /**
     * A secret containing docker credentials for authenticating to a registry.
     *
     * @default - No auth. Images are assumed to be publicly available.
     */
    readonly dockerRegistryAuth?: secret.DockerConfigSecret;
    /**
     * Indicates whether a service account token should be automatically mounted.
     *
     * @default true
     * @see https://kubernetes.io/docs/tasks/configure-pod-container/configure-service-account/#use-the-default-service-account-to-access-the-api-server
     */
    readonly automountServiceAccountToken?: boolean;
}
/**
 * Properties for `Pod`.
 */
export interface PodProps extends AbstractPodProps {
}
/**
 * Pod is a collection of containers that can run on a host. This resource is
 * created by clients and scheduled onto hosts.
 */
export declare class Pod extends AbstractPod {
    /**
     * @see base.Resource.apiObject
     */
    protected readonly apiObject: ApiObject;
    readonly resourceType = "pods";
    constructor(scope: Construct, id: string, props?: PodProps);
    /**
     * @internal
     */
    _toKube(): k8s.PodSpec;
}
/**
 * Properties for `PodDns`.
 */
export interface PodDnsProps {
    /**
     * Specifies the hostname of the Pod.
     *
     * @default - Set to a system-defined value.
     */
    readonly hostname?: string;
    /**
     * If specified, the fully qualified Pod hostname will be "<hostname>.<subdomain>.<pod namespace>.svc.<cluster domain>".
     *
     * @default - No subdomain.
     */
    readonly subdomain?: string;
    /**
     * If true the pod's hostname will be configured as the pod's FQDN, rather than the leaf name (the default).
     * In Linux containers, this means setting the FQDN in the hostname field of the kernel (the nodename field of struct utsname).
     * In Windows containers, this means setting the registry value of hostname for the registry
     * key HKEY_LOCAL_MACHINE\SYSTEM\CurrentControlSet\Services\Tcpip\Parameters to FQDN.
     * If a pod does not have FQDN, this has no effect.
     *
     * @default false
     */
    readonly hostnameAsFQDN?: boolean;
    /**
     * Set DNS policy for the pod.
     *
     * If policy is set to `None`, other configuration must be supplied.
     *
     * @default DnsPolicy.CLUSTER_FIRST
     */
    readonly policy?: DnsPolicy;
    /**
     * A list of IP addresses that will be used as DNS servers for the Pod. There can be at most 3 IP addresses specified.
     * When the policy is set to "NONE", the list must contain at least one IP address,
     * otherwise this property is optional.
     * The servers listed will be combined to the base nameservers generated from
     * the specified DNS policy with duplicate addresses removed.
     */
    readonly nameservers?: string[];
    /**
     * A list of DNS search domains for hostname lookup in the Pod.
     * When specified, the provided list will be merged into the base
     * search domain names generated from the chosen DNS policy.
     * Duplicate domain names are removed.
     *
     * Kubernetes allows for at most 6 search domains.
     */
    readonly searches?: string[];
    /**
     * List of objects where each object may have a name property (required)
     * and a value property (optional). The contents in this property
     * will be merged to the options generated from the specified DNS policy.
     * Duplicate entries are removed.
     */
    readonly options?: DnsOption[];
}
/**
 * Holds dns settings of the pod.
 */
export declare class PodDns {
    /**
     * The DNS policy of this pod.
     */
    readonly policy: DnsPolicy;
    /**
     * The configured hostname of the pod. Undefined means its set to a system-defined value.
     */
    readonly hostname?: string;
    /**
     * The configured subdomain of the pod.
     */
    readonly subdomain?: string;
    /**
     * Whether or not the pods hostname is set to its FQDN.
     */
    readonly hostnameAsFQDN: boolean;
    private readonly _nameservers;
    private readonly _searches;
    private readonly _options;
    constructor(props?: PodDnsProps);
    /**
     * Nameservers defined for this pod.
     */
    get nameservers(): string[];
    /**
     * Search domains defined for this pod.
     */
    get searches(): string[];
    /**
     * Custom dns options defined for this pod.
     */
    get options(): DnsOption[];
    /**
     * Add a nameserver.
     */
    addNameserver(...nameservers: string[]): void;
    /**
     * Add a search domain.
     */
    addSearch(...searches: string[]): void;
    /**
     * Add a custom option.
     */
    addOption(...options: DnsOption[]): void;
    /**
     * @internal
     */
    _toKube(): {
        hostname?: string;
        subdomain?: string;
        hostnameAsFQDN: boolean;
        policy: string;
        config: k8s.PodDnsConfig;
    };
}
/**
 * Holds pod-level security attributes and common container settings.
 */
export declare class PodSecurityContext {
    readonly ensureNonRoot: boolean;
    readonly user?: number;
    readonly group?: number;
    readonly fsGroup?: number;
    readonly fsGroupChangePolicy: FsGroupChangePolicy;
    private readonly _sysctls;
    constructor(props?: PodSecurityContextProps);
    get sysctls(): Sysctl[];
    /**
     * @internal
     */
    _toKube(): k8s.PodSecurityContext;
}
/**
 * Restart policy for all containers within the pod.
 */
export declare enum RestartPolicy {
    /**
     * Always restart the pod after it exits.
     */
    ALWAYS = "Always",
    /**
     * Only restart if the pod exits with a non-zero exit code.
     */
    ON_FAILURE = "OnFailure",
    /**
     * Never restart the pod.
     */
    NEVER = "Never"
}
export declare enum FsGroupChangePolicy {
    /**
     * Only change permissions and ownership if permission and ownership of root directory does
     * not match with expected permissions of the volume.
     * This could help shorten the time it takes to change ownership and permission of a volume
     */
    ON_ROOT_MISMATCH = "OnRootMismatch",
    /**
     * Always change permission and ownership of the volume when volume is mounted.
     */
    ALWAYS = "Always"
}
/**
 * Custom DNS option.
 */
export interface DnsOption {
    /**
     * Option name.
     */
    readonly name: string;
    /**
     * Option value.
     *
     * @default - No value.
     */
    readonly value?: string;
}
/**
 * Pod DNS policies.
 */
export declare enum DnsPolicy {
    /**
     * Any DNS query that does not match the configured cluster domain suffix,
     * such as "www.kubernetes.io", is forwarded to the
     * upstream nameserver inherited from the node.
     * Cluster administrators may have extra stub-domain and upstream DNS servers configured.
     */
    CLUSTER_FIRST = "ClusterFirst",
    /**
     * For Pods running with hostNetwork, you should
     * explicitly set its DNS policy "ClusterFirstWithHostNet".
     */
    CLUSTER_FIRST_WITH_HOST_NET = "ClusterFirstWithHostNet",
    /**
     * The Pod inherits the name resolution configuration
     * from the node that the pods run on.
     */
    DEFAULT = "Default",
    /**
     * It allows a Pod to ignore DNS settings from the Kubernetes environment.
     * All DNS settings are supposed to be provided using the dnsConfig
     * field in the Pod Spec.
     */
    NONE = "None"
}
/**
 * HostAlias holds the mapping between IP and hostnames that will be injected as
 * an entry in the pod's /etc/hosts file.
 */
export interface HostAlias {
    /**
     * Hostnames for the chosen IP address.
     */
    readonly hostnames: string[];
    /**
     * IP address of the host file entry.
     */
    readonly ip: string;
}
