"use strict";
var _a, _b, _c;
Object.defineProperty(exports, "__esModule", { value: true });
exports.DeploymentStrategy = exports.PercentOrAbsolute = exports.Deployment = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cdk8s_1 = require("cdk8s");
const k8s = require("./imports/k8s");
const service = require("./service");
const workload = require("./workload");
/**
*
* A Deployment provides declarative updates for Pods and ReplicaSets.
*
* You describe a desired state in a Deployment, and the Deployment Controller changes the actual
* state to the desired state at a controlled rate. You can define Deployments to create new ReplicaSets, or to remove
* existing Deployments and adopt all their resources with new Deployments.
*
* > Note: Do not manage ReplicaSets owned by a Deployment. Consider opening an issue in the main Kubernetes repository if your use case is not covered below.
*
* Use Case
* ---------
*
* The following are typical use cases for Deployments:
*
* - Create a Deployment to rollout a ReplicaSet. The ReplicaSet creates Pods in the background.
*   Check the status of the rollout to see if it succeeds or not.
* - Declare the new state of the Pods by updating the PodTemplateSpec of the Deployment.
*   A new ReplicaSet is created and the Deployment manages moving the Pods from the old ReplicaSet to the new one at a controlled rate.
*   Each new ReplicaSet updates the revision of the Deployment.
* - Rollback to an earlier Deployment revision if the current state of the Deployment is not stable.
*   Each rollback updates the revision of the Deployment.
* - Scale up the Deployment to facilitate more load.
* - Pause the Deployment to apply multiple fixes to its PodTemplateSpec and then resume it to start a new rollout.
* - Use the status of the Deployment as an indicator that a rollout has stuck.
* - Clean up older ReplicaSets that you don't need anymore.
*
**/
class Deployment extends workload.Workload {
    constructor(scope, id, props = {}) {
        var _d, _e, _f, _g;
        super(scope, id, props);
        this.resourceType = 'deployments';
        this.apiObject = new k8s.KubeDeployment(this, 'Resource', {
            metadata: props.metadata,
            spec: cdk8s_1.Lazy.any({ produce: () => this._toKube() }),
        });
        this.minReady = (_d = props.minReady) !== null && _d !== void 0 ? _d : cdk8s_1.Duration.seconds(0);
        this.progressDeadline = (_e = props.progressDeadline) !== null && _e !== void 0 ? _e : cdk8s_1.Duration.seconds(600);
        if (this.progressDeadline.toSeconds() <= this.minReady.toSeconds()) {
            throw new Error(`'progressDeadline' (${this.progressDeadline.toSeconds()}s) must be greater than 'minReady' (${this.minReady.toSeconds()}s)`);
        }
        this.replicas = (_f = props.replicas) !== null && _f !== void 0 ? _f : 1;
        this.strategy = (_g = props.strategy) !== null && _g !== void 0 ? _g : DeploymentStrategy.rollingUpdate();
    }
    /**
     * Expose a deployment via a service.
     *
     * This is equivalent to running `kubectl expose deployment <deployment-name>`.
     *
     * @param options Options to determine details of the service and port exposed.
     */
    exposeViaService(options = {}) {
        var _d;
        const ser = new service.Service(this, 'Service', {
            metadata: options.name ? { name: options.name } : undefined,
            type: (_d = options.serviceType) !== null && _d !== void 0 ? _d : service.ServiceType.CLUSTER_IP,
        });
        ser.addDeployment(this, { protocol: options.protocol, targetPort: options.targetPort, port: options.port });
        return ser;
    }
    /**
     * Expose a deployment via an ingress.
     *
     * This will first expose the deployment with a service, and then expose the service via an ingress.
     *
     * @param path The ingress path to register under.
     * @param options Additional options.
     */
    exposeViaIngress(path, options = {}) {
        const ser = this.exposeViaService(options);
        return ser.exposeViaIngress(path, options);
    }
    /**
     * @internal
     */
    _toKube() {
        return {
            replicas: this.replicas,
            minReadySeconds: this.minReady.toSeconds(),
            progressDeadlineSeconds: this.progressDeadline.toSeconds(),
            template: {
                metadata: this.podMetadata.toJson(),
                spec: this._toPodSpec(),
            },
            selector: {
                matchExpressions: this.matchExpressions,
                matchLabels: this.matchLabels,
            },
            strategy: this.strategy._toKube(),
        };
    }
}
exports.Deployment = Deployment;
_a = JSII_RTTI_SYMBOL_1;
Deployment[_a] = { fqn: "cdk8s-plus-22.Deployment", version: "2.0.0-beta.0" };
/**
 * Union like class repsenting either a ration in
 * percents or an absolute number.
 */
class PercentOrAbsolute {
    constructor(value) {
        this.value = value;
    }
    /**
     * Percent ratio.
     */
    static percent(percent) {
        return new PercentOrAbsolute(`${percent}%`);
    }
    /**
     * Absolute number.
     */
    static absolute(num) {
        return new PercentOrAbsolute(num);
    }
    isZero() {
        return this.value === PercentOrAbsolute.absolute(0).value || this.value === PercentOrAbsolute.percent(0).value;
    }
}
exports.PercentOrAbsolute = PercentOrAbsolute;
_b = JSII_RTTI_SYMBOL_1;
PercentOrAbsolute[_b] = { fqn: "cdk8s-plus-22.PercentOrAbsolute", version: "2.0.0-beta.0" };
/**
 * Deployment strategies.
 */
class DeploymentStrategy {
    constructor(strategy) {
        this.strategy = strategy;
    }
    /**
     * All existing Pods are killed before new ones are created.
     *
     * @see https://kubernetes.io/docs/concepts/workloads/controllers/deployment/#recreate-deployment
     */
    static recreate() {
        return new DeploymentStrategy({
            type: 'Recreate',
        });
    }
    static rollingUpdate(options = {}) {
        var _d, _e;
        const maxSurge = (_d = options.maxSurge) !== null && _d !== void 0 ? _d : PercentOrAbsolute.percent(25);
        const maxUnavailable = (_e = options.maxSurge) !== null && _e !== void 0 ? _e : PercentOrAbsolute.percent(25);
        if (maxSurge.isZero() && maxUnavailable.isZero()) {
            throw new Error('\'maxSurge\' and \'maxUnavailable\' cannot be both zero');
        }
        return new DeploymentStrategy({
            type: 'RollingUpdate',
            rollingUpdate: { maxSurge, maxUnavailable },
        });
    }
    /**
     * @internal
     */
    _toKube() {
        return this.strategy;
    }
}
exports.DeploymentStrategy = DeploymentStrategy;
_c = JSII_RTTI_SYMBOL_1;
DeploymentStrategy[_c] = { fqn: "cdk8s-plus-22.DeploymentStrategy", version: "2.0.0-beta.0" };
//# sourceMappingURL=data:application/json;base64,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