"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ConfigMap = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const configmap = require("fs");
const path = require("path");
const cdk8s_1 = require("cdk8s");
const minimatch_1 = require("minimatch");
const base = require("./base");
const k8s = require("./imports/k8s");
const utils_1 = require("./utils");
/**
 * ConfigMap holds configuration data for pods to consume.
 */
class ConfigMap extends base.Resource {
    constructor(scope, id, props = {}) {
        var _b, _c, _d;
        super(scope, id);
        this.resourceType = 'configmaps';
        this._binaryData = {};
        this._data = {};
        this.immutable = (_b = props.immutable) !== null && _b !== void 0 ? _b : false;
        this.apiObject = new k8s.KubeConfigMap(this, 'Resource', {
            metadata: props.metadata,
            // we need lazy here because we filter empty
            data: cdk8s_1.Lazy.any({ produce: () => this.synthesizeData() }),
            binaryData: cdk8s_1.Lazy.any({ produce: () => this.synthesizeBinaryData() }),
            immutable: this.immutable,
        });
        for (const [k, v] of Object.entries((_c = props.data) !== null && _c !== void 0 ? _c : {})) {
            this.addData(k, v);
        }
        for (const [k, v] of Object.entries((_d = props.binaryData) !== null && _d !== void 0 ? _d : {})) {
            this.addBinaryData(k, v);
        }
    }
    /**
     * Represents a ConfigMap created elsewhere.
     * @param name The name of the config map to import
     */
    static fromConfigMapName(name) {
        return {
            apiGroup: '',
            name,
            ...k8s.KubeConfigMap.GVK,
        };
    }
    /**
     * Adds a data entry to the config map.
     * @param key The key
     * @param value The value
     *
     * @throws if there is either a `data` or `binaryData` entry with the same key
     */
    addData(key, value) {
        this.verifyKeyAvailable(key);
        this._data[key] = value;
    }
    /**
     * The data associated with this config map.
     *
     * Returns an copy. To add data records, use `addData()` or `addBinaryData()`.
     */
    get data() {
        return { ...this._data };
    }
    /**
     * Adds a binary data entry to the config map. BinaryData can contain byte
     * sequences that are not in the UTF-8 range.
     * @param key The key
     * @param value The value
     *
     * @throws if there is either a `data` or `binaryData` entry with the same key
     */
    addBinaryData(key, value) {
        this.verifyKeyAvailable(key);
        this._binaryData[key] = value;
    }
    /**
     * The binary data associated with this config map.
     *
     * Returns a copy. To add data records, use `addBinaryData()` or `addData()`.
     */
    get binaryData() {
        return { ...this._binaryData };
    }
    /**
     * Adds a file to the ConfigMap.
     * @param localFile The path to the local file
     * @param key The ConfigMap key (default to the file name).
     */
    addFile(localFile, key) {
        key = key !== null && key !== void 0 ? key : path.basename(localFile);
        const value = configmap.readFileSync(localFile, 'utf-8');
        this.addData(key, value);
    }
    /**
     * Adds a directory to the ConfigMap.
     * @param localDir A path to a local directory
     * @param options Options
     */
    addDirectory(localDir, options = {}) {
        var _b, _c;
        const exclude = (_b = options.exclude) !== null && _b !== void 0 ? _b : [];
        const shouldInclude = (file) => {
            for (const pattern of exclude) {
                const mm = new minimatch_1.Minimatch(pattern);
                if (mm.match(file)) {
                    return false;
                }
            }
            return true;
        };
        const keyPrefix = (_c = options.keyPrefix) !== null && _c !== void 0 ? _c : '';
        for (const file of configmap.readdirSync(localDir)) {
            const filePath = path.join(localDir, file);
            if (configmap.statSync(filePath).isDirectory()) {
                continue;
            }
            if (!shouldInclude(file)) {
                continue;
            }
            const relativeFilePath = keyPrefix + file;
            this.addFile(filePath, relativeFilePath);
        }
    }
    verifyKeyAvailable(key) {
        if (key in this._data || key in this._binaryData) {
            throw new Error(`unable to add a ConfigMap entry with key "${key}". It is already used`);
        }
    }
    synthesizeData() {
        return utils_1.undefinedIfEmpty(this._data);
    }
    synthesizeBinaryData() {
        return utils_1.undefinedIfEmpty(this._binaryData);
    }
}
exports.ConfigMap = ConfigMap;
_a = JSII_RTTI_SYMBOL_1;
ConfigMap[_a] = { fqn: "cdk8s-plus-22.ConfigMap", version: "2.0.0-beta.0" };
//# sourceMappingURL=data:application/json;base64,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