import subprocess
from subprocess import Popen, PIPE
from pathlib import Path
from mstk import logger
from .pbsjob import PbsJob
from .scheduler import Scheduler, JobParameter


class Slurm(Scheduler):
    '''
    Slurm job scheduler with support of GPU allocation and MPI/OpenMP hybrid parallelization.

    Slurm is a rather powerful and *complicated* job scheduler with tons of configurations and options.
    It is not the goal of `mstk` to provide a comprehensive wrapper for Slurm.
    Therefore, it's very likely that the job script generated by this class doesn't fully fit the requirement of a specific computing center.
    In that case, it's viable to do some process on the generated job script before submitting it.

    Attributes
    ----------
    username : str
        The current user
    sh : str
        The default name of the job script
    job_parameter : JobParameter
        The default parameters for submitting a job
    submit_cmd : str
        The command for submitting the job script.
        If is `sbatch` by default. But extra argument can be provided, e.g. `sbatch --qos=debug`.
    cached_jobs_expire : int
        The lifetime of cached jobs in seconds.
    '''

    #: Whether this is a remote job scheduler
    is_remote = False

    def __init__(self):
        super().__init__()

        self.sh = '_job_slurm.sh'
        self.submit_cmd = 'sbatch'

    def is_working(self) -> bool:
        '''
        Check whether Slurm is working normally on this machine.

        It calls `sinfo --version` and check the output.

        Returns
        -------
        is : bool
        '''
        cmd = 'sinfo --version'
        sp = Popen(cmd.split(), stdout=PIPE, stderr=PIPE)
        stdout, stderr = sp.communicate()
        return stdout.decode().startswith('slurm')

    def generate_sh(self, commands, name, parameter=None, workdir=None, sh=None, id_prior=None, **kwargs):
        '''
        Generate a shell script for commands to be executed by the job scheduler on compute nodes.

        Because of the complexity of Slurm configurations, it's probable that the job script generated here is not fully valid.
        In that case, it's viable to do some process on the generated job script before submitting it.

        Parameters
        ----------
        commands : list of str
            The commands to be executed step by step
        name : str
            The name of the job to be submitted
        parameter : JobParameter, Optional
            The parameter for submitting this job.
            If not set, will use the default :attr:job_parameter.
        workdir : str, Optional
            Explicitly set the working directory of this job
        sh : str, Optional
            The name (path) of shell script to be generated.
            If not set, will use the default :attr:`sh`.
        id_prior: int, Optional
            The id of prior job this one depends on
        '''
        sh = sh or self.sh
        sh_basename = Path(sh).stem
        out = sh_basename + '.out'
        err = sh_basename + '.err'
        para = parameter or self.job_parameter
        workdir_cmd = f'#SBATCH -D {Path(workdir).absolute().as_posix()}\n' if workdir else ''
        node_cmd = f'#SBATCH --nodes={para.n_node}\n' if para.n_node > 0 else ''
        gpu_cmd = f'#SBATCH --gres=gpu:{para.n_gpu}\n' if para.n_gpu > 0 else ''
        dep_cmd = f'#SBATCH --dependency=afterok:{id_prior}\n' if id_prior is not None else ''  # id_prior can be 0
        exclude_cmd = f'#SBATCH --exclude={para.exclude}\n' if para.exclude else ''

        with open(sh, 'w') as f:
            f.write(f'#!/bin/bash\n'
                    f'#SBATCH -J {name}\n'
                    f'{workdir_cmd}'
                    f'#SBATCH -o {out}\n'
                    f'#SBATCH -e {err}\n'
                    f'{dep_cmd}'
                    f'#SBATCH -p {para.queue}\n'
                    f'#SBATCH --time={para.max_running_hour}:00:00\n'
                    f'{node_cmd}'
                    f'#SBATCH --ntasks={para.n_proc}\n'
                    f'{gpu_cmd}'
                    f'{exclude_cmd}'
                    f'\n'
                    f'{para.env_cmd}\n\n'
                    )
            for cmd in commands:
                f.write(cmd + '\n')

    def submit(self, sh=None, workdir=None):
        sh = sh or self.sh
        p = Popen([self.submit_cmd, sh], cwd=workdir, stdout=PIPE, stderr=PIPE)
        b_out, b_err = p.communicate()
        out, err = b_out.strip().decode(), b_err.strip().decode()
        if p.returncode == 0 and out.startswith('Submitted batch job'):
            logger.info(f'{out}')
            return int(out.split()[-1])
        else:
            if out:
                logger.error(f'{out}')
            if err:
                logger.error(f'{err}')
            if not out and not err:
                logger.error(f'sbatch failed')
            return -1

    def kill_job(self, name) -> bool:
        job = self.get_job_from_name(name)
        if job is None:
            return False

        cmd = f'scancel {job.id}'
        return subprocess.call(cmd.split()) == 0

    def _get_jobs(self):
        # Show all jobs. Then check the user
        cmd = 'scontrol show job'
        sp = Popen(cmd.split(), stdout=PIPE, stderr=PIPE)
        stdout, stderr = sp.communicate()
        if sp.returncode != 0:
            logger.error(stderr.decode())
            return []

        jobs = []
        for job_str in stdout.decode().split('\n\n'):  # split jobs
            if job_str.startswith('JobId'):
                job = self._get_job_from_str(job_str)
                # Show all jobs. Then check the user
                if job.user == self.username:
                    jobs.append(job)
        return jobs

    @staticmethod
    def _get_job_from_str(job_str) -> PbsJob:
        workdir = None
        for line in job_str.split():  # split properties
            try:
                key, val = line.split('=')[0:2]
            except:
                continue
            if key == 'JobId':
                id = int(val)
            elif key == 'UserId':
                user = val.split('(')[0]  # UserId=username(uid)
            elif key == 'JobName' or key == 'Name':
                name = val
            elif key == 'Partition':
                queue = val
            elif key == 'JobState':
                state_str = val
                if val in ('PENDING', 'RESV_DEL_HOLD'):
                    state = PbsJob.State.PENDING
                elif val in ('CONFIGURING', 'RUNNING', 'COMPLETING', 'STOPPED', 'SUSPENDED'):
                    state = PbsJob.State.RUNNING
                else:
                    state = PbsJob.State.DONE
            elif key == 'WorkDir':
                workdir = val
        job = PbsJob(id=id, name=name, state=state, workdir=workdir, user=user, queue=queue)
        job.state_str = state_str
        return job
