"use strict";
var _a, _b, _c;
Object.defineProperty(exports, "__esModule", { value: true });
exports.EcsRunTask = exports.EcsEc2LaunchTarget = exports.EcsFargateLaunchTarget = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const ec2 = require("@aws-cdk/aws-ec2");
const ecs = require("@aws-cdk/aws-ecs");
const iam = require("@aws-cdk/aws-iam");
const sfn = require("@aws-cdk/aws-stepfunctions");
const cdk = require("@aws-cdk/core");
const task_utils_1 = require("../private/task-utils");
/**
 * Configuration for running an ECS task on Fargate.
 *
 * @see https://docs.aws.amazon.com/AmazonECS/latest/userguide/launch_types.html#launch-type-fargate
 * @stability stable
 */
class EcsFargateLaunchTarget {
    /**
     * @stability stable
     */
    constructor(options) {
        this.options = options;
    }
    /**
     * Called when the Fargate launch type configured on RunTask.
     *
     * @stability stable
     */
    bind(_task, launchTargetOptions) {
        var _d;
        if (!launchTargetOptions.taskDefinition.isFargateCompatible) {
            throw new Error('Supplied TaskDefinition is not compatible with Fargate');
        }
        return {
            parameters: {
                LaunchType: 'FARGATE',
                PlatformVersion: (_d = this.options) === null || _d === void 0 ? void 0 : _d.platformVersion,
            },
        };
    }
}
exports.EcsFargateLaunchTarget = EcsFargateLaunchTarget;
_a = JSII_RTTI_SYMBOL_1;
EcsFargateLaunchTarget[_a] = { fqn: "@aws-cdk/aws-stepfunctions-tasks.EcsFargateLaunchTarget", version: "1.108.0" };
/**
 * Configuration for running an ECS task on EC2.
 *
 * @see https://docs.aws.amazon.com/AmazonECS/latest/userguide/launch_types.html#launch-type-ec2
 * @stability stable
 */
class EcsEc2LaunchTarget {
    /**
     * @stability stable
     */
    constructor(options) {
        this.options = options;
    }
    /**
     * Called when the EC2 launch type is configured on RunTask.
     *
     * @stability stable
     */
    bind(_task, launchTargetOptions) {
        var _d, _e, _f, _g, _h;
        if (!launchTargetOptions.taskDefinition.isEc2Compatible) {
            throw new Error('Supplied TaskDefinition is not compatible with EC2');
        }
        if (!((_d = launchTargetOptions.cluster) === null || _d === void 0 ? void 0 : _d.hasEc2Capacity)) {
            throw new Error('Cluster for this service needs Ec2 capacity. Call addCapacity() on the cluster.');
        }
        return {
            parameters: {
                LaunchType: 'EC2',
                // takes an array of placement constraints each of which contain a single item array of constraints, flattens it
                // and renders the Json to be passed as a parameter in the state machine.
                // input: [ecs.PlacementConstraint.distinctInstances()] - distinctInstances() returns [{ type: 'distinctInstance' }]
                // output: {Type: 'distinctInstance'}
                PlacementConstraints: noEmpty(flatten(((_f = (_e = this.options) === null || _e === void 0 ? void 0 : _e.placementConstraints) !== null && _f !== void 0 ? _f : []).map((c) => c.toJson().map(uppercaseKeys)))),
                PlacementStrategy: noEmpty(flatten(((_h = (_g = this.options) === null || _g === void 0 ? void 0 : _g.placementStrategies) !== null && _h !== void 0 ? _h : []).map((c) => c.toJson().map(uppercaseKeys)))),
            },
        };
        function uppercaseKeys(obj) {
            const ret = {};
            for (const key of Object.keys(obj)) {
                ret[key.slice(0, 1).toUpperCase() + key.slice(1)] = obj[key];
            }
            return ret;
        }
        function flatten(xs) {
            return Array.prototype.concat([], ...xs);
        }
        function noEmpty(xs) {
            if (xs.length === 0) {
                return undefined;
            }
            return xs;
        }
    }
}
exports.EcsEc2LaunchTarget = EcsEc2LaunchTarget;
_b = JSII_RTTI_SYMBOL_1;
EcsEc2LaunchTarget[_b] = { fqn: "@aws-cdk/aws-stepfunctions-tasks.EcsEc2LaunchTarget", version: "1.108.0" };
/**
 * Run a Task on ECS or Fargate.
 *
 * @stability stable
 */
class EcsRunTask extends sfn.TaskStateBase {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        var _d, _e;
        super(scope, id, props);
        this.props = props;
        /**
         * Manage allowed network traffic for this service.
         *
         * @stability stable
         */
        this.connections = new ec2.Connections();
        this.securityGroups = [];
        this.integrationPattern = (_d = props.integrationPattern) !== null && _d !== void 0 ? _d : sfn.IntegrationPattern.REQUEST_RESPONSE;
        task_utils_1.validatePatternSupported(this.integrationPattern, EcsRunTask.SUPPORTED_INTEGRATION_PATTERNS);
        if (this.integrationPattern === sfn.IntegrationPattern.WAIT_FOR_TASK_TOKEN && !sfn.FieldUtils.containsTaskToken(props.containerOverrides)) {
            throw new Error('Task Token is required in `containerOverrides` for callback. Use Context.taskToken to set the token.');
        }
        if (!this.props.taskDefinition.defaultContainer) {
            throw new Error('A TaskDefinition must have at least one essential container');
        }
        if (this.props.taskDefinition.networkMode === ecs.NetworkMode.AWS_VPC) {
            this.configureAwsVpcNetworking();
        }
        else {
            // Either None, Bridge or Host networking. Copy SecurityGroup from ASG.
            this.validateNoNetworkingProps();
            this.connections.addSecurityGroup(...this.props.cluster.connections.securityGroups);
        }
        for (const override of (_e = this.props.containerOverrides) !== null && _e !== void 0 ? _e : []) {
            const name = override.containerDefinition.containerName;
            if (!cdk.Token.isUnresolved(name)) {
                const cont = this.props.taskDefinition.node.tryFindChild(name);
                if (!cont) {
                    throw new Error(`Overrides mention container with name '${name}', but no such container in task definition`);
                }
            }
        }
        this.taskPolicies = this.makePolicyStatements();
    }
    /**
     * @internal
     */
    _renderTask() {
        return {
            Resource: task_utils_1.integrationResourceArn('ecs', 'runTask', this.integrationPattern),
            Parameters: sfn.FieldUtils.renderObject({
                Cluster: this.props.cluster.clusterArn,
                TaskDefinition: this.props.taskDefinition.family,
                NetworkConfiguration: this.networkConfiguration,
                Overrides: renderOverrides(this.props.containerOverrides),
                ...this.props.launchTarget.bind(this, { taskDefinition: this.props.taskDefinition, cluster: this.props.cluster }).parameters,
            }),
        };
    }
    configureAwsVpcNetworking() {
        var _d, _e;
        const subnetSelection = (_d = this.props.subnets) !== null && _d !== void 0 ? _d : { subnetType: this.props.assignPublicIp ? ec2.SubnetType.PUBLIC : ec2.SubnetType.PRIVATE };
        this.networkConfiguration = {
            AwsvpcConfiguration: {
                AssignPublicIp: this.props.assignPublicIp ? (this.props.assignPublicIp ? 'ENABLED' : 'DISABLED') : undefined,
                Subnets: this.props.cluster.vpc.selectSubnets(subnetSelection).subnetIds,
                SecurityGroups: cdk.Lazy.list({ produce: () => { var _d; return (_d = this.securityGroups) === null || _d === void 0 ? void 0 : _d.map(sg => sg.securityGroupId); } }),
            },
        };
        // Make sure we have a security group if we're using AWSVPC networking
        this.securityGroups = (_e = this.props.securityGroups) !== null && _e !== void 0 ? _e : [new ec2.SecurityGroup(this, 'SecurityGroup', { vpc: this.props.cluster.vpc })];
        this.connections.addSecurityGroup(...this.securityGroups);
    }
    validateNoNetworkingProps() {
        if (this.props.subnets !== undefined || this.props.securityGroups !== undefined) {
            throw new Error(`Supplied TaskDefinition must have 'networkMode' of 'AWS_VPC' to use 'vpcSubnets' and 'securityGroup'. Received: ${this.props.taskDefinition.networkMode}`);
        }
    }
    makePolicyStatements() {
        const stack = cdk.Stack.of(this);
        // https://docs.aws.amazon.com/step-functions/latest/dg/ecs-iam.html
        const policyStatements = [
            new iam.PolicyStatement({
                actions: ['ecs:RunTask'],
                resources: [this.getTaskDefinitionFamilyArn()],
            }),
            new iam.PolicyStatement({
                actions: ['ecs:StopTask', 'ecs:DescribeTasks'],
                resources: ['*'],
            }),
            new iam.PolicyStatement({
                actions: ['iam:PassRole'],
                resources: this.taskExecutionRoles().map((r) => r.roleArn),
            }),
        ];
        if (this.integrationPattern === sfn.IntegrationPattern.RUN_JOB) {
            policyStatements.push(new iam.PolicyStatement({
                actions: ['events:PutTargets', 'events:PutRule', 'events:DescribeRule'],
                resources: [
                    stack.formatArn({
                        service: 'events',
                        resource: 'rule',
                        resourceName: 'StepFunctionsGetEventsForECSTaskRule',
                    }),
                ],
            }));
        }
        return policyStatements;
    }
    /**
     * Returns the ARN of the task definition family by removing the
     * revision from the task definition ARN
     * Before - arn:aws:ecs:us-west-2:123456789012:task-definition/hello_world:8
     * After - arn:aws:ecs:us-west-2:123456789012:task-definition/hello_world
     */
    getTaskDefinitionFamilyArn() {
        const arnComponents = cdk.Stack.of(this).parseArn(this.props.taskDefinition.taskDefinitionArn);
        let { resourceName } = arnComponents;
        if (resourceName) {
            resourceName = resourceName.split(':')[0];
        }
        return cdk.Stack.of(this).formatArn({ ...arnComponents, resourceName });
    }
    taskExecutionRoles() {
        // Need to be able to pass both Task and Execution role, apparently
        const ret = new Array();
        ret.push(this.props.taskDefinition.taskRole);
        if (this.props.taskDefinition.executionRole) {
            ret.push(this.props.taskDefinition.executionRole);
        }
        return ret;
    }
}
exports.EcsRunTask = EcsRunTask;
_c = JSII_RTTI_SYMBOL_1;
EcsRunTask[_c] = { fqn: "@aws-cdk/aws-stepfunctions-tasks.EcsRunTask", version: "1.108.0" };
EcsRunTask.SUPPORTED_INTEGRATION_PATTERNS = [
    sfn.IntegrationPattern.REQUEST_RESPONSE,
    sfn.IntegrationPattern.RUN_JOB,
    sfn.IntegrationPattern.WAIT_FOR_TASK_TOKEN,
];
function renderOverrides(containerOverrides) {
    var _d;
    if (!containerOverrides || containerOverrides.length === 0) {
        return undefined;
    }
    const ret = new Array();
    for (const override of containerOverrides) {
        ret.push({
            Name: override.containerDefinition.containerName,
            Command: override.command,
            Cpu: override.cpu,
            Memory: override.memoryLimit,
            MemoryReservation: override.memoryReservation,
            Environment: (_d = override.environment) === null || _d === void 0 ? void 0 : _d.map((e) => ({
                Name: e.name,
                Value: e.value,
            })),
        });
    }
    return { ContainerOverrides: ret };
}
//# sourceMappingURL=data:application/json;base64,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