import * as ec2 from '@aws-cdk/aws-ec2';
import * as iam from '@aws-cdk/aws-iam';
import * as sfn from '@aws-cdk/aws-stepfunctions';
import { Size } from '@aws-cdk/core';
import { Construct } from 'constructs';
/**
 * The overrides that should be sent to a container.
 *
 * @stability stable
 */
export interface BatchContainerOverrides {
    /**
     * The command to send to the container that overrides the default command from the Docker image or the job definition.
     *
     * @default - No command overrides
     * @stability stable
     */
    readonly command?: string[];
    /**
     * The environment variables to send to the container.
     *
     * You can add new environment variables, which are added to the container
     * at launch, or you can override the existing environment variables from
     * the Docker image or the job definition.
     *
     * @default - No environment overrides
     * @stability stable
     */
    readonly environment?: {
        [key: string]: string;
    };
    /**
     * The instance type to use for a multi-node parallel job.
     *
     * This parameter is not valid for single-node container jobs.
     *
     * @default - No instance type overrides
     * @stability stable
     */
    readonly instanceType?: ec2.InstanceType;
    /**
     * Memory reserved for the job.
     *
     * @default - No memory overrides. The memory supplied in the job definition will be used.
     * @stability stable
     */
    readonly memory?: Size;
    /**
     * The number of physical GPUs to reserve for the container.
     *
     * The number of GPUs reserved for all containers in a job
     * should not exceed the number of available GPUs on the compute
     * resource that the job is launched on.
     *
     * @default - No GPU reservation
     * @stability stable
     */
    readonly gpuCount?: number;
    /**
     * The number of vCPUs to reserve for the container.
     *
     * This value overrides the value set in the job definition.
     *
     * @default - No vCPUs overrides
     * @stability stable
     */
    readonly vcpus?: number;
}
/**
 * An object representing an AWS Batch job dependency.
 *
 * @stability stable
 */
export interface BatchJobDependency {
    /**
     * The job ID of the AWS Batch job associated with this dependency.
     *
     * @default - No jobId
     * @stability stable
     */
    readonly jobId?: string;
    /**
     * The type of the job dependency.
     *
     * @default - No type
     * @stability stable
     */
    readonly type?: string;
}
/**
 * Properties for RunBatchJob.
 *
 * @stability stable
 */
export interface BatchSubmitJobProps extends sfn.TaskStateBaseProps {
    /**
     * The arn of the job definition used by this job.
     *
     * @stability stable
     */
    readonly jobDefinitionArn: string;
    /**
     * The name of the job.
     *
     * The first character must be alphanumeric, and up to 128 letters (uppercase and lowercase),
     * numbers, hyphens, and underscores are allowed.
     *
     * @stability stable
     */
    readonly jobName: string;
    /**
     * The arn of the job queue into which the job is submitted.
     *
     * @stability stable
     */
    readonly jobQueueArn: string;
    /**
     * The array size can be between 2 and 10,000.
     *
     * If you specify array properties for a job, it becomes an array job.
     * For more information, see Array Jobs in the AWS Batch User Guide.
     *
     * @default - No array size
     * @stability stable
     */
    readonly arraySize?: number;
    /**
     * A list of container overrides in JSON format that specify the name of a container in the specified job definition and the overrides it should receive.
     *
     * @default - No container overrides
     * @see https://docs.aws.amazon.com/batch/latest/APIReference/API_SubmitJob.html#Batch-SubmitJob-request-containerOverrides
     * @stability stable
     */
    readonly containerOverrides?: BatchContainerOverrides;
    /**
     * A list of dependencies for the job.
     *
     * A job can depend upon a maximum of 20 jobs.
     *
     * @default - No dependencies
     * @see https://docs.aws.amazon.com/batch/latest/APIReference/API_SubmitJob.html#Batch-SubmitJob-request-dependsOn
     * @stability stable
     */
    readonly dependsOn?: BatchJobDependency[];
    /**
     * The payload to be passed as parameters to the batch job.
     *
     * @default - No parameters are passed
     * @stability stable
     */
    readonly payload?: sfn.TaskInput;
    /**
     * The number of times to move a job to the RUNNABLE status.
     *
     * You may specify between 1 and 10 attempts.
     * If the value of attempts is greater than one,
     * the job is retried on failure the same number of attempts as the value.
     *
     * @default 1
     * @stability stable
     */
    readonly attempts?: number;
}
/**
 * Task to submits an AWS Batch job from a job definition.
 *
 * @see https://docs.aws.amazon.com/step-functions/latest/dg/connect-batch.html
 * @stability stable
 */
export declare class BatchSubmitJob extends sfn.TaskStateBase {
    private readonly props;
    private static readonly SUPPORTED_INTEGRATION_PATTERNS;
    /**
     * @stability stable
     */
    protected readonly taskMetrics?: sfn.TaskMetricsConfig;
    /**
     * @stability stable
     */
    protected readonly taskPolicies?: iam.PolicyStatement[];
    private readonly integrationPattern;
    /**
     * @stability stable
     */
    constructor(scope: Construct, id: string, props: BatchSubmitJobProps);
    /**
     * @internal
     */
    protected _renderTask(): any;
    private configurePolicyStatements;
    private configureContainerOverrides;
}
