"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CrossRegionSupportStack = exports.CrossRegionSupportConstruct = void 0;
const kms = require("@aws-cdk/aws-kms");
const s3 = require("@aws-cdk/aws-s3");
const cdk = require("@aws-cdk/core");
// keep this import separate from other imports to reduce chance for merge conflicts with v2-main
// eslint-disable-next-line no-duplicate-imports, import/order
const core_1 = require("@aws-cdk/core");
const REQUIRED_ALIAS_PREFIX = 'alias/';
/**
 * A class needed to work around CodePipeline's extremely small (100 characters)
 * limit for the name/ARN of the key in the ArtifactStore.
 * Limits the length of the alias' auto-generated name to 50 characters.
 */
class AliasWithShorterGeneratedName extends kms.Alias {
    generatePhysicalName() {
        let baseName = super.generatePhysicalName();
        if (baseName.startsWith(REQUIRED_ALIAS_PREFIX)) {
            // remove the prefix, because we're taking the last characters of the name below
            baseName = baseName.substring(REQUIRED_ALIAS_PREFIX.length);
        }
        const maxLength = 50 - REQUIRED_ALIAS_PREFIX.length;
        // take the last characters, as they include the hash,
        // and so have a higher chance of not colliding
        return REQUIRED_ALIAS_PREFIX + lastNCharacters(baseName, maxLength);
    }
}
function lastNCharacters(str, n) {
    const startIndex = Math.max(str.length - n, 0);
    return str.substring(startIndex);
}
class CrossRegionSupportConstruct extends core_1.Construct {
    constructor(scope, id, props = {}) {
        var _a;
        super(scope, id);
        const createKmsKey = (_a = props.createKmsKey) !== null && _a !== void 0 ? _a : true;
        let encryptionAlias;
        if (createKmsKey) {
            const encryptionKey = new kms.Key(this, 'CrossRegionCodePipelineReplicationBucketEncryptionKey', {
                removalPolicy: cdk.RemovalPolicy.DESTROY,
                enableKeyRotation: props.enableKeyRotation,
            });
            encryptionAlias = new AliasWithShorterGeneratedName(this, 'CrossRegionCodePipelineReplicationBucketEncryptionAlias', {
                targetKey: encryptionKey,
                aliasName: cdk.PhysicalName.GENERATE_IF_NEEDED,
                removalPolicy: cdk.RemovalPolicy.DESTROY,
            });
        }
        this.replicationBucket = new s3.Bucket(this, 'CrossRegionCodePipelineReplicationBucket', {
            bucketName: cdk.PhysicalName.GENERATE_IF_NEEDED,
            encryption: encryptionAlias ? s3.BucketEncryption.KMS : s3.BucketEncryption.KMS_MANAGED,
            encryptionKey: encryptionAlias,
            blockPublicAccess: s3.BlockPublicAccess.BLOCK_ALL,
        });
    }
}
exports.CrossRegionSupportConstruct = CrossRegionSupportConstruct;
/**
 * A Stack containing resources required for the cross-region CodePipeline functionality to work.
 * This class is private to the aws-codepipeline package.
 */
class CrossRegionSupportStack extends cdk.Stack {
    constructor(scope, id, props) {
        super(scope, id, {
            stackName: generateStackName(props),
            env: {
                region: props.region,
                account: props.account,
            },
            synthesizer: props.synthesizer,
        });
        const crossRegionSupportConstruct = new CrossRegionSupportConstruct(this, 'Default', {
            createKmsKey: props.createKmsKey,
            enableKeyRotation: props.enableKeyRotation,
        });
        this.replicationBucket = crossRegionSupportConstruct.replicationBucket;
    }
}
exports.CrossRegionSupportStack = CrossRegionSupportStack;
function generateStackName(props) {
    return `${props.pipelineStackName}-support-${props.region}`;
}
//# sourceMappingURL=data:application/json;base64,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