# coding: utf-8
from django.apps import apps
from django.core.exceptions import FieldDoesNotExist
from django.core.management.base import BaseCommand
from django.core.management.base import CommandError

from ... import clear_table


class Command(BaseCommand):
    """Удаляет записи из таблицы БД по условию.

    С помощью данной команды удаляются записи из основной (не секционированной)
    таблицы, у которых значение в field_name меньше значения из before_value.
    Подробнее см. в `django_postgres_partitioning.clear_table`.

    """
    help = (
        'Command deletes all the records from database table when '
        'field_name < before_value.'
    )

    def add_arguments(self, parser):
        parser.add_argument(
            'app_label',
            help='App label of an application.',
        )
        parser.add_argument(
            'model_name',
            help='Model name.',
        )
        parser.add_argument(
            'field_name',
            help='Field name. It will be a check column.',
        )
        parser.add_argument(
            'before_value',
            help='Deleting rows before this value.',
        )
        parser.add_argument(
            '--timeout', action='store', dest='timeout',
            default=.0, type=float,
            help=('Timeout (in seconds) between the data removes iterations. '
                  'It used to reduce the database load.')
        )

    def handle(self, *args, **options):
        app_label = options['app_label']
        model_name = options['model_name']
        field_name = options['field_name']
        before_value = options['before_value']
        timeout = options['timeout']

        try:
            model = apps.get_model(app_label, model_name)
        except LookupError as e:
            raise CommandError(e.message)

        try:
            model._meta.get_field(field_name)
        except FieldDoesNotExist:
            raise CommandError('Invalid field name ({0})'.format(field_name))

        clear_table(model, field_name, before_value, timeout)
