"""Everything regarding execution of the built app."""

from __future__ import annotations

import hashlib
import json
import os
import asyncio
import platform
import re
import sys
from pathlib import Path
from urllib.parse import urljoin
import jurigged

import psutil
import uvicorn

from nextpy import constants
from nextpy.core.config import get_config
from nextpy.utils.source_reload import watch
from nextpy.utils import console, path_ops, prerequisites, processes
from nextpy.utils.watch import AssetFolderWatch

from concurrent.futures import ThreadPoolExecutor, wait, as_completed, FIRST_EXCEPTION


def start_watching_assets_folder(root):
    """Start watching assets folder.

    Args:
        root: root path of the project.
    """
    asset_watch = AssetFolderWatch(root)
    asset_watch.start()


def detect_package_change(json_file_path: str) -> str:
    """Calculates the SHA-256 hash of a JSON file and returns it as a hexadecimal string.

    Args:
        json_file_path: The path to the JSON file to be hashed.

    Returns:
        str: The SHA-256 hash of the JSON file as a hexadecimal string.

    Example:
        >>> detect_package_change("package.json")
        'a1b2c3d4e5f6g7h8i9j0k1l2m3n4o5p6q7r8s9t0u1v2w3x4y5z6a7b8c9d0e1f2'
    """
    with open(json_file_path, "r") as file:
        json_data = json.load(file)

    # Calculate the hash
    json_string = json.dumps(json_data, sort_keys=True)
    hash_object = hashlib.sha256(json_string.encode())
    return hash_object.hexdigest()


def kill(proc_pid: int):
    """Kills a process and all its child processes.

    Args:
        proc_pid (int): The process ID of the process to be killed.

    Example:
        >>> kill(1234)
    """
    process = psutil.Process(proc_pid)
    for proc in process.children(recursive=True):
        proc.kill()
    process.kill()


def run_process_and_launch_url(run_command: list[str]):
    """Run the process and launch the URL.

    Args:
        run_command: The command to run.
    """
    config = get_config()
    watch("./" + get_config().app_name + "/", ".".join([config.app_name, config.app_name]))

    # TODO: This ThreadPoolExecutor is the residual of the previous approach where 
    # the source reload had to run in parallel to the handleUpdates
    # This is currently not required and can be cleaned up
    with ThreadPoolExecutor(max_workers=10) as executor:
        handleUpdatesTask = executor.submit(handle_updates, run_command)
        while True:
            tasks = wait([handleUpdatesTask], return_when=FIRST_EXCEPTION)
            for task in tasks.done:
                print(task.exception())

def handle_updates(run_command):
    json_file_path = os.path.join(constants.Dirs.WEB, "package.json")
    last_hash = detect_package_change(json_file_path)
    process = None
    first_run = True
    should_break = False
    should_restart = True

    while True:
        if should_restart is True:
               process = processes.new_process(
                    run_command, cwd=constants.Dirs.WEB, shell=constants.IS_WINDOWS
                )
               should_restart = False
        
        if should_break is True:
            break

        for line in processes.stream_logs("Starting frontend", process):
            if line is None:
                should_restart = True
                break
            match = re.search(constants.Next.FRONTEND_LISTENING_REGEX, line)
            if match:
                url = match.group(1)
                if get_config().frontend_path != "":
                    url = urljoin(url, get_config().frontend_path)
                if first_run is False:
                    console.print("Changes were detected")

                console.print(f"App running at: [bold green]{url}")
                first_run = False
            else:
                new_hash = detect_package_change(json_file_path)
                if new_hash != last_hash:
                    last_hash = new_hash
                    print("Killing process with PID ", process.pid)
                    kill(process.pid)
                    should_break = True

def run_frontend(root: Path, port: str):
    """Run the frontend.

    Args:
        root: The root path of the project.
        port: The port to run the frontend on.
    """
    # Start watching asset folder.
    start_watching_assets_folder(root)
    # validate dependencies before run
    prerequisites.validate_frontend_dependencies(init=False)
    # Run the frontend in development mode.
    console.rule("[bold green]App Running")
    os.environ["PORT"] = str(get_config().frontend_port if port is None else port)
    run_process_and_launch_url([prerequisites.get_package_manager(), "run", "dev"])  # type: ignore


def run_frontend_prod(root: Path, port: str):
    """Run the frontend.

    Args:
        root: The root path of the project (to keep same API as run_frontend).
        port: The port to run the frontend on.
    """
    # Set the port.
    os.environ["PORT"] = str(get_config().frontend_port if port is None else port)
    # validate dependencies before run
    prerequisites.validate_frontend_dependencies(init=False)
    # Run the frontend in production mode.
    console.rule("[bold green]App Running")
    run_process_and_launch_url([prerequisites.get_package_manager(), "run", "prod"])  # type: ignore


def run_backend(
    host: str,
    port: int,
    loglevel: constants.LogLevel = constants.LogLevel.ERROR,
):
    """Run the backend.

    Args:
        host: The app host
        port: The app port
        loglevel: The log level.
    """
    config = get_config()
    app_module = f"{config.app_name}.{config.app_name}:{constants.CompileVars.APP}"

    #TODO: Handle the reloads for the backend API

    # Create a .nocompile file to skip compile for backend.
    if os.path.exists(constants.Dirs.WEB):
        with open(constants.NOCOMPILE_FILE, "w"):
            pass

    # Run the backend in development mode.
    uvicorn.run(
        app=f"{app_module}.{constants.CompileVars.API}",
        host=host,
        port=port,
        log_level=loglevel.value,
    )


def run_backend_prod(
    host: str,
    port: int,
    loglevel: constants.LogLevel = constants.LogLevel.ERROR,
):
    """Run the backend.

    Args:
        host: The app host
        port: The app port
        loglevel: The log level.
    """
    num_workers = processes.get_num_workers()
    config = get_config()
    RUN_BACKEND_PROD = f"gunicorn --worker-class {config.gunicorn_worker_class} --preload --timeout {config.timeout} --log-level critical".split()
    RUN_BACKEND_PROD_WINDOWS = f"uvicorn --timeout-keep-alive {config.timeout}".split()
    app_module = f"{config.app_name}.{config.app_name}:{constants.CompileVars.APP}"
    command = (
        [
            *RUN_BACKEND_PROD_WINDOWS,
            "--host",
            host,
            "--port",
            str(port),
            app_module,
        ]
        if constants.IS_WINDOWS
        else [
            *RUN_BACKEND_PROD,
            "--bind",
            f"{host}:{port}",
            "--threads",
            str(num_workers),
            f"{app_module}()",
        ]
    )

    command += [
        "--log-level",
        loglevel.value,
        "--workers",
        str(num_workers),
    ]
    processes.new_process(
        command,
        run=True,
        show_logs=True,
        env={constants.SKIP_COMPILE_ENV_VAR: "yes"},  # skip compile for prod backend
    )


def output_system_info():
    """Show system information if the loglevel is in DEBUG."""
    if console._LOG_LEVEL > constants.LogLevel.DEBUG:
        return

    config = get_config()
    try:
        config_file = sys.modules[config.__module__].__file__
    except Exception:
        config_file = None

    console.rule(f"System Info")
    console.debug(f"Config file: {config_file!r}")
    console.debug(f"Config: {config}")

    dependencies = [
        f"[Nextpy {constants.Nextpy.VERSION} with Python {platform.python_version()} (PATH: {sys.executable})]",
        f"[Node {prerequisites.get_node_version()} (Expected: {constants.Node.VERSION}) (PATH:{path_ops.get_node_path()})]",
    ]

    system = platform.system()

    if system != "Windows":
        dependencies.extend(
            [
                f"[FNM {prerequisites.get_fnm_version()} (Expected: {constants.Fnm.VERSION}) (PATH: {constants.Fnm.EXE})]",
                f"[Bun {prerequisites.get_bun_version()} (Expected: {constants.Bun.VERSION}) (PATH: {config.bun_path})]",
            ],
        )
    else:
        dependencies.append(
            f"[FNM {prerequisites.get_fnm_version()} (Expected: {constants.Fnm.VERSION}) (PATH: {constants.Fnm.EXE})]",
        )

    if system == "Linux":
        import distro  # type: ignore

        os_version = distro.name(pretty=True)
    else:
        os_version = platform.version()

    dependencies.append(f"[OS {platform.system()} {os_version}]")

    for dep in dependencies:
        console.debug(f"{dep}")

    console.debug(
        f"Using package installer at: {prerequisites.get_install_package_manager()}"  # type: ignore
    )
    console.debug(f"Using package executer at: {prerequisites.get_package_manager()}")  # type: ignore
    if system != "Windows":
        console.debug(f"Unzip path: {path_ops.which('unzip')}")
