"use strict";
const assert_1 = require("@aws-cdk/assert");
const iam = require("@aws-cdk/aws-iam");
const kms = require("@aws-cdk/aws-kms");
const s3 = require("@aws-cdk/aws-s3");
const cdk = require("@aws-cdk/core");
const codepipeline = require("../lib");
const fake_build_action_1 = require("./fake-build-action");
const fake_source_action_1 = require("./fake-source-action");
module.exports = {
    'Pipeline': {
        'can be passed an IAM role during pipeline creation'(test) {
            const stack = new cdk.Stack();
            const role = new iam.Role(stack, 'Role', {
                assumedBy: new iam.ServicePrincipal('codepipeline.amazonaws.com')
            });
            new codepipeline.Pipeline(stack, 'Pipeline', {
                role
            });
            assert_1.expect(stack, true).to(assert_1.haveResourceLike('AWS::CodePipeline::Pipeline', {
                "RoleArn": {
                    "Fn::GetAtt": [
                        "Role1ABCC5F0",
                        "Arn",
                    ]
                }
            }));
            test.done();
        },
        'can be imported by ARN'(test) {
            const stack = new cdk.Stack();
            const pipeline = codepipeline.Pipeline.fromPipelineArn(stack, 'Pipeline', 'arn:aws:codepipeline:us-east-1:123456789012:MyPipeline');
            test.equal(pipeline.pipelineArn, 'arn:aws:codepipeline:us-east-1:123456789012:MyPipeline');
            test.equal(pipeline.pipelineName, 'MyPipeline');
            test.done();
        },
        'that is cross-region': {
            'validates that source actions are in the same account as the pipeline'(test) {
                const app = new cdk.App();
                const stack = new cdk.Stack(app, 'PipelineStack', { env: { region: 'us-west-1', account: '123456789012' } });
                const pipeline = new codepipeline.Pipeline(stack, 'Pipeline');
                const sourceStage = pipeline.addStage({
                    stageName: 'Source',
                });
                const sourceAction = new fake_source_action_1.FakeSourceAction({
                    actionName: 'FakeSource',
                    output: new codepipeline.Artifact(),
                    region: 'ap-southeast-1',
                });
                test.throws(() => {
                    sourceStage.addAction(sourceAction);
                }, /Source action 'FakeSource' must be in the same region as the pipeline/);
                test.done();
            },
            'allows passing an Alias in place of the KMS Key in the replication Bucket'(test) {
                const app = new cdk.App();
                const replicationRegion = 'us-west-1';
                const replicationStack = new cdk.Stack(app, 'ReplicationStack', {
                    env: { region: replicationRegion, account: '123456789012' },
                });
                const replicationKey = new kms.Key(replicationStack, 'ReplicationKey');
                const replicationAlias = replicationKey.addAlias('alias/my-replication-alias');
                const replicationBucket = new s3.Bucket(replicationStack, 'ReplicationBucket', {
                    encryptionKey: replicationAlias,
                    bucketName: cdk.PhysicalName.GENERATE_IF_NEEDED,
                });
                const pipelineRegion = 'us-west-2';
                const pipelineStack = new cdk.Stack(app, 'PipelineStack', {
                    env: { region: pipelineRegion, account: '123456789012' },
                });
                const sourceOutput = new codepipeline.Artifact();
                new codepipeline.Pipeline(pipelineStack, 'Pipeline', {
                    crossRegionReplicationBuckets: {
                        [replicationRegion]: replicationBucket,
                    },
                    stages: [
                        {
                            stageName: 'Source',
                            actions: [new fake_source_action_1.FakeSourceAction({
                                    actionName: 'Source',
                                    output: sourceOutput,
                                })],
                        },
                        {
                            stageName: 'Build',
                            actions: [new fake_build_action_1.FakeBuildAction({
                                    actionName: 'Build',
                                    input: sourceOutput,
                                    region: replicationRegion,
                                })],
                        },
                    ],
                });
                assert_1.expect(pipelineStack).to(assert_1.haveResourceLike('AWS::CodePipeline::Pipeline', {
                    "ArtifactStores": [
                        {
                            "Region": replicationRegion,
                            "ArtifactStore": {
                                "Type": "S3",
                                "EncryptionKey": {
                                    "Type": "KMS",
                                    "Id": {
                                        "Fn::Join": [
                                            "",
                                            [
                                                "arn:",
                                                {
                                                    "Ref": "AWS::Partition",
                                                },
                                                ":kms:us-west-1:123456789012:alias/my-replication-alias",
                                            ],
                                        ],
                                    },
                                },
                            },
                        },
                        {
                            "Region": pipelineRegion,
                        },
                    ],
                }));
                assert_1.expect(replicationStack).to(assert_1.haveResourceLike('AWS::KMS::Key', {
                    "KeyPolicy": {
                        "Statement": [
                            {
                            // owning account management permissions - we don't care about them in this test
                            },
                            {
                                // KMS verifies whether the principal given in its key policy exists when creating that key.
                                // Since the replication bucket must be deployed before the pipeline,
                                // we cannot put the pipeline role as the principal here -
                                // hence, we put the account itself
                                "Action": [
                                    "kms:Decrypt",
                                    "kms:DescribeKey",
                                    "kms:Encrypt",
                                    "kms:ReEncrypt*",
                                    "kms:GenerateDataKey*",
                                ],
                                "Effect": "Allow",
                                "Principal": {
                                    "AWS": {
                                        "Fn::Join": ["", [
                                                "arn:",
                                                { "Ref": "AWS::Partition" },
                                                ":iam::123456789012:root",
                                            ]],
                                    },
                                },
                                "Resource": "*",
                            },
                        ],
                    },
                }));
                test.done();
            },
            "generates ArtifactStores with the alias' ARN as the KeyID"(test) {
                const app = new cdk.App();
                const replicationRegion = 'us-west-1';
                const pipelineRegion = 'us-west-2';
                const pipelineStack = new cdk.Stack(app, 'MyStack', {
                    env: { region: pipelineRegion, account: '123456789012' },
                });
                const sourceOutput = new codepipeline.Artifact();
                const pipeline = new codepipeline.Pipeline(pipelineStack, 'Pipeline', {
                    stages: [
                        {
                            stageName: 'Source',
                            actions: [new fake_source_action_1.FakeSourceAction({
                                    actionName: 'Source',
                                    output: sourceOutput,
                                })],
                        },
                        {
                            stageName: 'Build',
                            actions: [new fake_build_action_1.FakeBuildAction({
                                    actionName: 'Build',
                                    input: sourceOutput,
                                    region: replicationRegion,
                                })],
                        },
                    ],
                });
                assert_1.expect(pipelineStack).to(assert_1.haveResourceLike('AWS::CodePipeline::Pipeline', {
                    "ArtifactStores": [
                        {
                            "Region": replicationRegion,
                            "ArtifactStore": {
                                "Type": "S3",
                                "EncryptionKey": {
                                    "Type": "KMS",
                                    "Id": {
                                        "Fn::Join": [
                                            "",
                                            [
                                                "arn:",
                                                {
                                                    "Ref": "AWS::Partition",
                                                },
                                                ":kms:us-west-1:123456789012:alias/s-west-1tencryptionalias9b344b2b8e6825cb1f7d",
                                            ],
                                        ],
                                    },
                                },
                            },
                        },
                        {
                            "Region": pipelineRegion,
                        },
                    ],
                }));
                assert_1.expect(pipeline.crossRegionSupport[replicationRegion].stack).to(assert_1.haveResourceLike('AWS::KMS::Alias', {
                    "DeletionPolicy": "Delete",
                    "UpdateReplacePolicy": "Delete",
                }, assert_1.ResourcePart.CompleteDefinition));
                test.done();
            },
            'allows passing an imported Bucket and Key for the replication Bucket'(test) {
                const replicationRegion = 'us-west-1';
                const pipelineRegion = 'us-west-2';
                const pipelineStack = new cdk.Stack(undefined, undefined, {
                    env: { region: pipelineRegion },
                });
                const sourceOutput = new codepipeline.Artifact();
                new codepipeline.Pipeline(pipelineStack, 'Pipeline', {
                    crossRegionReplicationBuckets: {
                        [replicationRegion]: s3.Bucket.fromBucketAttributes(pipelineStack, 'ReplicationBucket', {
                            bucketArn: 'arn:aws:s3:::my-us-west-1-replication-bucket',
                            encryptionKey: kms.Key.fromKeyArn(pipelineStack, 'ReplicationKey', `arn:aws:kms:${replicationRegion}:123456789012:key/1234-5678-9012`),
                        }),
                    },
                    stages: [
                        {
                            stageName: 'Source',
                            actions: [new fake_source_action_1.FakeSourceAction({
                                    actionName: 'Source',
                                    output: sourceOutput,
                                })],
                        },
                        {
                            stageName: 'Build',
                            actions: [new fake_build_action_1.FakeBuildAction({
                                    actionName: 'Build',
                                    input: sourceOutput,
                                })],
                        },
                    ],
                });
                assert_1.expect(pipelineStack).to(assert_1.haveResourceLike('AWS::CodePipeline::Pipeline', {
                    "ArtifactStores": [
                        {
                            "Region": replicationRegion,
                            "ArtifactStore": {
                                "Type": "S3",
                                "Location": "my-us-west-1-replication-bucket",
                                "EncryptionKey": {
                                    "Type": "KMS",
                                    "Id": "arn:aws:kms:us-west-1:123456789012:key/1234-5678-9012",
                                },
                            },
                        },
                        {
                            "Region": pipelineRegion,
                        },
                    ],
                }));
                test.done();
            },
        },
        'that is cross-account': {
            'does not allow passing a dynamic value in the Action account property'(test) {
                const app = new cdk.App();
                const stack = new cdk.Stack(app, 'PipelineStack', { env: { account: '123456789012' } });
                const sourceOutput = new codepipeline.Artifact();
                const pipeline = new codepipeline.Pipeline(stack, 'Pipeline', {
                    stages: [
                        {
                            stageName: 'Source',
                            actions: [new fake_source_action_1.FakeSourceAction({ actionName: 'Source', output: sourceOutput })],
                        },
                    ],
                });
                const buildStage = pipeline.addStage({ stageName: 'Build' });
                test.throws(() => {
                    buildStage.addAction(new fake_build_action_1.FakeBuildAction({
                        actionName: 'FakeBuild',
                        input: sourceOutput,
                        account: cdk.Aws.ACCOUNT_ID,
                    }));
                }, /The 'account' property must be a concrete value \(action: 'FakeBuild'\)/);
                test.done();
            },
            'does not allow an env-agnostic Pipeline Stack if an Action account has been provided'(test) {
                const app = new cdk.App();
                const stack = new cdk.Stack(app, 'PipelineStack');
                const sourceOutput = new codepipeline.Artifact();
                const pipeline = new codepipeline.Pipeline(stack, 'Pipeline', {
                    stages: [
                        {
                            stageName: 'Source',
                            actions: [new fake_source_action_1.FakeSourceAction({ actionName: 'Source', output: sourceOutput })],
                        },
                    ],
                });
                const buildStage = pipeline.addStage({ stageName: 'Build' });
                test.throws(() => {
                    buildStage.addAction(new fake_build_action_1.FakeBuildAction({
                        actionName: 'FakeBuild',
                        input: sourceOutput,
                        account: '123456789012',
                    }));
                }, /Pipeline stack which uses cross-environment actions must have an explicitly set account/);
                test.done();
            },
        },
    },
};
//# sourceMappingURL=data:application/json;base64,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