"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const events = require("@aws-cdk/aws-events");
const cdk = require("@aws-cdk/core");
const validation = require("./validation");
/**
 * A Stage in a Pipeline.
 *
 * Stages are added to a Pipeline by calling {@link Pipeline#addStage},
 * which returns an instance of {@link codepipeline.IStage}.
 *
 * This class is private to the CodePipeline module.
 */
class Stage {
    /**
     * Create a new Stage.
     */
    constructor(props, pipeline) {
        this._actions = new Array();
        validation.validateName('Stage', props.stageName);
        this.stageName = props.stageName;
        this._pipeline = pipeline;
        this.scope = new cdk.Construct(pipeline, this.stageName);
        for (const action of props.actions || []) {
            this.addAction(action);
        }
    }
    /**
     * Get a duplicate of this stage's list of actions.
     */
    get actionDescriptors() {
        return this._actions.slice();
    }
    get actions() {
        return this._actions.map(actionDescriptor => actionDescriptor.action);
    }
    get pipeline() {
        return this._pipeline;
    }
    render() {
        // first, assign names to output Artifacts who don't have one
        for (const action of this._actions) {
            const outputArtifacts = action.outputs;
            const unnamedOutputs = outputArtifacts.filter(o => !o.artifactName);
            for (const outputArtifact of outputArtifacts) {
                if (!outputArtifact.artifactName) {
                    const unsanitizedArtifactName = `Artifact_${this.stageName}_${action.actionName}` + (unnamedOutputs.length === 1
                        ? ''
                        : '_' + (unnamedOutputs.indexOf(outputArtifact) + 1));
                    const artifactName = sanitizeArtifactName(unsanitizedArtifactName);
                    outputArtifact._setName(artifactName);
                }
            }
        }
        return {
            name: this.stageName,
            actions: this._actions.map(action => this.renderAction(action)),
        };
    }
    addAction(action) {
        const actionName = action.actionProperties.actionName;
        // validate the name
        validation.validateName('Action', actionName);
        // check for duplicate Actions and names
        if (this._actions.find(a => a.actionName === actionName)) {
            throw new Error(`Stage ${this.stageName} already contains an action with name '${actionName}'`);
        }
        this._actions.push(this.attachActionToPipeline(action));
    }
    onStateChange(name, target, options) {
        const rule = new events.Rule(this.scope, name, options);
        rule.addTarget(target);
        rule.addEventPattern({
            detailType: ['CodePipeline Stage Execution State Change'],
            source: ['aws.codepipeline'],
            resources: [this.pipeline.pipelineArn],
            detail: {
                stage: [this.stageName],
            },
        });
        return rule;
    }
    validate() {
        return [
            ...this.validateHasActions(),
            ...this.validateActions(),
        ];
    }
    validateHasActions() {
        if (this._actions.length === 0) {
            return [`Stage '${this.stageName}' must have at least one action`];
        }
        return [];
    }
    validateActions() {
        const ret = new Array();
        for (const action of this.actionDescriptors) {
            ret.push(...this.validateAction(action));
        }
        return ret;
    }
    validateAction(action) {
        return validation.validateArtifactBounds('input', action.inputs, action.artifactBounds.minInputs, action.artifactBounds.maxInputs, action.category, action.provider)
            .concat(validation.validateArtifactBounds('output', action.outputs, action.artifactBounds.minOutputs, action.artifactBounds.maxOutputs, action.category, action.provider));
    }
    attachActionToPipeline(action) {
        // notify the Pipeline of the new Action
        const actionScope = new cdk.Construct(this.scope, action.actionProperties.actionName);
        return this._pipeline._attachActionToPipeline(this, action, actionScope);
    }
    renderAction(action) {
        const outputArtifacts = cdk.Lazy.anyValue({ produce: () => this.renderArtifacts(action.outputs) }, { omitEmptyArray: true });
        const inputArtifacts = cdk.Lazy.anyValue({ produce: () => this.renderArtifacts(action.inputs) }, { omitEmptyArray: true });
        return {
            name: action.actionName,
            inputArtifacts,
            outputArtifacts,
            actionTypeId: {
                category: action.category.toString(),
                version: action.version,
                owner: action.owner,
                provider: action.provider,
            },
            configuration: action.configuration,
            runOrder: action.runOrder,
            roleArn: action.role ? action.role.roleArn : undefined,
            region: action.region,
            namespace: action.namespace,
        };
    }
    renderArtifacts(artifacts) {
        return artifacts
            .filter(a => a.artifactName)
            .map(a => ({ name: a.artifactName }));
    }
}
exports.Stage = Stage;
function sanitizeArtifactName(artifactName) {
    // strip out some characters that are legal in Stage and Action names,
    // but not in Artifact names
    return artifactName.replace(/[@.]/g, '');
}
//# sourceMappingURL=data:application/json;base64,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