"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const kms = require("@aws-cdk/aws-kms");
const s3 = require("@aws-cdk/aws-s3");
const cdk = require("@aws-cdk/core");
const REQUIRED_ALIAS_PREFIX = 'alias/';
/**
 * A class needed to work around CodePipeline's extremely small (100 characters)
 * limit for the name/ARN of the key in the ArtifactStore.
 * Limits the length of the alias' auto-generated name to 50 characters.
 */
class AliasWithShorterGeneratedName extends kms.Alias {
    generatePhysicalName() {
        let baseName = super.generatePhysicalName();
        if (baseName.startsWith(REQUIRED_ALIAS_PREFIX)) {
            // remove the prefix, because we're taking the last characters of the name below
            baseName = baseName.substring(REQUIRED_ALIAS_PREFIX.length);
        }
        const maxLength = 50 - REQUIRED_ALIAS_PREFIX.length;
        // take the last characters, as they include the hash,
        // and so have a higher chance of not colliding
        return REQUIRED_ALIAS_PREFIX + lastNCharacters(baseName, maxLength);
    }
}
function lastNCharacters(str, n) {
    const startIndex = Math.max(str.length - n, 0);
    return str.substring(startIndex);
}
class CrossRegionSupportConstruct extends cdk.Construct {
    constructor(scope, id) {
        super(scope, id);
        const encryptionKey = new kms.Key(this, 'CrossRegionCodePipelineReplicationBucketEncryptionKey', {
            removalPolicy: cdk.RemovalPolicy.DESTROY,
        });
        const encryptionAlias = new AliasWithShorterGeneratedName(this, 'CrossRegionCodePipelineReplicationBucketEncryptionAlias', {
            targetKey: encryptionKey,
            aliasName: cdk.PhysicalName.GENERATE_IF_NEEDED,
            removalPolicy: cdk.RemovalPolicy.DESTROY,
        });
        this.replicationBucket = new s3.Bucket(this, 'CrossRegionCodePipelineReplicationBucket', {
            bucketName: cdk.PhysicalName.GENERATE_IF_NEEDED,
            encryptionKey: encryptionAlias,
        });
    }
}
exports.CrossRegionSupportConstruct = CrossRegionSupportConstruct;
/**
 * A Stack containing resources required for the cross-region CodePipeline functionality to work.
 * This class is private to the aws-codepipeline package.
 */
class CrossRegionSupportStack extends cdk.Stack {
    constructor(scope, id, props) {
        super(scope, id, {
            stackName: generateStackName(props),
            env: {
                region: props.region,
                account: props.account,
            },
        });
        const crossRegionSupportConstruct = new CrossRegionSupportConstruct(this, 'Default');
        this.replicationBucket = crossRegionSupportConstruct.replicationBucket;
    }
}
exports.CrossRegionSupportStack = CrossRegionSupportStack;
function generateStackName(props) {
    return `${props.pipelineStackName}-support-${props.region}`;
}
//# sourceMappingURL=data:application/json;base64,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