"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const core_1 = require("@aws-cdk/core");
const validation = require("./validation");
/**
 * An output artifact of an action. Artifacts can be used as input by some actions.
 */
class Artifact {
    constructor(artifactName) {
        validation.validateArtifactName(artifactName);
        this._artifactName = artifactName;
    }
    /**
     * A static factory method used to create instances of the Artifact class.
     * Mainly meant to be used from `decdk`.
     *
     * @param name the (required) name of the Artifact
     */
    static artifact(name) {
        return new Artifact(name);
    }
    get artifactName() {
        return this._artifactName;
    }
    /**
     * Returns an ArtifactPath for a file within this artifact.
     * CfnOutput is in the form "<artifact-name>::<file-name>"
     * @param fileName The name of the file
     */
    atPath(fileName) {
        return new ArtifactPath(this, fileName);
    }
    /**
     * The artifact attribute for the name of the S3 bucket where the artifact is stored.
     */
    get bucketName() {
        return artifactAttribute(this, 'BucketName');
    }
    /**
     * The artifact attribute for The name of the .zip file that contains the artifact that is
     * generated by AWS CodePipeline, such as 1ABCyZZ.zip.
     */
    get objectKey() {
        return artifactAttribute(this, 'ObjectKey');
    }
    /**
     * The artifact attribute of the Amazon Simple Storage Service (Amazon S3) URL of the artifact,
     * such as https://s3-us-west-2.amazonaws.com/artifactstorebucket-yivczw8jma0c/test/TemplateSo/1ABCyZZ.zip.
     */
    get url() {
        return artifactAttribute(this, 'URL');
    }
    /**
     * Returns a token for a value inside a JSON file within this artifact.
     * @param jsonFile The JSON file name.
     * @param keyName The hash key.
     */
    getParam(jsonFile, keyName) {
        return artifactGetParam(this, jsonFile, keyName);
    }
    /**
     * Returns the location of the .zip file in S3 that this Artifact represents.
     * Used by Lambda's `CfnParametersCode` when being deployed in a CodePipeline.
     */
    get s3Location() {
        return {
            bucketName: this.bucketName,
            objectKey: this.objectKey,
        };
    }
    toString() {
        return this.artifactName;
    }
    /** @internal */
    _setName(name) {
        if (this._artifactName) {
            throw new Error(`Artifact already has name '${this._artifactName}', cannot override it`);
        }
        else {
            this._artifactName = name;
        }
    }
}
exports.Artifact = Artifact;
/**
 * A specific file within an output artifact.
 *
 * The most common use case for this is specifying the template file
 * for a CloudFormation action.
 */
class ArtifactPath {
    constructor(artifact, fileName) {
        this.artifact = artifact;
        this.fileName = fileName;
    }
    static artifactPath(artifactName, fileName) {
        return new ArtifactPath(Artifact.artifact(artifactName), fileName);
    }
    get location() {
        const artifactName = this.artifact.artifactName
            ? this.artifact.artifactName
            : core_1.Lazy.stringValue({ produce: () => this.artifact.artifactName });
        return `${artifactName}::${this.fileName}`;
    }
}
exports.ArtifactPath = ArtifactPath;
function artifactAttribute(artifact, attributeName) {
    const lazyArtifactName = core_1.Lazy.stringValue({ produce: () => artifact.artifactName });
    return core_1.Token.asString({ 'Fn::GetArtifactAtt': [lazyArtifactName, attributeName] });
}
function artifactGetParam(artifact, jsonFile, keyName) {
    const lazyArtifactName = core_1.Lazy.stringValue({ produce: () => artifact.artifactName });
    return core_1.Token.asString({ 'Fn::GetParam': [lazyArtifactName, jsonFile, keyName] });
}
//# sourceMappingURL=data:application/json;base64,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