"use strict";
/**
 *  Copyright 2020 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
// Imports
const core_1 = require("@aws-cdk/core");
const lib_1 = require("../lib");
const assert_1 = require("@aws-cdk/assert");
require("@aws-cdk/assert/jest");
const aws_dynamodb_1 = require("@aws-cdk/aws-dynamodb");
const api = require("@aws-cdk/aws-apigateway");
test('snapshot test ApiGatewayToDynamoDB default params', () => {
    const stack = new core_1.Stack();
    const apiGatewayToDynamoDBProps = {};
    new lib_1.ApiGatewayToDynamoDB(stack, 'test-api-gateway-dynamodb-default', apiGatewayToDynamoDBProps);
    expect(assert_1.SynthUtils.toCloudFormation(stack)).toMatchSnapshot();
});
test('check properties', () => {
    const stack = new core_1.Stack();
    const apiGatewayToDynamoDBProps = {};
    const construct = new lib_1.ApiGatewayToDynamoDB(stack, 'test-api-gateway-dynamodb-default', apiGatewayToDynamoDBProps);
    expect(construct.dynamoTable !== null);
    expect(construct.apiGateway !== null);
    expect(construct.apiGatewayRole !== null);
    expect(construct.apiGatewayCloudWatchRole !== null);
    expect(construct.apiGatewayLogGroup !== null);
});
test('check allow CRUD operations', () => {
    const stack = new core_1.Stack();
    const apiGatewayToDynamoDBProps = {
        allowReadOperation: true,
        allowCreateOperation: true,
        createRequestTemplate: "{}",
        allowDeleteOperation: true,
        allowUpdateOperation: true,
        updateRequestTemplate: "{}"
    };
    new lib_1.ApiGatewayToDynamoDB(stack, 'test-api-gateway-dynamodb', apiGatewayToDynamoDBProps);
    expect(stack).toHaveResource("AWS::IAM::Policy", {
        PolicyDocument: {
            Statement: [
                {
                    Action: "dynamodb:PutItem",
                    Effect: "Allow",
                    Resource: {
                        "Fn::GetAtt": [
                            "testapigatewaydynamodbDynamoTableEEE3F463",
                            "Arn"
                        ]
                    }
                },
                {
                    Action: "dynamodb:Query",
                    Effect: "Allow",
                    Resource: {
                        "Fn::GetAtt": [
                            "testapigatewaydynamodbDynamoTableEEE3F463",
                            "Arn"
                        ]
                    }
                },
                {
                    Action: "dynamodb:UpdateItem",
                    Effect: "Allow",
                    Resource: {
                        "Fn::GetAtt": [
                            "testapigatewaydynamodbDynamoTableEEE3F463",
                            "Arn"
                        ]
                    }
                },
                {
                    Action: "dynamodb:DeleteItem",
                    Effect: "Allow",
                    Resource: {
                        "Fn::GetAtt": [
                            "testapigatewaydynamodbDynamoTableEEE3F463",
                            "Arn"
                        ]
                    }
                }
            ],
            Version: "2012-10-17"
        },
        PolicyName: "testapigatewaydynamodbapigatewayroleDefaultPolicy43AC565D",
        Roles: [
            {
                Ref: "testapigatewaydynamodbapigatewayrole961B19C4"
            }
        ]
    });
    expect(stack).toHaveResource("AWS::ApiGateway::Method", {
        HttpMethod: "GET",
        AuthorizationType: "AWS_IAM"
    });
    expect(stack).toHaveResource("AWS::ApiGateway::Method", {
        HttpMethod: "POST",
        AuthorizationType: "AWS_IAM"
    });
    expect(stack).toHaveResource("AWS::ApiGateway::Method", {
        HttpMethod: "PUT",
        AuthorizationType: "AWS_IAM"
    });
    expect(stack).toHaveResource("AWS::ApiGateway::Method", {
        HttpMethod: "DELETE",
        AuthorizationType: "AWS_IAM"
    });
    expect(stack).toHaveResource("AWS::ApiGateway::Resource", {
        PathPart: "{id}",
    });
});
test('check allow read and update only', () => {
    const stack = new core_1.Stack();
    const apiGatewayToDynamoDBProps = {
        allowUpdateOperation: true,
        updateRequestTemplate: "{}"
    };
    new lib_1.ApiGatewayToDynamoDB(stack, 'test-api-gateway-dynamodb', apiGatewayToDynamoDBProps);
    expect(stack).toHaveResource("AWS::IAM::Policy", {
        PolicyDocument: {
            Statement: [
                {
                    Action: "dynamodb:Query",
                    Effect: "Allow",
                    Resource: {
                        "Fn::GetAtt": [
                            "testapigatewaydynamodbDynamoTableEEE3F463",
                            "Arn"
                        ]
                    }
                },
                {
                    Action: "dynamodb:UpdateItem",
                    Effect: "Allow",
                    Resource: {
                        "Fn::GetAtt": [
                            "testapigatewaydynamodbDynamoTableEEE3F463",
                            "Arn"
                        ]
                    }
                }
            ],
            Version: "2012-10-17"
        },
        PolicyName: "testapigatewaydynamodbapigatewayroleDefaultPolicy43AC565D",
        Roles: [
            {
                Ref: "testapigatewaydynamodbapigatewayrole961B19C4"
            }
        ]
    });
    expect(stack).toHaveResource("AWS::ApiGateway::Method", {
        HttpMethod: "GET",
        AuthorizationType: "AWS_IAM"
    });
});
test('check using custom partition key for dynamodb', () => {
    const stack = new core_1.Stack();
    const apiGatewayToDynamoDBProps = {
        dynamoTableProps: {
            partitionKey: {
                name: 'page_id',
                type: aws_dynamodb_1.AttributeType.STRING
            }
        }
    };
    new lib_1.ApiGatewayToDynamoDB(stack, 'test-api-gateway-dynamodb', apiGatewayToDynamoDBProps);
    expect(stack).toHaveResource("AWS::ApiGateway::Resource", {
        PathPart: "{page_id}",
    });
});
test('override apiGatewayProps for api gateway', () => {
    const stack = new core_1.Stack();
    const apiGatewayToDynamoDBProps = {
        apiGatewayProps: {
            description: 'This is a sample description for api gateway'
        }
    };
    new lib_1.ApiGatewayToDynamoDB(stack, 'test-api-gateway-dynamodb', apiGatewayToDynamoDBProps);
    expect(stack).toHaveResource("AWS::ApiGateway::RestApi", {
        Description: "This is a sample description for api gateway"
    });
});
test('Test deployment ApiGateway AuthorizationType override', () => {
    const stack = new core_1.Stack();
    new lib_1.ApiGatewayToDynamoDB(stack, 'api-gateway-dynamodb', {
        apiGatewayProps: {
            defaultMethodOptions: {
                authorizationType: api.AuthorizationType.NONE
            }
        }
    });
    expect(stack).toHaveResourceLike("AWS::ApiGateway::Method", {
        HttpMethod: "GET",
        AuthorizationType: "NONE"
    });
});
//# sourceMappingURL=data:application/json;base64,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